package gov.va.fnod.security;

import gov.va.fnod.security.authentication.AccountMaintenceInterface;
import gov.va.fnod.security.authentication.UserPswdChangeContext;
import gov.va.fnod.security.authorization.AppPrivilege;
import gov.va.fnod.security.authorization.AppRole;
import gov.va.fnod.security.authorization.UserContext;
import gov.va.fnod.security.exception.AccountLockedException;
import gov.va.fnod.security.exception.AuthenticationException;
import gov.va.fnod.security.exception.AuthorizationException;
import gov.va.fnod.security.exception.PasswordTooNewException;
import gov.va.fnod.security.exception.PswdValidationException;
import gov.va.fnod.security.service.PswdChangeConstraints;
import gov.va.fnod.security.service.SystemContextService;
import gov.va.fnod.security.service.SystemContextServiceFactory;
import gov.va.fnod.security.service.UserContextServiceFactory;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public class UserMaintenceUtility implements AccountMaintenceInterface {

	private static final Logger logr = Logger.getLogger(UserMaintenceUtility.class.getName());

	
	public void changePswd(UserContext curUserContext, String oldPswd, String newPswd)
			throws AuthorizationException, AuthenticationException, PswdValidationException {

		// Is curUser authorized for this operation
		if (!canChangePasswords(curUserContext)) {
			logr.log(Level.WARNING, "{0}[id: {1}] not allowed to change password.", new Object[] {
					curUserContext.getUserName(), curUserContext.getUserId() });
			throw new AuthorizationException("user not allowed to change password");
		}

		authenticate(curUserContext, oldPswd);

		UserPswdChangeContext pswdContext = getUserPswdChangeContext(curUserContext);

		checkCanChangeNow(pswdContext);

		new PswdValidatorUtility().validate(pswdContext, newPswd);

		String newPswdHash = PswdHelper.hashPassword(pswdContext, newPswd);

		pswdContext.setPswdHash(newPswdHash);
		pswdContext.setPswdHistory(updatePasswordHistory(newPswdHash, pswdContext.getPswdHistory()));

		pswdContext.setLastPswdChangeDate(new Date());

		pswdContext.setExpiryDate(computeExpiryDate(pswdContext.getLastPswdChangeDate()));

		UserContextServiceFactory.getFactory().userContextService().save(pswdContext);

	}

	private void checkCanChangeNow(UserPswdChangeContext curUserContext) throws PasswordTooNewException {

		DateFormat fmt = new SimpleDateFormat("hh:mm a dd-MMM-yyyy");

		SystemContextService<? extends AppRole<?>, ? extends AppPrivilege<?>> service;
		service = SystemContextServiceFactory.getFactory().getService();

		// Always allow change of expired password
		if (!isPswdExpired(curUserContext.getExpiryDate())) {

			// If password not expired, check against hours parameter
			PswdChangeConstraints constraints = service.getPswdConstraints();
			int maxAgeHrs = constraints.getMinHrsBeforeNextPswdChange();

			Calendar allowedTime = Calendar.getInstance();
			allowedTime.setTime(curUserContext.getLastPswdChangeDate());
			allowedTime.add(Calendar.HOUR, maxAgeHrs);
			boolean canChangeNow = (new Date()).after(allowedTime.getTime()); 
			if ( ! canChangeNow ) {
				logr.log(Level.WARNING,
						"{0}[id: {1}] not allowed to change password yet. Password cannot be changed until {2}",
						new Object[] { curUserContext.getUserName(), curUserContext.getUserId(),
								fmt.format(allowedTime.getTime()) });
				throw new PasswordTooNewException("Password cannot be changed until: "
						+ fmt.format(allowedTime.getTime()));
			}
		}
	}

	private boolean isPswdExpired(Date expiryDate) {
		return (new Date()).after(expiryDate);
	}

	/**
	 * 
	 * @throws AuthorizationException
	 * @throws AccountLockedException
	 * @throws PswdValidationException
	 */

	
	public void resetPswd(UserContext curUserContext, UserContext userContext, String pswd)
			throws AuthorizationException, PswdValidationException {

		checkCanResetPswd(curUserContext, userContext);

		UserPswdChangeContext pswdContext = getUserPswdChangeContext(userContext);
	
		if ( canChangePasswords(pswdContext) ) {
			resetPswd(pswdContext,pswd);
		} else {
			changeSystemPswd(pswdContext, pswd);
		}
	
		if (LoginStatus.TEMP_LOCKED.equals(pswdContext.getLoginStatus())) {
			pswdContext.setLoginStatus(LoginStatus.OPEN);
			pswdContext.setTimeLockedDate(null);
		}

		UserContextServiceFactory.getFactory().userContextService().save(pswdContext);
	}
	
	/**
	 * When resetting a password for a user with the ability to change their own
	 * password, history is not effected, and the new password is expired immediately.
	 * 
	 * @param pswdContext
	 * @param pswd
	 * @throws PswdValidationException
	 */
	private void resetPswd(UserPswdChangeContext pswdContext, String pswd ) throws PswdValidationException {
		
		new PswdValidatorUtility().validateForReset(pswdContext, pswd);

		String newPswdHash = PswdHelper.hashPassword(pswdContext, pswd);

		pswdContext.setPswdHash(newPswdHash);

		pswdContext.setLastPswdChangeDate(new Date());
		
		pswdContext.setExpiryDate(pswdContext.getLastPswdChangeDate()); 

	}
	
	/**
	 * When resetting a password for a user that cannot change their own password, then the password
	 * is treated like any other password change, except it is being done by an administrator.  In this
	 * case validations and date setting operation are performed as if the user was setting their own password.
	 * 
	 * @param pswdContext
	 * @param pswd
	 * @throws PswdValidationException 
	 */
	private void changeSystemPswd(UserPswdChangeContext pswdContext, String pswd) throws PswdValidationException {
		
		new PswdValidatorUtility().validate(pswdContext, pswd);

		String newPswdHash = PswdHelper.hashPassword(pswdContext, pswd);

		pswdContext.setPswdHash(newPswdHash);
		pswdContext.setPswdHistory(updatePasswordHistory(newPswdHash, pswdContext.getPswdHistory()));
		
		pswdContext.setLastPswdChangeDate(new Date());
		
		pswdContext.setExpiryDate(computeExpiryDate(pswdContext.getLastPswdChangeDate()));
		
	}

	private void checkCanResetPswd(UserContext curUserContext, UserContext userContext)
			throws AuthorizationException {

		// Is curUser authorized for this operation
		if (!getAuthorizationUtil().canResetPasswords(curUserContext)) {
			logr.log(Level.WARNING, "{0}[id: {1}] not allowed to reset password.", new Object[] {
					curUserContext.getUserName(), curUserContext.getUserId() });
			throw new AuthorizationException("user not allowed to reset passwords");
		}

		if (curUserContext.getUserName().equals(userContext.getUserName())) {
			logr.log(Level.WARNING, "{0}[id: {1}] not allowed to change their own password.", new Object[] {
					curUserContext.getUserName(), curUserContext.getUserId() });
			throw new AuthorizationException("Can't change own password");
		}
	}

	private void authenticate(UserContext curUserContext, String pswd) throws AuthenticationException {
		getAuthenticationUtil().authenticatePswd(curUserContext, pswd);
	}

	
	public void lockAccount(UserContext curUserContext, UserContext userContext) throws AuthorizationException {
		setAccountLoginStatus(curUserContext, userContext, LoginStatus.LOCKED);
	}

	
	public void unLockAccount(UserContext curUserContext, UserContext userContext)
			throws AuthorizationException {
		setAccountLoginStatus(curUserContext, userContext, LoginStatus.OPEN);
	}

	private void setAccountLoginStatus(UserContext curUserContext, UserContext uc, LoginStatus status)
			throws AuthorizationException {
		if (curUserContext.getUserName().equals(uc.getUserName())) {
			logr.log(Level.WARNING, "{0}[id: {1}] not allowed to change their own status.", new Object[] {
					curUserContext.getUserName(), curUserContext.getUserId() });
			throw new AuthorizationException("a user cannot change their own status");
		} else {
			if (canLockAccount(curUserContext, uc)) {
				UserPswdChangeContext pswdContext = getUserPswdChangeContext(uc);
				pswdContext.setLoginStatus(status);
				if (LoginStatus.LOCKED.equals(status)) {
					pswdContext.setTimeLockedDate(new Date());
				} else {
					pswdContext.setTimeLockedDate(null);
				}
				UserContextServiceFactory.getFactory().userContextService().save(pswdContext);
			} else {
				logr.log(Level.WARNING, "{0}[id: {1}] not allowed to change the account status for {2}[id: {4}].",
						new Object[] { curUserContext.getUserName(), curUserContext.getUserId(), uc.getUserName(),
								uc.getUserId() });
				throw new AuthorizationException("user not authorized to change account status");
			}
		}
	}

	private boolean canLockAccount(UserContext curUserContext, UserContext userToLock)
			throws AccountLockedException {
		return getAuthorizationUtil().canLock(curUserContext, userToLock);
	}

	private boolean canChangePasswords(UserContext curUserContext) throws AccountLockedException {
		return getAuthorizationUtil().canChangePassword(curUserContext);
	}

	private UserPswdChangeContext getUserPswdChangeContext(UserContext userContext) {
		return UserContextServiceFactory.getFactory().userContextService().getUserPswdChangeContext(userContext);
	}

	private Date computeExpiryDate(Date changedDate) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(changedDate);
		PswdChangeConstraints constraints = SystemContextServiceFactory.getFactory().getService()
				.getPswdConstraints();
		calendar.add(Calendar.DATE, constraints.getExpiryDays());
		return calendar.getTime();
	}

	private List<String> updatePasswordHistory(String newPswdHash, List<String> pswdHistory) {

		PswdChangeConstraints constraints = SystemContextServiceFactory.getFactory().getService()
				.getPswdConstraints();

		int maxHistEntries = constraints.getMaxHistoryEntries();

		// Building new history list
		List<String> newHistory = new ArrayList<String>();
		newHistory.addAll(pswdHistory);

		// Add new password hash to top of list
		newHistory.add(0,newPswdHash);

		// Now remove excess history entries from bottom of the list
		while( newHistory.size() > maxHistEntries ) {
			newHistory.remove(newHistory.size()-1);
		}
		return newHistory;
	}

	
	public UserContext getUserContext(String username) {
		// This returns an initial context;
		UserContext uc = UserContextServiceFactory.getFactory().userContextService().getUserContext(username);
		if (uc == null) {
			logr.log(Level.WARNING, "User context for {0} not found.", username);
			throw new IllegalArgumentException("User context not found");
		}
		return downCastToUserContext(uc);
	}

	/**
	 * This method take any UserContext and strips it down to the bare minimum.
	 * 
	 * @param uc
	 * @return
	 */
	UserContext downCastToUserContext(final UserContext uc) {

		UserContext retval = null;
		if (uc != null) {
			retval = new UserContext() {
				private String name = uc.getUserName();
				private long id = uc.getUserId();

				
				public String getUserName() {
					return name;
				}

				
				public long getUserId() {
					return id;
				}
			};
		}
		return retval;
	}

	private AuthorizationUtility getAuthorizationUtil() {
		return new AuthorizationUtility();
	}

	private AuthenticationUtility getAuthenticationUtil() {
		return new AuthenticationUtility();
	}

}
