package gov.va.fnod.security;

import gov.va.fnod.security.authorization.AppPrivilege;
import gov.va.fnod.security.authorization.AppRole;
import gov.va.fnod.security.authorization.AuthorizationInterface;
import gov.va.fnod.security.authorization.SecurityContext;
import gov.va.fnod.security.authorization.UserContext;
import gov.va.fnod.security.authorization.UserRoleContext;
import gov.va.fnod.security.exception.AccountLockedException;
import gov.va.fnod.security.service.SystemContextServiceFactory;
import gov.va.fnod.security.service.UserContextServiceFactory;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Burk Price
 * @version 1.0
 */
public class AuthorizationUtility implements AuthorizationInterface {
	private static final Logger logr = Logger.getLogger(AuthorizationUtility.class.getName());
	
	
	public <P extends AppPrivilege<?>>boolean isUserAuthorized(UserContext curUser,
			SecurityContext<P> sysContext) throws AccountLockedException {
		boolean retval = false;
		
		isAccountLockedFor(curUser);
		
		Set<? extends AppRole<?>> uroles = getUserRoleContext(curUser).getRoles();
		Set<? extends AppPrivilege<?>> sysRoles = sysContext
				.requestPrivileges();

		retval = getPrivileges(uroles).containsAll(sysRoles);

		return retval;
	}
	
	public boolean canLogin(UserContext curUser) throws AccountLockedException {
		boolean retval = false;
		
		isAccountLockedFor(curUser);
		
		Set<? extends AppRole<?>> uroles = UserContextServiceFactory
				.getFactory().userContextService().getUserRoleContext(curUser)
				.getRoles();
		
		SecurityContext<? extends AppPrivilege<?>> canLoginPriv = SystemContextServiceFactory
				.getFactory().getService().getCanLoginContext();
		
		retval = getPrivileges(uroles).containsAll(canLoginPriv.requestPrivileges());

		return retval;
	}

	public boolean canResetPasswords(UserContext curUser)
			throws AccountLockedException {
		boolean retval = false;
		
		isAccountLockedFor(curUser);
		
		Set<? extends AppRole<?>> uroles = UserContextServiceFactory
				.getFactory().userContextService().getUserRoleContext(curUser)
				.getRoles();
		
		SecurityContext<? extends AppPrivilege<?>> canResetPswd = SystemContextServiceFactory
				.getFactory().getService().getCanResetPswdContext();
		
		retval = getPrivileges(uroles).containsAll(canResetPswd.requestPrivileges());

		return retval;
	}

	public boolean canChangePassword(UserContext curUser)
			throws AccountLockedException {
		boolean retval = false;
		
		isAccountLockedFor(curUser);
		
		Set<? extends AppRole<?>> uroles = UserContextServiceFactory
				.getFactory().userContextService().getUserRoleContext(curUser)
				.getRoles();
		
		SecurityContext<? extends AppPrivilege<?>> canChangePswd = SystemContextServiceFactory
				.getFactory().getService().getCanChangePswdContext();
		
		retval = getPrivileges(uroles).containsAll(canChangePswd.requestPrivileges());

		return retval;
	}

	public boolean canConnect(UserContext curUser)
			throws AccountLockedException {
		return canConnect(curUser, new SecurityContextWrapper<AppPrivilege<?>>());
	}
	
	boolean canConnect(UserContext curUser, SecurityContext<? extends AppPrivilege<?>> context) throws AccountLockedException {
		boolean retval = false;
		
		isAccountLockedFor(curUser);
		
		Set<? extends AppRole<?>> uroles = UserContextServiceFactory
				.getFactory().userContextService().getUserRoleContext(curUser)
				.getRoles();
		
		SecurityContext<? extends AppPrivilege<?>> canConnect = SystemContextServiceFactory
				.getFactory().getService().getCanConnectContext();
		
		@SuppressWarnings("unchecked")
		Set<AppPrivilege<?>> contextprivs = (Set<AppPrivilege<?>>) context.requestPrivileges();
		contextprivs.addAll((Set<? extends AppPrivilege<?>>) canConnect.requestPrivileges());
		
		retval = getPrivileges(uroles).containsAll(contextprivs);

		return retval;

	}

	public boolean canLock(UserContext curUser, UserContext userToLock)
			throws AccountLockedException {
		boolean retval = false;
		
		isAccountLockedFor(curUser);
		
		Set<? extends AppRole<?>> uroles = UserContextServiceFactory
				.getFactory().userContextService().getUserRoleContext(curUser)
				.getRoles();
		
		SecurityContext<? extends AppPrivilege<?>> canLock = SystemContextServiceFactory
				.getFactory().getService().getCanLockUserContext();
		
		retval = getPrivileges(uroles).containsAll(canLock.requestPrivileges());

		return retval;
	}

	private void isAccountLockedFor(UserContext chkUser)
			throws AccountLockedException {
		LoginStatus status;
		status = UserContextServiceFactory.getFactory().userContextService()
				.getUserRoleContext(chkUser).getLoginStatus();
		if ( LoginStatus.LOCKED.equals(status) ) {
			logr.log(Level.WARNING, "{0} has attempted an activity on a locked account", chkUser.getUserName());
			throw new AccountLockedException();
		}
	}
	
	private UserRoleContext<? extends AppRole<?>> getUserRoleContext(UserContext context) {
		return UserContextServiceFactory.getFactory().userContextService()
				.getUserRoleContext(context);
	}
	
	private Set<? extends AppPrivilege<?>> getPrivileges(Set<? extends AppRole<?>> wkRoles ) {
		
		Set<AppPrivilege<?>> retval = new HashSet<AppPrivilege<?>>();
		Map<? extends AppRole<?>, ?> rolePrivMap = SystemContextServiceFactory.getFactory().getService().getRolePrivMap();
		
		
		for ( AppRole<?> role: wkRoles ) {
			@SuppressWarnings("unchecked")
			Set<? extends AppPrivilege<?>> privs = (Set<? extends AppPrivilege<?>>) rolePrivMap.get(role);
			retval.addAll(privs);
		}
		
		return retval;
	}

}
