package gov.va.fnod.security;



import gov.va.fnod.security.authentication.AuthenticationInterface;

import gov.va.fnod.security.authentication.LoginConstraints;

import gov.va.fnod.security.authentication.UserInitialContext;

import gov.va.fnod.security.authorization.AppPrivilege;

import gov.va.fnod.security.authorization.AppRole;

import gov.va.fnod.security.authorization.SecurityContext;

import gov.va.fnod.security.authorization.UserContext;

import gov.va.fnod.security.exception.AccountLockedException;

import gov.va.fnod.security.exception.AuthenticationException;

import gov.va.fnod.security.exception.AuthorizationException;

import gov.va.fnod.security.exception.PasswordExpiredException;

import gov.va.fnod.security.exception.PasswordTooNewException;

import gov.va.fnod.security.exception.PswdValidationException;

import gov.va.fnod.security.service.SystemContextService;

import gov.va.fnod.security.service.SystemContextServiceFactory;

import gov.va.fnod.security.service.UserContextService;

import gov.va.fnod.security.service.UserContextServiceFactory;



import java.util.Calendar;

import java.util.Date;

import java.util.logging.Level;

import java.util.logging.Logger;



public final class AuthenticationUtility implements AuthenticationInterface {



	private static final Logger logr = Logger.getLogger(AuthenticationUtility.class.getName());

	

	private UserMaintenceUtility maintenceUtil = new UserMaintenceUtility();

	

	private AuthorizationUtility getAuthorizationUtil() {

		return new AuthorizationUtility();

	}



	/**

	 * Authenticates user with provided password to determine of user is allowed

	 * to login into system.

	 * 

	 * @param username

	 * @param pswd

	 */



	public UserContext login(String username, String pswd) throws AuthenticationException {



		// Get UserInitalContext to be used for Authentication

		UserInitialContext uic = getUserInitialContext(username);



		// A valid account may not be allowed to login to a given software

		// interface, so we need to check for that.

		checkIsLoginAuthorized(uic);



		// Check for expired password

		try {

			checkIsPasswordExpired(uic);

		} catch (PasswordExpiredException ex) {

			// If the user has the right password, but the 

			// password has expired, let them know

			if ( PswdHelper.pswdMatch(uic, pswd, uic.getPswdHash()) ) {

				throw ex;

			} else {

				// Otherwise, just report failed login

				throw new AuthenticationException(ex);

			}

		}



		// Handle temporally locked accounts

		clearTempLocked(uic);



		return attemptLogin(uic, pswd);



	}



	/**

	 * Authenticates user with provided password to determine of user is allowed

	 * to login into system; provides additional password change functionality.

	 * 

	 * @param username

	 * @param oldPswd

	 * @param newPswd

	 * @throws AuthenticationException

	 * @throws PswdValidationException

	 * @throws AuthorizationException

	 * @throws PasswordTooNewException

	 */

	
	public UserContext login(String username, String oldPswd, String newPswd)

			throws AuthenticationException, AuthorizationException,

			PswdValidationException {



		// authenticate with old password

		UserInitialContext uic = getUserInitialContext(username);



		// Make sure we are dealing with a valid user

		authenticatePswd(uic, oldPswd);



		// Change the password

		maintenceUtil.changePswd(uic, oldPswd, newPswd);



		// login with the new password

		return login(username, newPswd);

	}



	

	public UserContext connect(String username, String pswd)

			throws AuthenticationException {



		return connect(username, pswd,

				new SecurityContextWrapper<AppPrivilege<?>>());



	}



	

	public UserContext connect(String username, String pswd,

			SecurityContext<? extends AppPrivilege<?>> context)

			throws AuthenticationException {



		UserInitialContext uic = getUserInitialContext(username);



		// A valid account may not be allowed to connect to a given application

		// interface, so we need to check for that.

		checkIsConnectAuthorized(uic, context);



		// Check for expired password

		checkIsPasswordExpired(uic);



		// Handle locked accounts

		clearTempLocked(uic);



		return attemptLogin(uic, pswd);



	}



	

	public UserContext connect(String username, String oldPswd, String newPswd)

			throws AuthenticationException, AuthorizationException,

			PswdValidationException {



		return connect(username, oldPswd, newPswd,

				new SecurityContextWrapper<AppPrivilege<?>>());



	}



	

	public UserContext connect(String username, String oldPswd, String newPswd,

			SecurityContext<? extends AppPrivilege<?>> context)

			throws AuthenticationException, AuthorizationException,

			PswdValidationException {



		// get user context

		UserInitialContext uic = getUserInitialContext(username);



		// Make user user is authorized to connect to application

		// before changing password

		checkIsConnectAuthorized(uic, context);



		// Make sure we are dealing with a valid user

		authenticatePswd(uic, oldPswd);



		// Change the password

		maintenceUtil.changePswd(uic, oldPswd, newPswd);



		// login with the new password

		return connect(username, newPswd, context);

	}



	private UserInitialContext getUserInitialContext(String username)

			throws AuthenticationException {



		UserContextService<? extends AppRole<?>> ucs;

		ucs = UserContextServiceFactory.getFactory().userContextService();



		UserInitialContext uic = ucs.getUserContext(username);



		if (uic == null) {

			// TODO: Logger/change exception

			throw new AuthenticationException();

		}



		return uic;

	}



	private void checkIsLoginAuthorized(UserInitialContext uic)

			throws AuthenticationException {

		try {

			if (!getAuthorizationUtil().canLogin(uic)) {

				throw new AuthenticationException();

			}

		} catch (AccountLockedException ex) {

			throw new AuthenticationException(ex);

		}

	}



	private void checkIsConnectAuthorized(UserInitialContext uic,

			SecurityContext<? extends AppPrivilege<?>> context)

			throws AuthenticationException {



		try {

			if (!getAuthorizationUtil().canConnect(uic, context)) {

				throw new AuthenticationException();

			}

		} catch (AccountLockedException ex) {

			throw new AuthenticationException(ex);

		}



	}



	private void checkIsPasswordExpired(UserInitialContext uic)

			throws AuthenticationException {



		if ((new Date()).after(uic.getExpiryDate())) {

			throw new PasswordExpiredException("Password Expired: please change password now.");

		}

	}



	private void clearTempLocked(final UserInitialContext uic)

			throws AuthenticationException {

		if (LoginStatus.TEMP_LOCKED.equals(uic.getLoginStatus())) {

			if (isPswdTimeoutExpired(uic.getTimeLockedDate())) {

				// Clear lock - allow n-more attempts

				uic.setLoginAttempts(0);

				clearLock(uic);

			} else {

				logr.log(Level.WARNING, "{0} attempted (and failed) a clear lock", uic.getUserName());

				throw new AuthenticationException();

			}

		}

	}



	private boolean isPswdTimeoutExpired(Date timeTempLocked) {

		LoginConstraints constraints;

		constraints = SystemContextServiceFactory.getFactory().getService()

				.getLoginConstraints();



		Calendar allowedTime = Calendar.getInstance();

		allowedTime.setTime(timeTempLocked);

		allowedTime.add(Calendar.MINUTE, + constraints.getTempLockDurationMin());



		boolean retval = (new Date()).after(allowedTime.getTime());

		

		return retval;

	}



	private UserContext attemptLogin(UserInitialContext uic, String pswd)

			throws AuthenticationException {



		try {

			authenticatePswd(uic, pswd);



			// On success

			uic.setLastLoginDate(new Date());

			uic.setLoginAttempts(0);



			return maintenceUtil.downCastToUserContext(uic);



		} catch (AuthenticationException ex) {



			// An attempt was made (and failed),

			// so increment login attempts

			logr.log(Level.WARNING, "{0} attempted (and failed) a login.  Incrementing login attempts. {1}", new Object[]{uic.getUserName(), ex.getMessage()});

			uic.setLoginAttempts(uic.getLoginAttempts()+1);

			

			if (isPswdRetriesExceeded(uic.getLoginAttempts())) {

				uic.setLoginStatus(LoginStatus.TEMP_LOCKED);

				uic.setTimeLockedDate(new Date());

			}



			throw ex;



		} finally {

			// Record results of login attempt

			save(uic);

		}



	}



	private boolean isPswdRetriesExceeded(int loginAttempts) {



		SystemContextService<? extends AppRole<?>, ? extends AppPrivilege<?>> service;

		service = SystemContextServiceFactory.getFactory().getService();

		LoginConstraints constraints = service.getLoginConstraints();

		int maxAttempts = constraints.getMaxLoginAttempts();



		return maxAttempts <= loginAttempts;

	}



	private void save(UserInitialContext uic) {

		UserContextServiceFactory.getFactory().userContextService().save(uic);

	}



	private void clearLock(UserInitialContext context) {

		context.setLoginStatus(LoginStatus.OPEN);

		context.setTimeLockedDate(null);

	}



	void authenticatePswd(UserContext userContext, String pswd)

			throws AuthenticationException {



		UserInitialContext uic = UserContextServiceFactory.getFactory().userContextService()

				.getUserContext(userContext.getUserName());



		authenticatePswd(uic, pswd);

	}



	void authenticatePswd(UserInitialContext userCtxt, String pswd)

			throws AuthenticationException {

		

		if (! PswdHelper.pswdMatch(userCtxt, pswd, userCtxt.getPswdHash())) {

			logr.log(Level.WARNING, "{0} attempted (and failed) to authenticate password", userCtxt.getUserName());

			throw new AuthenticationException();

		}

	}



}

