package gov.va.fnod.flagloader.bl;



import gov.va.fnod.flagloader.be.ConfigEntity;
import gov.va.fnod.flagloader.be.FnodMetadata;
import gov.va.fnod.flagloader.be.ProcessState;
import gov.va.fnod.flagloader.be.UserCredentials;
import gov.va.fnod.flagloader.bl.TaskStatusCallback.Level;
import gov.va.fnod.flagloader.dl.FlagAppService;




import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;
import java.util.regex.Pattern;



/**

 * This class is responsible for transmitting the Flag Applications to the FNOD

 * server.

 * <p/>

 * The following process steps will be executed for each application:

 * <p/>

 * <ol>

 * <li>read data file</li>

 * <li>using data from data file, transmit Flag Application w/ data to server</li>

 * <li>move (archive, then delete) successfully transmitted files into archive</li>

 * <li>add line to transmitted log file

 * </ol>

 * <p/>

 * The archive process acts as a process check point; any files left after

 * failure need to be transmitted.

 * 

 */

public class TaskTransmitMessages extends TaskBase implements TaskAuthRequired {



	private FlagAppService flagService;

	private UserCredentials userCredentials;

	private Archiver archiver;

	

	@Override

	public void init(ProcessState state, ConfigEntity config, TaskStatusCallback statusAppender) {
		log(Level.INFO,"TaskTransmitMessages.init");
		System.out.println("Endpoint: " + config.getEndPoint());
		System.out.println("Source dir: " + config.getSourceDirPath());
		System.out.println("Working Dir: " + config.getWorkingDirPath());
		System.out.println("Completed Dir: " + config.getCompletedDirPath());
				
		log(Level.INFO,"Web Service: " + config.getEndPoint());
		log(Level.INFO,"Source dir: " + config.getSourceDirPath());
		log(Level.INFO,"Working Dir: " + config.getWorkingDirPath());
		log(Level.INFO,"Completed Dir: " + config.getCompletedDirPath());
		
		
		
		super.init(state,config,statusAppender);		

	}

	

	

	public void setCredentials(UserCredentials credentials) {

		userCredentials = credentials;

	}

	

	@Override

	protected void doRun() {

	

		archiver = new Archiver(getWorkingDir(), getState());

		flagService = new FlagAppService(getConfig().getEndPoint());

		
		log(Level.INFO,"Transmitting files to server. Service: " + getConfig().getEndPoint());
		

		for (String filename : getApplicationDataFiles()) {
			
			log(Level.INFO,"Processing file: " + filename);
			
			FnodMetadata data = loadMetadata(filename);
			
			log(Level.INFO,"Transmitting message: " + data.toString());
			
			transmitMessage(data);

			log(Level.INFO,"Reporting Transmission for file: " + filename);
			
			reportTransmission(data);
			
			log(Level.INFO,"Archiving file: " + filename);

			archiveFiles(filename);

		}

		

		log(Level.INFO, "Archiving transmitted files.");

		archiver.makeZip();

		

		log(Level.INFO, "Transmit process complete.");

		

	}



	private String[] getApplicationDataFiles() {

		final Pattern pattern = FilenameHelper.getProcessDataFilePattern();

		String[] retval = getWorkingDir().list(new FilenameFilter() {

			

			public boolean accept(File dir, String filename) {

				return pattern.matcher(filename).matches();

			}

		});

		Arrays.sort(retval);

		return retval;

	}



	private FnodMetadata loadMetadata(String filename) {

		File file = new File(getWorkingDir(), filename);

		String line = null;

		try {

			BufferedReader reader = new BufferedReader(new FileReader(file));

			try {

				line = reader.readLine();

			} finally {

				reader.close();

			}

		} catch (FileNotFoundException ex) {

			throw new RuntimeException("File not found: " + filename);

		} catch (IOException ex) {

			throw new RuntimeException("Error processing file: " + filename);

		}



		return FnodMetadataHelper.parseFormattedMetadata(line);

	}



	private void transmitMessage(FnodMetadata data) {

		

		String pdfFilename = data.getDocumentId().toString() + ".pdf";
		log(Level.INFO,"TaskTransmitMessage.transmitMessage: " + ": sending doc: " +  pdfFilename);
		Long caseLinkId = flagService.submitFlagApp(userCredentials, data, loadPdf(pdfFilename), pdfFilename);
		log(Level.INFO,"Done TaskTransmitMessage.transmitMessage: " + ": sending doc: " +  pdfFilename);
		data.setCaseLinkId(caseLinkId);

	

	}



	private byte[] loadPdf(String filename) {

	

		File pdfFile = new File(getWorkingDir(), filename);

		

		ByteArrayOutputStream bos = new ByteArrayOutputStream();

		byte[] buf = new byte[8*1024];  // read 8K blocks

		try {

			BufferedInputStream bis = new BufferedInputStream(new FileInputStream(pdfFile));

			int bytesRead=0;

			try {

				while ( (bytesRead=bis.read(buf)) > 0) {

					bos.write(buf,0,bytesRead);

				}

			} finally {

				bis.close();

			}

			

			return bos.toByteArray();

			

		} catch (FileNotFoundException ex) {

			throw new RuntimeException("pdf file not found:"+filename,ex);

		} catch (IOException ex) {

			throw new RuntimeException("Error reading pdf file:"+filename,ex);

		}

		

	}

	

	

	private void reportTransmission(FnodMetadata data) {

		String formattedData = FnodMetadataHelper.formatMetadata(data);

		File file = new File(getWorkingDir(),

				FilenameHelper.getTransmissionReportFilename(getState()));

		try {

			Writer fw = new FileWriter(file, true);

			try {

				fw.write(formattedData);

				fw.write('\n');

			} finally {

				fw.close();

			}

		} catch (IOException ex) {

			throw new RuntimeException("Failed to log tramission for: "

					+ formattedData);

		}



	}



	private void archiveFiles(String filename) { 

		archiver.moveFilesToArchive(getWorkingDir(), filename, filename.replace("dat", "pdf"));

	}



}

