package gov.va.fnod.flagloader.bl;



import gov.va.cem.common.pdfutils.ImageOnlyPdfFileWriter;
import gov.va.fnod.flagloader.be.FnodMetadata;
import gov.va.fnod.flagloader.be.ProcessState;
import gov.va.fnod.flagloader.be.VipDataEntity;
import gov.va.fnod.flagloader.bl.TaskStatusCallback.Level;
import gov.va.fnod.flagloader.dl.VipReader;




import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;



/**

 * This class is responsible for converting the VIP formatted data into a format that can be transmitted

 * to the FNOD Application. 

 * 

 * The VIP scanned data consists of data files, and scanned TIFF images.

 * 

 * 		<ol type "I">Text files contain the data come in two flavors:

 * 		<p/>

 * 		<ol>

 * 			<li>PBC (Primary Bar Code) which contains the region number</li>

 * 			<li>Document data which contains a sequenced list of TIFF file paths

 * 			    that represents the application.</li>    

 * 			<li>There is 3rd file *.job that is always empty, but which must be removed</li>

 *              after completion of document processing.</li>

 *       </ol>          

 *       </ol>

 *       

 * 	<ol type "I">The source VIP data is processed into the following elements:

 * 		<p/>

 * 		<ol>

 * 			<li>Data file for each application.</li>

 * 			<li>PDF for application, associated with data file by filename.</li>

 * 			<li>Report data recording documents converted by this process.</li>

 *  	</ol>

 *  </ol>

 * 

 * This process deletes VIP data as it is converted to FNOD data.  This acts as process check point

 * mechanism.  Since the Region number is gather from a prior file, the region and other batch 

 * information is maintained in the process state file allowing for process restart.

 */

public class TaskCreateMessage extends TaskBase {



	private VipReader vipReader = new VipReader();



	@Override

	protected void doRun() {

		log(Level.INFO,"TaskBackupSource.dorun..");
		log(Level.INFO,"Endpoint: " + getConfig().getEndPoint());
		System.out.println("Endpoint: " + getConfig().getEndPoint());

		log(Level.INFO, "Converting VIP Scanned files to FNOD format.");



		// Process all Batch text files	

		for (String filename : getInputFileList() ) {

			log(Level.INFO,"TaskBackupSource.doRun processing file: " + filename);
			System.out.println("TaskBackupSource.doRun processing file: " + filename);

			VipDataEntity vipData = getVipData(filename);

			if (!(vipData.getPbc() == null || vipData.getPbc().isEmpty())) {

				// A Primary Bar Code represents change in Region 

				onPBCFoundUpdateState(vipData);

			} 

			// changed Steve Orleski, a PBC can have a flag app attached to it and if 
			// we process 10 files from 10 different regions they will all have unique barcode
			// If its not PBC, then is a Flag Application

			createPDF(vipData);

			String formattedMetadata = getFormattedMetadata(vipData);

			writeMetadata(formattedMetadata, vipData.getBatchId());

			writeReportData(formattedMetadata);

			



			removeFileset(filename);



		}



	}



	/**

	 * VIP Outputs its data files a sequential set of Batch*.txt files which

	 * contain data needed to control the process. This routine get a current

	 * list of text files, sorts the collection, and then returns them to

	 * the calling routine. 

	 * 

	 * @return

	 */

	private String[] getInputFileList() {

		String[] filenames = getWorkingDir().list(new FilenameFilter() {

			

			public boolean accept(File dir, String name) {

				return name.matches(FilenameHelper.getSrcFileRegEx());

			}

		});



		Arrays.sort(filenames);

		

		log(Level.INFO, filenames.length+" files found to process.");



		return filenames;

	}



	/**

	 * This routine controls the reading and parsing of the VIP text file.

	 *  

	 * @param filename

	 * @return

	 */

	private VipDataEntity getVipData(String filename) {

		FileReader reader;

		try {

			reader = new FileReader(new File(getWorkingDir(), filename));

			try {

				return vipReader.read(reader);

			} finally {

				reader.close();

			}

		} catch (FileNotFoundException ex) {

			throw new RuntimeException("Failed to read file: <working dir>/"

					+ filename);

		} catch (IOException ex) {

			throw new RuntimeException("Failed to read file: <working dir>/"

					+ filename);

		}

	}



	/**

	 * Any time a new Primary Bar Code is found, process state information must

	 * be updated, this tracks the current Region data, and supports restarting

	 * the process in the event of failure.

	 * 

	 * @param vipData

	 */

	private void onPBCFoundUpdateState(VipDataEntity vipData) {

		ProcessState pc = getState();

		

		pc.setBatchDate(vipData.getTimestamp());

		pc.setBatchId(vipData.getBatchId());

		pc.setRegionNum(parsePBCForRegion(vipData.getPbc()));



		saveState();

	}



	/**

	 * This routine parse the Region code from the VIP text files PBC

	 * field.  (1% ###)

	 * 

	 * @param pbcValue

	 * @return

	 */

	private Integer parsePBCForRegion(String pbcValue) {

		Integer retval = null;

		if (pbcValue.matches("1% \\d{3}$")) {

		   retval = Integer.valueOf(pbcValue.substring(pbcValue.length() - 3));

		} else {

			throw new RuntimeException("PBC value not in expected format: "

					+ pbcValue);

		}

		return retval;

	}



	private void createPDF(VipDataEntity vipData) {

		

		ImageOnlyPdfFileWriter pdfWriter = new ImageOnlyPdfFileWriter();

		

		List<File> imageFiles = new ArrayList<File>();

		for ( String filePath: vipData.getFilePaths() ) {			

			// The image files have been move, so the path is being

			// corrected as we compile list of files to add to PDF

			

			//TODO: This is fragile - if config changed to not use these names, we are broken

			//      need to change based on sourceDirPath and workingDirPath - VIP will have

			//      written to source dir path.

			String newFiilename = filePath.replace("source","working");

			imageFiles.add(new File(newFiilename));

		}

		

		File pdfFile = new File(getWorkingDir(), vipData.getBatchId()+".pdf");				

		pdfWriter.write(pdfFile, imageFiles);

	}



	/**

	 * Data converted from VIP input are saved to support two 

	 * different operations:

	 * <ol>

	 * 		<li>Transmitting data to FNOD server</li>

	 * 		<li>Reporting processed record counts</li>

	 * </ol>

	 * For simplicity, both processes support the same data format 

	 * with one exception, files for transmission contain only one

	 * record allowing for process check pointing, the report file

	 * contains one record for each application processed.

	 * <p />

	 * 

	 * @param vipData

	 * @return

	 */

	private String getFormattedMetadata(VipDataEntity vipData) {

		FnodMetadata metadata = new FnodMetadata();



		// These value come from the PBC (Primary Bar Code) text files

		// which represent the current regional office code and process

		// batch/date

		metadata.setProcessDt(getState().getProcessDate());

		metadata.setBatchId(getState().getBatchId());

		metadata.setBatchDate(getState().getBatchDate());

		metadata.setRegionCode(getState().getRegionNum());



		// The current VIP data contains information about a specific

		// application.

		// This information is captured so that it can be transmitted to the

		// server

		// later in the process.

		metadata.setDocumentId(vipData.getBatchId());

		metadata.setDocumentDate(vipData.getTimestamp());

		metadata.setNumPages(vipData.getFilePaths().size());



		return FnodMetadataHelper.formatMetadata(metadata);



	}



	/**

	 * Write data to unique file used during transmission process.

	 * 

	 * @param formattedMetadata

	 * @param documentId

	 */

	private void writeMetadata(String formattedMetadata, Long documentId) {

		try {

			File file = new File(getWorkingDir(), documentId + ".dat");

			FileWriter fw = new FileWriter(file);

			try {

				fw.write(formattedMetadata);

			} finally {

				fw.close();

			}

		} catch (IOException ex) {

			// TODO Auto-generated catch block

			ex.printStackTrace();

		}

	}

	

	/**

	 * This method appends the formatted metadata to the end of the process report data.  Each call

	 * opens the file, writes the data then closes the file ensuring data is not lost in the event

	 * of process failure.

	 *   

	 * @param formattedMetadata

	 */

	private void writeReportData(String formattedMetadata) {

		File file = new File(getWorkingDir(), FilenameHelper.getProcessReportFilename(getState()));

		try {

			Writer fw = new FileWriter(file,true);

			try {

				fw.write(formattedMetadata);

				fw.write("\n");

			} finally {

				fw.close();

			}		

		} catch (IOException ex) {

			throw new RuntimeException("Failed to log process data: "+formattedMetadata);

		} 

	}



	/**

	 * The file set consists of a *.txt file, a *.job file and a data directory which consists 

	 * of other sub-directories.  

	 * 	  Batch###.txt - input file

	 * 	  Batch###.job - not used, but needs to be deleted

	 * 	  Batch###     - directory containing sub-directories and images. 

	 * 

	 * @param filename

	 */

	private void removeFileset(String filename) {

		String[] files = new String[3];

		

		files[0] = new File(getWorkingDir(),filename).getAbsolutePath();

		files[1] = new File(getWorkingDir(),filename.replace("txt","job")).getAbsolutePath();

		files[2] = new File(getWorkingDir(),filename.substring(0,filename.lastIndexOf('.'))).getAbsolutePath();

		

		DirectoryUtils.removeFiles(files);				

	}



}

