package gov.va.fnod.flagloader.bl;



import gov.va.fnod.flagloader.be.ProcessState;

import gov.va.fnod.flagloader.be.Tasks;



import java.io.File;

import java.io.FileInputStream;

import java.io.FileNotFoundException;

import java.io.FileOutputStream;

import java.io.IOException;

import java.text.ParseException;

import java.text.SimpleDateFormat;

import java.util.Date;

import java.util.Properties;



public class StateFileManager {



	private File stateFile;

	private SimpleDateFormat sdf = new SimpleDateFormat("yyyy.MM.dd HH.mm.ss");



	public StateFileManager(String workingDirPath) {



		if (workingDirPath == null || workingDirPath.isEmpty()) {

			throw new IllegalArgumentException(

					"Non-null/empty value expected for workingDirPath");

		}



		File dir = new File(workingDirPath);

		if (dir.isDirectory()) {

			stateFile = new File(dir, "process.state.properties");

		} else {

			throw new IllegalArgumentException(String.format(

					"Directory %s does not exist", workingDirPath));

		}

	}

	

	public boolean stateFileExists() {

		return stateFile.isFile();

	}



	public void writeStateFile(ProcessState ps) {



		// Convert to properties files

		Properties p = new Properties();



		p.put("task", ps.getTask().name());

		addDate(p, "processDate", ps.getProcessDate());



		addLong(p, "batchId", ps.getBatchId());

		addDate(p, "batchDate", ps.getBatchDate());

		addInteger(p, "regionNum", ps.getRegionNum());

		addString(p,"substate",ps.getSubState());



		try {

			FileOutputStream fos = new FileOutputStream(stateFile);

			try {

				p.store(fos, "Flag App Loader State File");

			} finally {

				if( fos!=null ) {

					fos.close();

				}

			}

		} catch (FileNotFoundException ex) {

			throw new RuntimeException("Error opening state file: "

					+ stateFile.getAbsolutePath(), ex);

		} catch (IOException ex) {

			throw new RuntimeException("Failed to save state file: "

					+ stateFile.getAbsolutePath(), ex);

		}

	}

	

	public ProcessState readStateFile() {

		Properties p = loadStateProperties();

		

		ProcessState ps = new ProcessState();



		ps.setTask(Tasks.valueOf(p.getProperty("task")));

		ps.setProcessDate(getDate(p, "processDate"));



		ps.setBatchId(getLong(p,"batchId"));

		ps.setBatchDate(getDate(p, "batchDate"));

		ps.setRegionNum(getInteger(p,"regionNum"));

		ps.setSubState(p.getProperty("substate"));



		return ps;



	}

	

	private Properties loadStateProperties() {

		Properties p = new Properties();

		FileInputStream fis;

		try {

			fis = new FileInputStream(stateFile);

			try {

				p.load(fis);

			} finally {

				fis.close();

			}

		} catch (FileNotFoundException ex) {

			throw new RuntimeException("State file not found",ex);			

		} catch (IOException ex) {

			throw new RuntimeException("Error loading state file", ex);

		}

		return p;

	}



	private void addDate(Properties p, String key, Date date) {

		if (date != null) {

			p.put(key, sdf.format(date));

		}

	}



	private Date getDate(Properties p, String key) {



		Date retval = null;

		String value = p.getProperty(key);

		if (!(value == null || value.isEmpty())) {

			try {

				retval = sdf.parse(value);

			} catch (ParseException ex) {

				throw new RuntimeException("Date Format exception on key: "

						+ key, ex);

			}

		}

		return retval;

	}



	private void addString(Properties p, String key, String value) {

		if (!(value == null || value.isEmpty())) {

			p.put(key, value);

		}

	}



	private void addInteger(Properties p, String key, Integer value) {

		if ( value != null ) {

			p.put(key, Integer.toString(value));

		}

	}

	

	private Integer getInteger(Properties p, String key) {

		Integer retval = null;

		String value = p.getProperty(key);

		if (!(value == null || value.isEmpty())) {

			try {

				retval = Integer.valueOf(value);

			} catch (NumberFormatException ex) {

				throw new RuntimeException("Integer Format exception on key: "

						+ key, ex);

			}

		}

		return retval;

	}

	

	private void addLong(Properties p, String key, Long value) {

		if (value != null) {

			p.put(key, value.toString());

		}

	}

	

	private Long getLong(Properties p, String key) {

		Long retval = null;

		String value = p.getProperty(key);

		if (!(value == null || value.isEmpty())) {

			try {

				retval = Long.valueOf(value);

			} catch (NumberFormatException ex) {

				throw new RuntimeException("Integer Format exception on key: "

						+ key, ex);

			}

		}

		return retval;

	}



	public void removeStateFile() {

		stateFile.delete();		

	}

	

	

}

