package gov.va.fnod.flagloader.bl;



import static gov.va.fnod.flagloader.bl.TaskStatusCallback.Level.INFO;
import gov.va.fnod.flagloader.be.ConfigEntity;
import gov.va.fnod.flagloader.be.ProcessState;
import gov.va.fnod.flagloader.be.Tasks;
import gov.va.fnod.flagloader.be.UserCredentials;
import gov.va.fnod.flagloader.bl.TaskStatusCallback.Level;
import gov.va.fnod.flagloader.exception.NoDataToProcessException;




import java.io.File;
import java.io.FilenameFilter;
import java.text.SimpleDateFormat;
import java.util.Date;



public class FlagAppLoader implements Runnable {

	

	private ConfigEntity config;

	private StateFileManager sfm;

	private ProcessState ps;

	private UserCredentials credentials;

	private TaskStatusCallback statusAppender;

	

	private SimpleDateFormat sdf = new SimpleDateFormat("yyyy.MM.dd.HH.mm.ss");

	

	public FlagAppLoader(ConfigEntity config, UserCredentials credentials, TaskStatusCallback statusAppender) {
		log(Level.INFO,"FlagAppLoader.constructor..");
		System.out.println("Endpoint: " + config.getEndPoint());
		System.out.println("Source dir: " + config.getSourceDirPath());
		System.out.println("Working Dir: " + config.getWorkingDirPath());
		System.out.println("Completed Dir: " + config.getCompletedDirPath());
				
		log(Level.INFO,"Web Service: " + config.getEndPoint());
		log(Level.INFO,"Source dir: " + config.getSourceDirPath());
		log(Level.INFO,"Working Dir: " + config.getWorkingDirPath());
		log(Level.INFO,"Completed Dir: " + config.getCompletedDirPath());
		
		
		
		this.config = config;

		this.sfm = new StateFileManager(config.getWorkingDirPath());

		this.credentials = credentials;

		this.statusAppender = statusAppender;

	}

	

	public void run() {

		try {

			/*

			 * If a state file exists, a prior process has not completed and

			 * must be completed before new data can be processed.

			 */

			if (sfm.stateFileExists()) {

				log(INFO, "Restarting unfinished process.");

				ps = sfm.readStateFile();

				processApplications(ps);

			}



			checkForNewDataToProcess();

			

			/*

			 * Process new data

			 */

			log(INFO, "Starting to process new data.");

			processApplications(initNewProcess());

			

			doCompleted(true);

			

		} catch (NoDataToProcessException ex) {

			doCompleted(true);

		} catch (Exception ex) {

			log(Level.ERROR, ex);

			doCompleted(false);

		}



	}

	

	private void checkForNewDataToProcess() {

		File srcDir = new File(config.getSourceDirPath());

		if (!srcDir.isDirectory()) {

			throw new IllegalArgumentException("Source directory does not exist");

		}

		

		String[] files = srcDir.list(new FilenameFilter() {

			

			public boolean accept(File dir, String filename) {				

				return filename.matches(FilenameHelper.getSrcFileRegEx());

			}

		});

		

		if ( files == null || files.length == 0 ) {

			throw new NoDataToProcessException("No new Flag Apps found to process");

		}

	}



	private void doCompleted(boolean isSuccess) {

		if ( statusAppender != null ) {

			statusAppender.onCompleted(isSuccess);

		}

	}

	

	private void log(Level level, String message ) {

		if ( statusAppender != null ) {

			statusAppender.log(level, message);

		}

	}

	

	private void log(Level level, Throwable cause ) {

		if ( statusAppender != null ) {

			statusAppender.log(level, cause);

		}

	}



	private void processApplications(ProcessState ps) {

		

		TaskBase task = TaskFactory.getInstance(ps, config, statusAppender);

		while( task != null ) {			

			if ( task instanceof TaskAuthRequired ) {

				((TaskAuthRequired)task).setCredentials(credentials);

			}

			task.run();

			task = TaskFactory.getNextInstance(ps, config, statusAppender);

		}

		

		new StateFileManager(config.getWorkingDirPath()).removeStateFile();

		statusAppender.log(Level.INFO, "Process completed for: "+sdf.format(ps.getProcessDate())+"\n" );

	}

	

	private ProcessState initNewProcess() {

		ProcessState ps = new ProcessState();

		

		ps.setTask(Tasks.BACKUP_SOURCE);

		ps.setProcessDate(new Date());

		

		return ps;

	}

	

	

}

