package gov.va.fnod.model;

import java.util.Calendar;
import java.util.List;
import java.util.Map;

import junit.framework.Assert;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * contains unit tests for Password management
 */
public class PasswordManagerTest  {
	
	private static PasswordManager passwordManager;
	
	@BeforeClass
	public static void beforeAllTests() {
		passwordManager = new PasswordManager(new PasswordHasher(), new PasswordValidator());
	}
	
	@AfterClass
	public static void afterAllTests() {
		passwordManager = null;
	}
	
	/**
	 * test changepassword 
	 * test if the new password is valid and if it exists in the passwordHistoryList
	 * errors should be empty
	 */
	@Test
	public void testChangePasswordNotExists() {
		boolean loginSuccess = true;
		Map<ErrorCode, String> errors = null;		
		String newPassword = DataHelper.getPassword();
		PasswordData passwordData = DataHelper.getPasswordData();	
		List<String> passwordHistoryList = DataHelper.getPasswordHistoryListNotExists();				
		for(String pwdChangeHistory: passwordHistoryList) {		
			passwordData.setPasswordHistory(pwdChangeHistory);
			errors = passwordManager.changePassword(newPassword, passwordData);
			if(errors != null && errors.keySet().contains(ErrorCode.PASSWORD_EXISTS_HISTORY)) {
				loginSuccess = false;
			}
			Assert.assertEquals(true, loginSuccess);
		}	
	}
	
	/**
	 * test changepassword 
	 * test if the new password is valid and if it exists in the passwordHistoryList
	 * errors should not be empty as the list contains the new password hash
	 */
	@Test
	public void testChangePasswordExists() {
		boolean loginSuccess = true;
		Map<ErrorCode, String> errors;	
		String newPassword = DataHelper.getPassword();
		PasswordData passwordData = DataHelper.getPasswordData();
		List<String> passwordHistoryList = DataHelper.getPasswordHistoryListExists();		
		for(String pwdChangeHistory: passwordHistoryList) {		
			passwordData.setPasswordHistory(pwdChangeHistory);
			errors = passwordManager.changePassword(newPassword, passwordData);
			if(errors != null && errors.keySet().contains(ErrorCode.PASSWORD_EXISTS_HISTORY)) {
				loginSuccess = false;
			}
			Assert.assertEquals(false, loginSuccess);
		}			
	}
	
	/**
	 * test the password for valid characters
	 * test should fail as the passwordList has invalid passwords
	 */
	@Test
	public void testLoginInvalidChars() {
		Map<ErrorCode, String> errors;	
		boolean loginSuccess = true;
		PasswordData passwordData = DataHelper.getPasswordData();
		DataHelper.getPasswordParams();
		for(String password: DataHelper.getInvalidPasswords()) {
			errors = passwordManager.login(password, passwordData);
			if(errors != null && errors.keySet().contains(ErrorCode.INVALID_CHARS)) {
				loginSuccess = false;
			}
			Assert.assertEquals(false, loginSuccess);
		}
	}
	
	/**
	 * test the password for valid characters
	 * test should pass as the passwordList has valid passwords
	 */
	@Test
	public void testLoginValidChars() {
		Map<ErrorCode, String> errors;	
		boolean loginSuccess = true;
		PasswordData passwordData = DataHelper.getPasswordData();
		DataHelper.getPasswordParams();
		for(String password: DataHelper.getValidPasswords()) {
			errors = passwordManager.login(password, passwordData);
			if(errors != null && errors.keySet().contains(ErrorCode.INVALID_CHARS)) {
				loginSuccess = false;
			}
			Assert.assertEquals(true, loginSuccess);
		}
	}
	
	/**
	 * test if the password matches
	 * test should fail as the password is different 
	 */
	@Test
	public void testLoginPasswordMismatch() {
		Map<ErrorCode, String> errors;	
		boolean loginSuccess = true;
		PasswordData passwordData = DataHelper.getPasswordData();
		DataHelper.getPasswordParams();
		errors = passwordManager.login(DataHelper.getPassword(), passwordData);
		if(errors != null && errors.keySet().contains(ErrorCode.PASSWORD_NOT_MATCH)) {
			loginSuccess = false;
		}
		Assert.assertEquals(false, loginSuccess);
	}
	
	/**
	 * test if the password matches
	 * test should pass as the password matches 
	 */
	@Test
	public void testLoginPasswordMatched() {
		Map<ErrorCode, String> errors;	
		boolean loginSuccess = true;
		PasswordData passwordData = DataHelper.getPasswordData();
		passwordData.setPassword("bbb79b6683d2ad41fc5d61a40a5319cd589326d2632a3713c0ef0764f4ce145cc709304d");
		DataHelper.getPasswordParams();
		errors = passwordManager.login(DataHelper.getPassword(), passwordData);
		if(errors != null && errors.keySet().contains(ErrorCode.PASSWORD_NOT_MATCH)) {
			loginSuccess = false;
		}
		Assert.assertEquals(true, loginSuccess);
	}
	
	/**
	 * test if the password expired
	 */
	@Test
	public void testLoginPasswordExpired() {
		Map<ErrorCode, String> errors;	
		boolean loginSuccess = true;
		PasswordData passwordData = DataHelper.getPasswordData();
		passwordData.setPassword("bbb79b6683d2ad41fc5d61a40a5319cd589326d2632a3713c0ef0764f4ce145cc709304d");
		Calendar expiryDate = Calendar.getInstance();
		expiryDate.set(2012, 2, 25);
		passwordData.setExpiryDate(expiryDate.getTime());
		DataHelper.getPasswordParams();
		errors = passwordManager.login(DataHelper.getPassword(), passwordData);
		if(errors != null && errors.keySet().contains(ErrorCode.PASSWORD_EXPIRED)) {
			loginSuccess = false;
		}
		Assert.assertEquals(false, loginSuccess);
	}
	
	/**
	 * test for password warning 
	 */
	@Test
	public void testLoginPasswordWarning() {
		Map<ErrorCode, String> errors;	
		boolean loginSuccess = true;
		PasswordData passwordData = DataHelper.getPasswordData();
		passwordData.setPassword("bbb79b6683d2ad41fc5d61a40a5319cd589326d2632a3713c0ef0764f4ce145cc709304d");
		Calendar expiryDate = Calendar.getInstance();
		expiryDate.add(Calendar.DATE, 5);
		passwordData.setExpiryDate(expiryDate.getTime());
		DataHelper.getPasswordParams();
		errors = passwordManager.login(DataHelper.getPassword(), passwordData);
		if(errors != null && errors.keySet().contains(ErrorCode.PASSWORD_EXPIRY_WARNING)) {
			loginSuccess = false;
		}
		Assert.assertEquals(false, loginSuccess);
	}
	
//	@Test
//	public void testInvalidUsername() {
//		Map<ErrorCode, String> errors;	
//		boolean isUsernameValid = true;
//		for(String username: DataHelper.getInvalidUsernames()) {
//			errors = passwordManager.createUser(username, password, passwordParams)(username, DataHelper.getPassword());
//			if(errors != null && errors.contains(FNODModelConstants.INVALID_USERNAME)) {
//				isUsernameValid = false;
//			}
//			Assert.assertEquals(false, isUsernameValid);
//		}
//	}
//	
//	@Test
//	public void testValidUsername() {
//		Map<ErrorCode, String> errors;	
//		boolean isUsernameValid = true;
//		for(String username: DataHelper.getValidUsernames()) {
//			errors = passwordManager.addUser(username, DataHelper.getPassword());
//			if(errors != null && errors.contains(FNODModelConstants.INVALID_USERNAME)) {
//				isUsernameValid = false;
//			}
//			Assert.assertEquals(true, isUsernameValid);
//		}		
//	}
}