package gov.va.fnod.service;

import gov.va.fnod.model.DateUtil;
import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.PasswordParameters;
import gov.va.fnod.model.SystemParameters;
import gov.va.fnod.model.UserPrivilege;
import gov.va.fnod.model.fnoddata.AppRole;
import gov.va.fnod.model.fnoddata.AppUser;
import gov.va.fnod.model.fnoddata.AppUserRole;
import gov.va.fnod.model.fnoddata.LockedStatus;
import gov.va.fnod.model.fnoddata.SystemParameter;
import gov.va.fnod.security.LoginStatus;
import gov.va.fnod.security.authorization.UserContext;
import gov.va.fnod.security.exception.AccountLockedException;

import java.security.PrivilegedActionException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ejb.EJB;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

@Stateless
public class UserSessionBean implements UserSession {

	private static final Logger logr = Logger.getLogger(UserSessionBean.class.getName());
	
	@PersistenceContext(unitName = FNODModelConstants.DEFAULT_PERSISTENCE_UNIT)
	private EntityManager em;

	@EJB
	UserAuthorizationSession userAuthorizationBean;
		
	/**
	 * get user information
	 * 
	 * @param username
	 * @return
	 */
	@Override
	public AppUser getUserInfo(String username) {
		AppUser user = (AppUser) em.createNamedQuery(AppUser.GET_APP_USER_BY_USERNAME)
				.setParameter("username", username.trim().toUpperCase())
				.getSingleResult();
		return user;
	}
	
	/**
	 * update the user
	 * 
	 * @param user
	 */
	@Override
	public void updateUser(AppUser user) {
		if (null == em.find(AppUser.class, user.getUserId())) {
			throw new IllegalArgumentException("Unknown user id: "
					+ user.getUserId());
		}
		em.merge(user);
	}

	/**
	 * get the app roles
	 * 
	 * @return
	 */
	@Override
	public List<AppRole> getAppRoles() {
		return em.createNamedQuery(AppRole.GET_ALL_APP_ROLES,AppRole.class).getResultList();
	}
	
	/**
	 * get all app users 
	 * 
	 * @return
	 */
	@Override
	public List<AppUser> getAppUsers() {
		List<AppUser> returnList =   em.createNamedQuery(AppUser.GET_ALL_APP_USERS, AppUser.class).getResultList();
		return returnList;
	}
	
	/**
	 * get app users who have pending or parked items 
	 * 
	 * @return
	 */
	@Override
	public List<AppUser> getParkingLotAppUsers() {
		List<AppUser> returnList =  em.createNamedQuery(AppUser.GET_PARKING_LOT_APP_USERS, AppUser.class)
			   .setParameter("lockedStatus", LockedStatus.PARKED)
			   .getResultList();
		return returnList;
	}

	/**
	 * Get the app users based on the search criteria
	 * 
	 * @param userNameCriteria permits percent sign '%' criteria
	 * @param roleNameCriteria selected from the role pick list.  FNODModelConstants.SELECT_ALL or null will be treated as "ALL"
	 * @param activeUserName the user that is wanting to see a user list
	 * @return A list of AppUser objects conforming to the selected criteria.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<AppUser> getAppUsers(String userNameCriteria,
			String roleNameCriteria, UserContext activeUser, boolean isSysAdmin) {
		
		List<AppUser> resultList = new ArrayList<AppUser>();
		List<AppUser> appUsers = null;
		
		// ensure the existence and appropriateness of the username for the "like" operator
		String username = userNameCriteria;
		if (username == null || username.trim().equals("")) {
			username = "%";
		}
		if (! username.contains("%")) {
			username= username.trim();
			username += "%";
		}
		
		// do one search for sysadmins and another for admins.  No one else will ever see the UI that invokes this
		// SYSADMIN
		if (isSysAdmin) {
			Query query = em.createQuery("select u from AppUser u where u.username <> :activeUsername and u.username like :username", AppUser.class);
			query.setParameter("activeUsername", activeUser.getUserName());
			query.setParameter("username", username.toUpperCase());
			appUsers = query.getResultList();
		}
		// ADMIN
		else {
			Query query = em.createNamedQuery(AppUser.GET_ALL_USERS_ADMIN_CAN_EDIT, AppUser.class);
			query.setParameter(1, activeUser.getUserName());
			query.setParameter(2, username.toUpperCase());
			appUsers = (List<AppUser>) query.getResultList();
		}
		
		// if role criteria exists, filter by it, else just use the queried list as the results
		if (roleNameCriteria == null || FNODModelConstants.SELECT_ALL.equals(roleNameCriteria)) {
			resultList = appUsers;	
		}
		else {
			// loop over fetched users and keep only those conforming to the role selection
			for (AppUser user : appUsers) {
				// now get the user's roles and check to see if they contain the role selected in the search criteria
				List<AppUserRole> appUserRoles = user.getAppUserRoles();

				for (AppUserRole role : appUserRoles) {
					// now compare the search criteria's role code with the (possibly) several roles assigned to the user
					String usersRoleCode = role.getAppRole().getRoleCd();
					if (roleNameCriteria.equals(usersRoleCode)) {
						resultList.add(user);
					}
				}
			}
		}

		return resultList;
	}

	@Override
	public AppUser createUser() {
		return new AppUser();
	}

	@Override
	public void saveUserProfile(AppUser activeUser, AppUser user, boolean isAddEditUser,
			boolean isResetPassword) {
		if (isAddEditUser) {
			// new user
			if (user.getUserId() == 0) {
				user.setUserCreatedDt(DateUtil.getCurrentTime());
				user.setUserUpdateDt(DateUtil.getCurrentTime());
				user.setStatus(LoginStatus.OPEN.toString()); //just default value, will be quickly changed
				user.setExpiryDate(DateUtil.getCurrentTime()); //just default value, will be quickly changed
				user.setPasswordChangedDate(DateUtil.getCurrentTime());
				em.persist(user);
			}
			// edit user
			else {
				user.setUserUpdateDt(DateUtil.getCurrentTime());
				if (null == em.find(AppUser.class, user.getUserId())) {
					throw new IllegalArgumentException("Unknown User Id: "
							+ user.getUserId());
				}
				em.merge(user);
			}
		} else if (isResetPassword) {
			// the security framework takes care of this now
			// TODO: Remove later
//			user.setUserUpdateDt(DateUtil.getCurrentTime());
//			user.setExpiryDate(DateUtil.getExpiryDate(0));
//			if (null == em.find(AppUser.class, user.getUserId())) {
//				throw new IllegalArgumentException("Unknown User Id: "
//						+ user.getUserId());
//			}
//			em.merge(user);
		}
		if (activeUser != null) {
			if (null == em.find(AppUser.class, activeUser.getUserId())) {
				throw new IllegalArgumentException("Unknown User Id: "
						+ activeUser.getUserId());
			}
			em.merge(activeUser);
		}
	}

	@Override
	public List<SystemParameter> getSystemParameters() {
		return em.createNamedQuery(SystemParameter.GET_ALL_SYSTEM_PARAMETERS, SystemParameter.class).getResultList();
	}

	@Override
	public PasswordParameters getPasswordParameters() {
		PasswordParameters passwordParams = new PasswordParameters();
		Map<SystemParameters, String> systemParams = mapSystemParameters(getSystemParameters());
	
		if (systemParams != null && !systemParams.isEmpty()) {
			try {
				passwordParams.setPasswordExpiryDays(Integer
						.parseInt(systemParams
								.get(SystemParameters.PASSWORD_EXPIRY_DAYS)));
				passwordParams.setNumOfPasswordHistory(Integer
						.parseInt(systemParams
								.get(SystemParameters.PASSWORD_HISTORY_COUNT)));
				passwordParams.setLoginRetries(Integer.parseInt(systemParams
						.get(SystemParameters.LOGIN_RETRIES)));
				passwordParams.setTimeLockedMin(Integer.parseInt(systemParams
						.get(SystemParameters.LOCKEDTIME_MIN)));
				passwordParams.setPasswordWarningDays(Integer
						.parseInt(systemParams
								.get(SystemParameters.PASSWORD_WARNING_DAYS)));
				passwordParams.setPasswordMinLength(Integer
						.parseInt(systemParams
								.get(SystemParameters.PASSWORD_MIN_LENGTH)));
				passwordParams.setPasswordMaxLength(Integer
						.parseInt(systemParams
								.get(SystemParameters.PASSWORD_MAX_LENGTH)));
				passwordParams.setUsernameMinLength(Integer
						.parseInt(systemParams
								.get(SystemParameters.USERNAME_MIN_LENGTH)));
				passwordParams.setUsernameMaxLength(Integer
						.parseInt(systemParams
								.get(SystemParameters.USERNAME_MAX_LENGTH)));
			} catch (NumberFormatException e) {
				throw new RuntimeException(
						"Failed to load the password parameters");
			}
		}
		return passwordParams;
	}
	
	private Map<SystemParameters, String> mapSystemParameters(
			List<SystemParameter> params) {
		Map<SystemParameters, String> systemParams = new HashMap<SystemParameters, String>();
		for (SystemParameter param : params) {
			systemParams.put(SystemParameters.valueOf(param.getParamName()),
					param.getParamValue());
		}
		return systemParams;
	}
	
	/**
	 * Submit system parameter changes
	 * @param systemParams
	 */
	@Override
	public void saveSystemParameters(List<SystemParameter> systemParams) {
		if(systemParams != null && !systemParams.isEmpty()) { 
			for(SystemParameter sysParam: systemParams) {
				if (null == em.find(SystemParameter.class, sysParam.getParamName())) {
					throw new IllegalArgumentException("Unknown systemparam ID: "
							+ sysParam.getSystemParamId());
				}
				em.merge(sysParam);
			}
		}		
	}
	
	/**
	 * delete user
	 * @param selAppUser
	 * @param activeUser 
	 */
	@Override
	public void deleteUserProfile(UserContext activeUser, AppUser selAppUser) throws PrivilegedActionException {
		if (activeUser == null && answerPrivQ(activeUser, UserPrivilege.LOCK_ACCOUNT)) {
			throw new PrivilegedActionException(new Exception("Insufficient privileges to delete a user account!"));
		}
		
		if(selAppUser != null) {
			AppUser deleteUser = em.find(AppUser.class, selAppUser.getUserId());
			em.remove(deleteUser);
		}

	}
	
	private boolean answerPrivQ(UserContext currentUser, UserPrivilege ... privileges) {
		try {
			logr.log(Level.FINE, "UserProfilePrivQuestions: {0}", privileges);
			return userAuthorizationBean.isUserAuthorized(currentUser, userAuthorizationBean
					.createSystemContext("UserProfilePrivQuestions", privileges));
		} catch (AccountLockedException ex) {
			logr.log(Level.WARNING, "UserProfilePrivQuestions: {0}", ex.getMessage());
			return false;
		}
	}
}