package gov.va.fnod.service;

import static gov.va.fnod.model.UserPrivilege.ADD_ANY_ROLE;
import static gov.va.fnod.model.UserPrivilege.ADD_USER_ROLES;
import static gov.va.fnod.model.UserPrivilege.ASSIGN_CASE;
import static gov.va.fnod.model.UserPrivilege.CHANGE_PASSWORD;
import static gov.va.fnod.model.UserPrivilege.CONNECT;
import static gov.va.fnod.model.UserPrivilege.CREATE_ACCOUNT;
import static gov.va.fnod.model.UserPrivilege.DELETE_ACCOUNT;
import static gov.va.fnod.model.UserPrivilege.DOWNLOAD_REPORT;
import static gov.va.fnod.model.UserPrivilege.EXECUTE_AUDIT;
import static gov.va.fnod.model.UserPrivilege.EXPORT_DATA;
import static gov.va.fnod.model.UserPrivilege.FLAG_APP_LOAD;
import static gov.va.fnod.model.UserPrivilege.FNOD_LOAD;
import static gov.va.fnod.model.UserPrivilege.LOCK_ACCOUNT;
import static gov.va.fnod.model.UserPrivilege.LOGIN;
import static gov.va.fnod.model.UserPrivilege.PROCESS_CASE;
import static gov.va.fnod.model.UserPrivilege.RESET_PASSWORD;
import static gov.va.fnod.model.UserPrivilege.RUN_REPORT;
import static gov.va.fnod.model.UserPrivilege.RUN_USER_REPORT;
import static gov.va.fnod.model.UserRole.ADMIN;
import static gov.va.fnod.model.UserRole.FLAG_APP_LOADER;
import static gov.va.fnod.model.UserRole.FNOD_DATA_LOADER;
import static gov.va.fnod.model.UserRole.OFFICE_SUPERVISOR;
import static gov.va.fnod.model.UserRole.PSA;
import static gov.va.fnod.model.UserRole.SYSTEM_ADMIN;
import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.UserPrivilege;
import gov.va.fnod.model.UserRole;
import gov.va.fnod.model.fnoddata.SystemParameter;
import gov.va.fnod.security.authentication.LoginConstraints;
import gov.va.fnod.security.authorization.SecurityContext;
import gov.va.fnod.security.service.PswdChangeConstraints;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

/**
 * This service is intended to used exclusively by the security sub-system and is 
 * registered in the SecurityInitializer.
 *  
 * @author Jim Lindelof
 *
 */
@Stateless
public class SystemContextSessionBean implements SystemContextSession {

	@PersistenceContext (unitName = FNODModelConstants.DEFAULT_PERSISTENCE_UNIT)
	EntityManager em;
	
	private static final Map<UserRole, Set<UserPrivilege>> rolePrivMap;

	private static final SecurityContext<UserPrivilege> canLoginContext;
	private static final SecurityContext<UserPrivilege> canConnectContext;

	private static final SecurityContext<UserPrivilege> canChangePswdContext;
	private static final SecurityContext<UserPrivilege> canResetPswdContext;
	private static final SecurityContext<UserPrivilege> canLockUserContext;
	
	public SystemContextSessionBean() {
		super();
	}

	/*
	 * Defining security contexts
	 */
	static {

		/*
		 * Setup the Role Privilege Map
		 */
		rolePrivMap = new EnumMap<UserRole, Set<UserPrivilege>>(
				UserRole.class);
		
		// System Administrator
		
		rolePrivMap.put(SYSTEM_ADMIN, 
				EnumSet.of(	LOGIN,
							CHANGE_PASSWORD,
							CREATE_ACCOUNT,
							DELETE_ACCOUNT,
							ADD_ANY_ROLE, 
							RESET_PASSWORD,
							LOCK_ACCOUNT));

		// Administrator
		
		rolePrivMap.put(ADMIN, 
				EnumSet.of(	LOGIN, 
							CHANGE_PASSWORD,
							ADD_USER_ROLES, 
							RESET_PASSWORD,
							LOCK_ACCOUNT));
		
		// Office Supervisor
		
		rolePrivMap.put(OFFICE_SUPERVISOR,				
				EnumSet.of(	LOGIN, 
							CHANGE_PASSWORD,
							RUN_REPORT,
							DOWNLOAD_REPORT,
							EXPORT_DATA,
							EXECUTE_AUDIT,
							PROCESS_CASE,
							ASSIGN_CASE));
		
		// PSA 
		rolePrivMap.put(PSA,				
				EnumSet.of(	LOGIN, 
							CHANGE_PASSWORD,
							RUN_USER_REPORT,
							PROCESS_CASE ));
		
		// Service
		rolePrivMap.put(FNOD_DATA_LOADER,				
				EnumSet.of(	CONNECT,
						    FNOD_LOAD));
		
		// Flag Application Loader
		rolePrivMap.put(FLAG_APP_LOADER,				
				EnumSet.of(CONNECT, 
						   FLAG_APP_LOAD,
						   CHANGE_PASSWORD));		
		 
		/*
		 * Setup specific privilege contexts 
		 */

		canLoginContext = new SecurityContext<UserPrivilege>() {

			@Override
			public String getContextLabel() {
				return "CAN_LOGIN";
			}

			@Override
			public Set<UserPrivilege> requestPrivileges() {
				return EnumSet.of(LOGIN);
			}

			@Override
			public Set<UserPrivilege> requestExclusionPrivileges() {
				return EnumSet.noneOf(UserPrivilege.class);
			}
		};
		
		canConnectContext = new SecurityContext<UserPrivilege>() {

			@Override
			public String getContextLabel() {
				return "CAN_CONNECT";
			}

			@Override
			public Set<UserPrivilege> requestPrivileges() {
				return EnumSet.of(CONNECT);
			}

			@Override
			public Set<UserPrivilege> requestExclusionPrivileges() {
				return EnumSet.noneOf(UserPrivilege.class);
			}
		};

		canChangePswdContext = new SecurityContext<UserPrivilege>() {

			@Override
			public String getContextLabel() {
				return "CAN_CHANGE_PASSWORD";
			}

			@Override
			public Set<UserPrivilege> requestPrivileges() {
				return EnumSet.of(CHANGE_PASSWORD);
			}

			@Override
			public Set<UserPrivilege> requestExclusionPrivileges() {
				return EnumSet.noneOf(UserPrivilege.class);
			}
		};
		
		canResetPswdContext = new SecurityContext<UserPrivilege>() {

			@Override
			public String getContextLabel() {
				return "CAN_RESET_PASSWORD";
			}

			@Override
			public Set<UserPrivilege> requestPrivileges() {
				return EnumSet.of(RESET_PASSWORD);
			}

			@Override
			public Set<UserPrivilege> requestExclusionPrivileges() {
				return EnumSet.noneOf(UserPrivilege.class);
			}
		};
		
		
		canLockUserContext = new SecurityContext<UserPrivilege>() {

			@Override
			public String getContextLabel() {
				return "CAN_LOCK_ACCOUNT";
			}

			@Override
			public Set<UserPrivilege> requestPrivileges() {
				return EnumSet.of(LOCK_ACCOUNT);
			}

			@Override
			public Set<UserPrivilege> requestExclusionPrivileges() {
				return EnumSet.noneOf(UserPrivilege.class);
			}
		};
	}
	
	@Override
	public PswdChangeConstraints getPswdConstraints() {
		
		final Map<String, String> paramMap = buildSystemParameterMap(Arrays
				.asList("PASSWORD_CHANGE_LIMIT_HRS", "PASSWORD_EXPIRY_DAYS",
						"PASSWORD_HISTORY_COUNT", "PASSWORD_MAX_LENGTH",
						"PASSWORD_MIN_LENGTH"));
		
		return new PswdChangeConstraints() {
			private Map<String,String> map = paramMap;
			
			@Override
			public int getMinPswdLength() {			
				return Integer.parseInt(map.get("PASSWORD_MIN_LENGTH"));
			}

			@Override
			public int getMaxPswdLength() {
				return Integer.parseInt(map.get("PASSWORD_MAX_LENGTH"));
			}

			@Override
			public String getCharValidatingRegEx() {
				// Any characters, not including space
				return "[\\p{Alnum}\\p{Punct}]+";
			}

			@Override
			public int getMinHrsBeforeNextPswdChange() {
				return Integer.parseInt(map.get("PASSWORD_CHANGE_LIMIT_HRS"));
			}

			@Override
			public int getMaxHistoryEntries() {
				return Integer.parseInt(map.get("PASSWORD_HISTORY_COUNT"));
			}

			@Override
			public int getExpiryDays() {
				return Integer.parseInt(map.get("PASSWORD_EXPIRY_DAYS"));
			}
			
		};
	}

	@Override
	public LoginConstraints getLoginConstraints() {
		
		//TypedQuery<SystemParameter> qry = em.createNamedQuery(SystemParameter.GET_LOGIN_PARAMETERS,SystemParameter.class);
		//List<SystemParameter> list = getSystem
		
		final Map<String, String> paramMap = buildSystemParameterMap(Arrays
				.asList("LOGIN_RETRIES", "LOCKEDTIME_MIN",
						"PASSWORD_WARNING_DAYS"));
		
		return new LoginConstraints() {
			
			private Map<String,String> map = paramMap;
			
			@Override
			public int getTempLockDurationMin() {
				return Integer.parseInt(map.get("LOCKEDTIME_MIN"));
			}

			@Override
			public int getMaxLoginAttempts() {
				return Integer.parseInt(map.get("LOGIN_RETRIES"));
			}

			@Override
			public int getPswdExpireWarningDays() {
				return Integer.parseInt(map.get("PASSWORD_WARNING_DAYS"));
			}
			
		};
	}

	@Override
	public Map<UserRole, Set<UserPrivilege>> getRolePrivMap() {
		return rolePrivMap;
	}

	@Override
	public SecurityContext<UserPrivilege> getCanLoginContext() {
		return canLoginContext;
	}

	@Override
	public SecurityContext<UserPrivilege> getCanConnectContext() {
		return canConnectContext;
	}

	@Override
	public SecurityContext<UserPrivilege> getCanChangePswdContext() {
		return canChangePswdContext;
	}

	@Override
	public SecurityContext<UserPrivilege> getCanResetPswdContext() {
		return canResetPswdContext;
	}

	@Override
	public SecurityContext<UserPrivilege> getCanLockUserContext() {
		return canLockUserContext;
	}

	private Map<String, String> buildSystemParameterMap(List<String> paramNames) {
		
		HashMap<String,String> map = new HashMap<String,String>();
		
		for( String name: paramNames) {
			SystemParameter param = em.find(SystemParameter.class, name);
			map.put(name, param.getParamValue());
		}
				
		return map;
	}
}
