package gov.va.fnod.service;

import gov.va.fnod.model.CaseLinkData;
import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.fnoddata.CaseLink;
import gov.va.fnod.model.fnoddata.CaseType;
import gov.va.fnod.model.fnoddata.LockedStatus;
import gov.va.fnod.model.fnoddata.Person;
import gov.va.fnod.model.fnoddata.SourceCaseTypeMap;
import gov.va.fnod.model.fnoddata.SourceSystem;
import gov.va.fnod.util.DatabaseUtil;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Resource;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

/**
 * Session Bean used for retrieving lookup values
 */
@Stateless(name = "SourceCaseTypeMapBean")
public class SourceCaseTypeMapSessionBean implements SourceCaseTypeMapSession {
	private static final Logger logr = Logger.getLogger(SourceCaseTypeMapSessionBean.class.getName());

	
	@PersistenceContext (unitName = FNODModelConstants.DEFAULT_PERSISTENCE_UNIT)
	private EntityManager em;
	
	
	@Resource(name = "parkingLotSQLUserOnly")
	private String parkingLotSQLUserOnly;
	
	@Resource(name = "parkingLotSQLUserAndType")
	private String parkingLotSQLUserAndType;

	@Resource(name = "parkingLotSQLAllUser")
	private String parkingLotSQLAllUser;

	@Resource(name = "parkingLotSQLTypeAllUser")
	private String parkingLotSQLTypeAllUser;

	
	
	
	/**
	 * 
	 * 
	 * get the list of all SourceCaseTypeMap objects
	 * 
	 * @return
	 */
	@Override
	public List<SourceCaseTypeMap> getSourceCaseTypeMapList() {
		return em.createNamedQuery(SourceCaseTypeMap.GET_ALL_SOURCE_CASE_TYPE_MAPS,SourceCaseTypeMap.class).getResultList();
	}

	/**
	 * get the list of all CaseData objects matching the SourcecaseTypeMap, assigned to the given user
	 * 
	 * @return
	 */
	@Override
	public List<CaseLinkData> getCaseLinkData(SourceCaseTypeMap sourceCaseTypeMap, String username) {
		List<CaseLinkData> caseLinks = null;
		
		if (sourceCaseTypeMap == null) {
			caseLinks = em.createNamedQuery(CaseLink.GET_CASE_LINK_DATA_BY_USER, CaseLinkData.class)
				.setParameter("username", username)
				.setParameter("lockedStatus", LockedStatus.PARKED)
				.getResultList();
		}
		else {
			caseLinks = em.createNamedQuery(CaseLink.GET_CASE_LINK_DATA_BY_SOURCE_CASE_TYPE_MAP, CaseLinkData.class)
				.setParameter("sourceSystem", sourceCaseTypeMap.getSourceSystem())
				.setParameter("caseType", sourceCaseTypeMap.getCaseType())
				.setParameter("username", username)
				.setParameter("lockedStatus", LockedStatus.PARKED)
				.getResultList();
		}
		
		
		return caseLinks;
	}
	
	
	@Override
	public List<CaseLinkData> getCaseLinkData2(SourceCaseTypeMap sourceCaseTypeMap, String username) {
		List<CaseLinkData> caseLinks = new ArrayList<CaseLinkData>();
		Long caseType = 0L;
		String sql = parkingLotSQLUserOnly;
		
		if(sourceCaseTypeMap != null) {
			caseType = (sourceCaseTypeMap.getSourceCaseTypeMapId() > 0) ? sourceCaseTypeMap.getSourceCaseTypeMapId() : 0l;
			sql = parkingLotSQLUserAndType;
		}
		
		Connection conn = null;

		PreparedStatement stmt = null;

		ResultSet rs = null;
		
		try {

			conn = DatabaseUtil.getConnection();
			
			stmt = conn.prepareStatement(sql);
			stmt.setString(1, username);
			
			logr.log(Level.FINEST, "SQL Parameter: {0}", username);
			
			logr.log(Level.FINEST, "SQL: {1}", sql);
			
			if(caseType > 0) { 
				stmt.setLong(2, caseType);
				logr.log(Level.FINEST, "SQL Parameter: {2}", caseType);
			}
			

			rs = stmt.executeQuery();



			while (rs.next()) {
				CaseLinkData caseLink = new CaseLinkData();
				
				caseLink.setCaseId(rs.getLong("CASE_ID"));
				caseLink.setCaseLockedDt(rs.getTimestamp("CASE_LOCKED_DT"));
				caseLink.setUserName(rs.getString("USERNAME"));
				caseLink.setDescription(rs.getString("DESCRIPTION"));
				caseLink.setNmi(rs.getString("NMI"));
				caseLink.setSensitive(rs.getString("SENSITIVE"));
				Person p = new Person();
				p.setFirstName(rs.getString("VETERAN_FIRST_NAME"));
				p.setMiddleName(rs.getString("VETERAN_MIDDLE_NAME"));
				p.setLastName(rs.getString("VETERAN_LAST_NAME"));
				caseLink.setVeteran(p);
				
				// timestamp fields do not display correctly using simple date format
				// this is for the parking lot page
				caseLink.copyTimeStampToDateFields();
				caseLinks.add(caseLink);
			}
		} catch (SQLException e) {

			e.printStackTrace();

			throw new RuntimeException(e);

		} finally {

			releaseSQLItems(rs, stmt);

			if (conn != null) {

				DatabaseUtil.releaseConnection(conn);

			}

		}
		
		
		return caseLinks;
	}
	
	@Override
	public List<CaseLinkData> getCaseLinkDataAllUser(SourceCaseTypeMap sourceCaseTypeMap) {
		List<CaseLinkData> caseLinks = new ArrayList<CaseLinkData>();
		Long caseType = 0L;
		String sql = parkingLotSQLAllUser;
		
		if(sourceCaseTypeMap != null) {
			caseType = (sourceCaseTypeMap.getSourceCaseTypeMapId() > 0) ? sourceCaseTypeMap.getSourceCaseTypeMapId() : 0l;
			sql = parkingLotSQLTypeAllUser;
		}
		
		Connection conn = null;

		PreparedStatement stmt = null;

		ResultSet rs = null;
		
		try {

			conn = DatabaseUtil.getConnection();
			
			stmt = conn.prepareStatement(sql);
			
			
			logr.log(Level.FINEST, "SQL: ", sql);
			
			if(caseType > 0) { 
				stmt.setLong(1, caseType);
				logr.log(Level.FINEST, "SQL Parameter: {1}", caseType);
			}
			

			rs = stmt.executeQuery();



			while (rs.next()) {
				CaseLinkData caseLink = new CaseLinkData();
				
				caseLink.setCaseId(rs.getLong("CASE_ID"));
				caseLink.setCaseLockedDt(rs.getTimestamp("CASE_LOCKED_DT"));
				caseLink.setUserName(rs.getString("USERNAME"));
				caseLink.setDescription(rs.getString("DESCRIPTION"));
				caseLink.setNmi(rs.getString("NMI"));
				caseLink.setSensitive(rs.getString("SENSITIVE"));
				Person p = new Person();
				p.setFirstName(rs.getString("VETERAN_FIRST_NAME"));
				p.setMiddleName(rs.getString("VETERAN_MIDDLE_NAME"));
				p.setLastName(rs.getString("VETERAN_LAST_NAME"));
				caseLink.setVeteran(p);
				
				// timestamp fields do not display correctly using simple date format
				// this is for the parking lot page
				caseLink.copyTimeStampToDateFields();
				caseLinks.add(caseLink);
			}
		} catch (SQLException e) {

			e.printStackTrace();

			throw new RuntimeException(e);

		} finally {

			releaseSQLItems(rs, stmt);

			if (conn != null) {

				DatabaseUtil.releaseConnection(conn);

			}

		}
		
		
		return caseLinks;
	}
	
	
	/**
	 * get the list of all SourcecaseTypeMap objects with the given sourceSystem and caseType
	 * 
	 * @param sourceSystem The source system to match
	 * @param caseTyep The case type to match
	 * @return
	 */
	@Override
	public SourceCaseTypeMap getSourceCaseTypeMap(SourceSystem sourceSystem, CaseType caseType) {
		SourceCaseTypeMap sourceCaseTypeMap = null;
		
		try {
			sourceCaseTypeMap = (SourceCaseTypeMap)em.createNamedQuery(SourceCaseTypeMap.GET_SOURCE_CASE_TYPE_MAP_BY_SOURCE_CASE_TYPE)
					.setParameter("sourceSystem", sourceSystem)
					.setParameter("caseType", caseType)
					.getSingleResult();
		}
		
		catch (Exception e) {
			e.printStackTrace();
		}
		
		return sourceCaseTypeMap;
	}
	
	private void releaseSQLItems(ResultSet rs, Statement stmt) {

		if (rs != null) {

			try {

				rs.close();

			} catch (SQLException e) {

				logr.log(Level.INFO, "Problem closing result set", e);

			}

		}



		if (stmt != null) {

			try {

				stmt.close();

			} catch (SQLException e) {

				logr.log(Level.INFO, "Problem closing statement", e);

			}

		}

	}
}
