package gov.va.fnod.service;

import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.SearchFNODRecordsCriteria;
import gov.va.fnod.model.SearchFNODRecordsResult;
import gov.va.fnod.model.fnoddata.CaseData;
import gov.va.fnod.model.fnoddata.FnodRecord;
import gov.va.fnod.model.fnoddata.LockedStatus;
import gov.va.fnod.model.fnoddata.Person;
import gov.va.fnod.model.fnoddata.SourceCaseTypeMap;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.ejb.EJB;
import javax.ejb.Stateful;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.PersistenceContextType;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.JoinType;
import javax.persistence.criteria.ParameterExpression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang.time.DateUtils;
import org.apache.log4j.Logger;

/**
 * Session Bean implementation class SearchDatabaseSessionBean
 */
@Stateful(name = "SearchFNODRecords")
public class SearchFNODRecordsSessionBean implements SearchFNODRecordsSession {

	@PersistenceContext (unitName = FNODModelConstants.DEFAULT_PERSISTENCE_UNIT, type=PersistenceContextType.EXTENDED)
	private EntityManager em;
	private Logger log = Logger.getLogger(SearchFNODRecordsSessionBean.class);
	
	@EJB
	SourceCaseTypeMapSession sourceCaseTypeMapSession;
	
    /**
     * Default constructor. 
     */
    public SearchFNODRecordsSessionBean() {
    }

	/**
	 * This method searches for matching FNOD records given the criteria, in the following
	 * order of precedence:
	 * 
	 * 	1) Distinguishing ID Fields
	 * 		a) Case ID
	 * 		b) SSN - Social Security Number
	 * 		c) VCN - Veteran Claim Number
	 * 		d) MSN - Military Service Number
	 *  2) By Name
	 *  	a) First, Last, Middle and Suffix
	 *  3) By FNOD Created Date
	 *  4) By Birth and Death Dates
	 *  
	 *  Note:  Fields with % support wild cards.  This excludes date fields.  
	 *  	   In all wild card searches, the first character must be provided.
	 *  
	 *  @author VHAISLHODGEJ
	 */
    @Override
	public List<SearchFNODRecordsResult> searchDatabase(SearchFNODRecordsCriteria criteria) {
    	List<FnodRecord> resultList = null;
    	
    	try {
			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<FnodRecord> c = cb.createQuery(FnodRecord.class);
			Root<FnodRecord> from = c.from(FnodRecord.class);
			c.select(from);
			from.join("caseLink", JoinType.INNER);
			c.orderBy(cb.desc(from.get("fnodCreatedDt")));
			List<Predicate> predicates = new ArrayList<Predicate>();
			
			boolean criteriaSatisfied = false;
			boolean parametersSatisfied = false;
			
	    	// return only COMPLETE records
			ParameterExpression<LockedStatus> p1 = cb.parameter(LockedStatus.class, "lockedStatus");
	        Expression<String> path1 = from.get("caseLink").get("lockedStatus");
	        Predicate notEquals = cb.equal(path1, p1);
			predicates.add(notEquals);
			
	    	// by user
	    	if (criteria.getSelectedUserName() != null) {
				ParameterExpression<String> p = cb.parameter(String.class, "username");
		        Expression<String> path = from.get("username");
		        Predicate equals = cb.equal(path, p);
				predicates.add(equals);
	    	}
			
			// by case id
	    	if (criteria.getCaseId() != null) {
				ParameterExpression<String> p = cb.parameter(String.class, "caseId");
		        Expression<String> path = from.get("caseLink").get("caseId");
		        Predicate equals = cb.equal(path, p);
				predicates.add(equals);
				criteriaSatisfied = true;
	    	}
	    	
	    	// by ssn
	    	if (!criteriaSatisfied && criteria.getSocialSecurityNumber() != null) {
				ParameterExpression<String> p = cb.parameter(String.class, "ssn");
		        Expression<String> path = from.get("veteranPerson").get("socialSecurityNumber");
		        Predicate equals = cb.equal(path, p);
				predicates.add(equals);
				criteriaSatisfied = true;
	    	}
	    	
	    	// by Veteran Claim Number
	    	if (!criteriaSatisfied && criteria.getVeteranClaimNumber() != null) {
				ParameterExpression<String> p = cb.parameter(String.class, "veteranClaimNum");
		        Expression<String> path = from.get("veteranClaimNum");
		        Predicate equals = cb.equal(path, p);
				predicates.add(equals);
				criteriaSatisfied = true;
	    	}
	    	
	    	// by Military Service Number
	    	if (!criteriaSatisfied && criteria.getMilitaryServiceNumber() != null) {
				ParameterExpression<String> p = cb.parameter(String.class, "militaryServiceNum");
		        Expression<String> path = from.get("militaryServiceNum");
		        Predicate equals = cb.equal(path, p);
				predicates.add(equals);
				criteriaSatisfied = true;
	    	}
	    	
	    	// by Name and Dates
	    	if (!criteriaSatisfied) {
				// first name
	    		if (criteria.getFirstName() != null) {
		    		Predicate cr = null;
		    		if (containsWildcard(criteria.getFirstName())) {
		    			cr = cb.like(from.get("veteranPerson").<String>get("firstName"), cb.parameter(String.class, "firstName"));
		    		}
		    		else {
		    			cr = cb.equal(from.get("veteranPerson").<String>get("firstName"), cb.parameter(String.class, "firstName"));
		    		}
					predicates.add(cr);
	    		}
		
				// last name
	    		if (criteria.getLastName() != null) {
		    		Predicate cr = null;
		    		if (containsWildcard(criteria.getLastName())) {
		    			cr = cb.like(from.get("veteranPerson").<String>get("lastName"), cb.parameter(String.class, "lastName"));
		    		}
		    		else {
		    			cr = cb.equal(from.get("veteranPerson").<String>get("lastName"), cb.parameter(String.class, "lastName"));
		    		}
					predicates.add(cr);
	    		}
	    		
				// middle name
	    		if (criteria.getMiddleName() != null) {
		    		Predicate cr = null;
		    		if (containsWildcard(criteria.getMiddleName())) {
		    			cr = cb.like(from.get("veteranPerson").<String>get("middleName"), cb.parameter(String.class, "middleName"));
		    		}
		    		else {
		    			cr = cb.equal(from.get("veteranPerson").<String>get("middleName"), cb.parameter(String.class, "middleName"));
		    		}
					predicates.add(cr);
	    		}
	    		
				// suffix
	    		if (criteria.getSuffixName() != null) {
		    		Predicate cr = null;
		    		if (containsWildcard(criteria.getSuffixName())) {
		    			cr = cb.like(from.get("veteranPerson").<String>get("suffixName"), cb.parameter(String.class, "suffixName"));
		    		}
		    		else {
		    			cr = cb.equal(from.get("veteranPerson").<String>get("suffixName"), cb.parameter(String.class, "suffixName"));
		    		}
					predicates.add(cr);
	    		}
	    		
				// DOB
	    		if (criteria.getBirthDate() != null) {
	    			Expression<Timestamp> exp1 = from.get("veteranPerson").get("birthDt");
					Predicate fromDt = cb.greaterThanOrEqualTo( exp1, cb.parameter(Timestamp.class, "birthDtFrom"));
					predicates.add(fromDt);
					
					Expression<Timestamp> exp2 = from.get("veteranPerson").get("birthDt");
					Predicate toDt = cb.lessThan(exp2, cb.parameter(Timestamp.class, "birthDtTo"));
					predicates.add(toDt);					
	    		}
	    		
				// DOD
	    		if (criteria.getDeathDate() != null) {
	    			Expression<Timestamp> exp1 = from.get("veteranPerson").get("deathDt");
					Predicate fromDt = cb.greaterThanOrEqualTo( exp1, cb.parameter(Timestamp.class, "deathDtFrom"));
					predicates.add(fromDt);
					
					Expression<Timestamp> exp2 = from.get("veteranPerson").get("deathDt");
					Predicate toDt = cb.lessThan(exp2, cb.parameter(Timestamp.class, "deathDtTo"));
					predicates.add(toDt);
	    		}
	    		
				// FNOD Created Date
	    		if (criteria.getFnodCreatedDate() != null) {
	    			Expression<Timestamp> exp1 = from.get("fnodCreatedDt");
					Predicate fromDt = cb.greaterThanOrEqualTo( exp1, cb.parameter(Timestamp.class, "fnodCreatedDtFrom"));
					predicates.add(fromDt);
					
					Expression<Timestamp> exp2 = from.get("fnodCreatedDt");
					Predicate toDt = cb.lessThan(exp2, cb.parameter(Timestamp.class, "fnodCreatedDtTo"));
					predicates.add(toDt);
	    		}
	    	}
	    	
	    	if (predicates.size() == 1) {
	    		c.where(predicates.get(0));
	    	}
	    	else {
	    		c.where(cb.and(predicates.toArray(new Predicate[0])));
	    	}
	    	
	    	
	    	TypedQuery<FnodRecord> q = em.createQuery(c);
	    	
	    	q.setParameter("lockedStatus", LockedStatus.COMPLETE);
	    	
	    	if (criteria.getSelectedUserName() != null) {
	    		q.setParameter("username", criteria.getSelectedUserName());
	    	}
	    	if (criteria.getCaseId() != null) {
	    		q.setParameter("caseId", criteria.getCaseId());
	    		parametersSatisfied = true;
	    	}
	    	if (!parametersSatisfied && criteria.getSocialSecurityNumber() != null) {
	    		q.setParameter("ssn", criteria.getSocialSecurityNumber());
	    		parametersSatisfied = true;
	    	}
	    	if (!parametersSatisfied && criteria.getVeteranClaimNumber() != null) {
	    		q.setParameter("veteranClaimNum", criteria.getVeteranClaimNumber());
	    		parametersSatisfied = true;
	    	}
	    	if (!parametersSatisfied && criteria.getMilitaryServiceNumber() != null) {
	    		q.setParameter("militaryServiceNum", criteria.getMilitaryServiceNumber());
	    		parametersSatisfied = true;
	    	}
	    	if (!parametersSatisfied) {
	    		if (criteria.getFirstName() != null) {
					q.setParameter("firstName", criteria.getFirstName());
				}
				if (criteria.getLastName() != null) {
					q.setParameter("lastName", criteria.getLastName());
				}
				if (criteria.getMiddleName() != null) {
					q.setParameter("middleName", criteria.getMiddleName());
				}
				if (criteria.getSuffixName() != null) {
					q.setParameter("suffixName", criteria.getSuffixName());
				}
				if (criteria.getBirthDate() != null) {
	    			Timestamp startOfDay = new Timestamp(DateUtils.truncate(criteria.getBirthDate(), Calendar.DAY_OF_MONTH).getTime());
	    			Calendar cal = Calendar.getInstance();
	    			cal.setTime(startOfDay);
	    			cal.add(Calendar.DAY_OF_MONTH, 1);
	    			Timestamp endOfDay = new Timestamp(cal.getTimeInMillis());
	    			
					q.setParameter("birthDtFrom", startOfDay);
					q.setParameter("birthDtTo", endOfDay);
				}
				if (criteria.getDeathDate() != null) {
	    			Timestamp startOfDay = new Timestamp(DateUtils.truncate(criteria.getDeathDate(), Calendar.DAY_OF_MONTH).getTime());
	    			Calendar cal = Calendar.getInstance();
	    			cal.setTime(startOfDay);
	    			cal.add(Calendar.DAY_OF_MONTH, 1);
	    			Timestamp endOfDay = new Timestamp(cal.getTimeInMillis());
	    			
					q.setParameter("deathDtFrom", startOfDay);
					q.setParameter("deathDtTo", endOfDay);
				}
				if (criteria.getFnodCreatedDate() != null) {
	    			Timestamp startOfDay = new Timestamp(DateUtils.truncate(criteria.getFnodCreatedDate(), Calendar.DAY_OF_MONTH).getTime());
	    			Calendar cal = Calendar.getInstance();
	    			cal.setTime(startOfDay);
	    			cal.add(Calendar.DAY_OF_MONTH, 1);
	    			Timestamp endOfDay = new Timestamp(cal.getTimeInMillis());
	    			
					q.setParameter("fnodCreatedDtFrom", startOfDay);
					q.setParameter("fnodCreatedDtTo", endOfDay);
				}
	    	}
	    	
	    	q.setMaxResults(SearchFNODRecordsSession.MAX_ROWS_TO_FETCH);
	    	resultList = q.getResultList();
    	}
	    catch (Exception ex) {
	    	log.error("Failure trying to query FNOD Records.", ex);
	    	System.out.println(ex.getMessage());
	    }
    	
    	return convert(resultList);
	}
    
//    public List<SearchFNODRecordsResult> searchDatabase(SearchFNODRecordsCriteria criteria) {
//    	List<CaseData> resultList = null;
//    	
//    	try {
//			CriteriaBuilder cb = em.getCriteriaBuilder();
//			CriteriaQuery<CaseData> c = cb.createQuery(CaseData.class);
//			Root<CaseData> from = c.from(CaseData.class);
//			c.select(from);
//			from.join("caseLink", JoinType.INNER);
//			c.orderBy(cb.desc(from.get("entryDt")));
//			List<Predicate> predicates = new ArrayList<Predicate>();
//			
//			boolean criteriaSatisfied = false;
//			boolean parametersSatisfied = false;
//			
//	    	// return only COMPLETE records
//			ParameterExpression<LockedStatus> p1 = cb.parameter(LockedStatus.class, "lockedStatus");
//	        Expression<String> path1 = from.get("caseLink").get("lockedStatus");
//	        Predicate notEquals = cb.equal(path1, p1);
//			predicates.add(notEquals);
//			
//	    	// by user
//	    	if (criteria.getSelectedUserName() != null) {
//				ParameterExpression<String> p = cb.parameter(String.class, "username");
//		        Expression<String> path = from.get("username");
//		        Predicate equals = cb.equal(path, p);
//				predicates.add(equals);
//	    	}
//			
//			// by case id
//	    	if (criteria.getCaseId() != null) {
//				ParameterExpression<String> p = cb.parameter(String.class, "caseId");
//		        Expression<String> path = from.get("caseLink").get("caseId");
//		        Predicate equals = cb.equal(path, p);
//				predicates.add(equals);
//				criteriaSatisfied = true;
//	    	}
//	    	
//	    	// by ssn
//	    	if (!criteriaSatisfied && criteria.getSocialSecurityNumber() != null) {
//				ParameterExpression<String> p = cb.parameter(String.class, "ssn");
//		        Expression<String> path = from.get("veteranPerson").get("socialSecurityNumber");
//		        Predicate equals = cb.equal(path, p);
//				predicates.add(equals);
//				criteriaSatisfied = true;
//	    	}
//	    	
//	    	// by Veteran Claim Number
//	    	if (!criteriaSatisfied && criteria.getVeteranClaimNumber() != null) {
//				ParameterExpression<String> p = cb.parameter(String.class, "veteranClaimNum");
//		        Expression<String> path = from.get("veteranClaimNum");
//		        Predicate equals = cb.equal(path, p);
//				predicates.add(equals);
//				criteriaSatisfied = true;
//	    	}
//	    	
//	    	// by Military Service Number
//	    	if (!criteriaSatisfied && criteria.getMilitaryServiceNumber() != null) {
//				ParameterExpression<String> p = cb.parameter(String.class, "militaryServiceNum");
//		        Expression<String> path = from.get("militaryServiceNum");
//		        Predicate equals = cb.equal(path, p);
//				predicates.add(equals);
//				criteriaSatisfied = true;
//	    	}
//	    	
//	    	// by Name and Dates
//	    	if (!criteriaSatisfied) {
//				// first name
//	    		if (criteria.getFirstName() != null) {
//		    		Predicate cr = null;
//		    		if (containsWildcard(criteria.getFirstName())) {
//		    			cr = cb.like(from.get("veteranPerson").<String>get("firstName"), cb.parameter(String.class, "firstName"));
//		    		}
//		    		else {
//		    			cr = cb.equal(from.get("veteranPerson").<String>get("firstName"), cb.parameter(String.class, "firstName"));
//		    		}
//					predicates.add(cr);
//	    		}
//		
//				// last name
//	    		if (criteria.getLastName() != null) {
//		    		Predicate cr = null;
//		    		if (containsWildcard(criteria.getLastName())) {
//		    			cr = cb.like(from.get("veteranPerson").<String>get("lastName"), cb.parameter(String.class, "lastName"));
//		    		}
//		    		else {
//		    			cr = cb.equal(from.get("veteranPerson").<String>get("lastName"), cb.parameter(String.class, "lastName"));
//		    		}
//					predicates.add(cr);
//	    		}
//	    		
//				// middle name
//	    		if (criteria.getMiddleName() != null) {
//		    		Predicate cr = null;
//		    		if (containsWildcard(criteria.getMiddleName())) {
//		    			cr = cb.like(from.get("veteranPerson").<String>get("middleName"), cb.parameter(String.class, "middleName"));
//		    		}
//		    		else {
//		    			cr = cb.equal(from.get("veteranPerson").<String>get("middleName"), cb.parameter(String.class, "middleName"));
//		    		}
//					predicates.add(cr);
//	    		}
//	    		
//				// suffix
//	    		if (criteria.getSuffixName() != null) {
//		    		Predicate cr = null;
//		    		if (containsWildcard(criteria.getSuffixName())) {
//		    			cr = cb.like(from.get("veteranPerson").<String>get("suffixName"), cb.parameter(String.class, "suffixName"));
//		    		}
//		    		else {
//		    			cr = cb.equal(from.get("veteranPerson").<String>get("suffixName"), cb.parameter(String.class, "suffixName"));
//		    		}
//					predicates.add(cr);
//	    		}
//	    		
//				// DOB
//	    		if (criteria.getBirthDate() != null) {
//	    			Expression<Timestamp> exp1 = from.get("veteranPerson").get("birthDt");
//					Predicate fromDt = cb.greaterThanOrEqualTo( exp1, cb.parameter(Timestamp.class, "birthDtFrom"));
//					predicates.add(fromDt);
//					
//					Expression<Timestamp> exp2 = from.get("veteranPerson").get("birthDt");
//					Predicate toDt = cb.lessThan(exp2, cb.parameter(Timestamp.class, "birthDtTo"));
//					predicates.add(toDt);					
//	    		}
//	    		
//				// DOD
//	    		if (criteria.getDeathDate() != null) {
//	    			Expression<Timestamp> exp1 = from.get("veteranPerson").get("deathDt");
//					Predicate fromDt = cb.greaterThanOrEqualTo( exp1, cb.parameter(Timestamp.class, "deathDtFrom"));
//					predicates.add(fromDt);
//					
//					Expression<Timestamp> exp2 = from.get("veteranPerson").get("deathDt");
//					Predicate toDt = cb.lessThan(exp2, cb.parameter(Timestamp.class, "deathDtTo"));
//					predicates.add(toDt);
//	    		}
//	    		
//				// FNOD Created Date
//	    		if (criteria.getFnodCreatedDate() != null) {
//	    			Expression<Timestamp> exp1 = from.get("entryDt");
//					Predicate fromDt = cb.greaterThanOrEqualTo( exp1, cb.parameter(Timestamp.class, "fnodCreatedDtFrom"));
//					predicates.add(fromDt);
//					
//					Expression<Timestamp> exp2 = from.get("entryDt");
//					Predicate toDt = cb.lessThan(exp2, cb.parameter(Timestamp.class, "fnodCreatedDtTo"));
//					predicates.add(toDt);
//	    		}
//	    	}
//	    	
//	    	if (predicates.size() == 1) {
//	    		c.where(predicates.get(0));
//	    	}
//	    	else {
//	    		c.where(cb.and(predicates.toArray(new Predicate[0])));
//	    	}
//	    	
//	    	
//	    	TypedQuery<CaseData> q = em.createQuery(c);
//	    	
//	    	q.setParameter("lockedStatus", LockedStatus.COMPLETE);
//	    	
//	    	if (criteria.getSelectedUserName() != null) {
//	    		q.setParameter("username", criteria.getSelectedUserName());
//	    	}
//	    	if (criteria.getCaseId() != null) {
//	    		q.setParameter("caseId", criteria.getCaseId());
//	    		parametersSatisfied = true;
//	    	}
//	    	if (!parametersSatisfied && criteria.getSocialSecurityNumber() != null) {
//	    		q.setParameter("ssn", criteria.getSocialSecurityNumber());
//	    		parametersSatisfied = true;
//	    	}
//	    	if (!parametersSatisfied && criteria.getVeteranClaimNumber() != null) {
//	    		q.setParameter("veteranClaimNum", criteria.getVeteranClaimNumber());
//	    		parametersSatisfied = true;
//	    	}
//	    	if (!parametersSatisfied && criteria.getMilitaryServiceNumber() != null) {
//	    		q.setParameter("militaryServiceNum", criteria.getMilitaryServiceNumber());
//	    		parametersSatisfied = true;
//	    	}
//	    	if (!parametersSatisfied) {
//	    		if (criteria.getFirstName() != null) {
//					q.setParameter("firstName", criteria.getFirstName());
//				}
//				if (criteria.getLastName() != null) {
//					q.setParameter("lastName", criteria.getLastName());
//				}
//				if (criteria.getMiddleName() != null) {
//					q.setParameter("middleName", criteria.getMiddleName());
//				}
//				if (criteria.getSuffixName() != null) {
//					q.setParameter("suffixName", criteria.getSuffixName());
//				}
//				if (criteria.getBirthDate() != null) {
//	    			Timestamp startOfDay = new Timestamp(DateUtils.truncate(criteria.getBirthDate(), Calendar.DAY_OF_MONTH).getTime());
//	    			Calendar cal = Calendar.getInstance();
//	    			cal.setTime(startOfDay);
//	    			cal.add(Calendar.DAY_OF_MONTH, 1);
//	    			Timestamp endOfDay = new Timestamp(cal.getTimeInMillis());
//	    			
//					q.setParameter("birthDtFrom", startOfDay);
//					q.setParameter("birthDtTo", endOfDay);
//				}
//				if (criteria.getDeathDate() != null) {
//	    			Timestamp startOfDay = new Timestamp(DateUtils.truncate(criteria.getDeathDate(), Calendar.DAY_OF_MONTH).getTime());
//	    			Calendar cal = Calendar.getInstance();
//	    			cal.setTime(startOfDay);
//	    			cal.add(Calendar.DAY_OF_MONTH, 1);
//	    			Timestamp endOfDay = new Timestamp(cal.getTimeInMillis());
//	    			
//					q.setParameter("deathDtFrom", startOfDay);
//					q.setParameter("deathDtTo", endOfDay);
//				}
//				if (criteria.getFnodCreatedDate() != null) {
//	    			Timestamp startOfDay = new Timestamp(DateUtils.truncate(criteria.getFnodCreatedDate(), Calendar.DAY_OF_MONTH).getTime());
//	    			Calendar cal = Calendar.getInstance();
//	    			cal.setTime(startOfDay);
//	    			cal.add(Calendar.DAY_OF_MONTH, 1);
//	    			Timestamp endOfDay = new Timestamp(cal.getTimeInMillis());
//	    			
//					q.setParameter("fnodCreatedDtFrom", startOfDay);
//					q.setParameter("fnodCreatedDtTo", endOfDay);
//				}
//	    	}
//	    	
//	    	q.setMaxResults(SearchFNODRecordsSession.MAX_ROWS_TO_FETCH);
//	    	resultList = q.getResultList();
//    	}
//	    catch (Exception ex) {
//	    	log.error("Failure trying to query FNOD Records.", ex);
//	    	System.out.println(ex.getMessage());
//	    }
//    	
//    	return convert(resultList);
//	}
    
    
    private List<SearchFNODRecordsResult> convert(List<FnodRecord> fnodRecords) {
    	List<SearchFNODRecordsResult> results = null;
    	if (fnodRecords != null) {
    		results = new ArrayList<SearchFNODRecordsResult>(fnodRecords.size());
    		for (FnodRecord rec : fnodRecords) {
	    		SearchFNODRecordsResult result = new SearchFNODRecordsResult();
	    		result.setCaseId(rec.getCaseLink().getCaseId());
    			Date birthDt = null;
				Date deathDt = null;
	    		Timestamp fnodCreatedDt = null;
	    		
	    		Person veteranPerson = rec.getVeteranPerson();
	    		if (veteranPerson != null) {
	    			result.setFirstName(veteranPerson.getFirstName());
	    			result.setLastName(veteranPerson.getLastName());
	    			result.setMiddleName(veteranPerson.getMiddleName());
	    			result.setSuffix(veteranPerson.getSuffixName());
	    			result.setSocialSecurityNum(veteranPerson.getSocialSecurityNumber());
	    			birthDt = veteranPerson.getBirthDt();
	    			deathDt = veteranPerson.getDeathDt();
	    			fnodCreatedDt = rec.getFnodCreatedDt();
	    		}
				
				if (birthDt != null) {
					result.setDateOfBirth(new Timestamp(birthDt.getTime()));
				}
				if (deathDt != null) {
					result.setDateOfDeath(new Timestamp(deathDt.getTime()));
				}
				if (fnodCreatedDt != null) {
					result.setFnodCreatedDt(fnodCreatedDt);
				}
				
				result.setLockedStatus(rec.getCaseLink().getLockedStatus().toString());
				
				SourceCaseTypeMap srcSystemCastType = sourceCaseTypeMapSession.getSourceCaseTypeMap(rec.getCaseLink().getSourceSystem(), rec.getCaseLink().getCaseType());
				
				result.setCaseType(srcSystemCastType.getDescription());
				result.setMilitaryServiceNum(rec.getMilitaryServiceNum());
				result.setVeteranClaimNum(rec.getVeteranClaimNum());
				
				results.add(result);
	    	}
    	}
    	return results;
    }
    private List<SearchFNODRecordsResult> convert2(List<CaseData> fnodRecords) {
    	List<SearchFNODRecordsResult> results = null;
    	if (fnodRecords != null) {
    		results = new ArrayList<SearchFNODRecordsResult>(fnodRecords.size());
    		for (CaseData rec : fnodRecords) {
	    		SearchFNODRecordsResult result = new SearchFNODRecordsResult();
	    		result.setCaseId(rec.getCaseLink().getCaseId());
    			Date birthDt = null;
				Date deathDt = null;
	    		Timestamp fnodCreatedDt = null;
	    		
	    		Person veteranPerson = rec.getVeteranPerson();
	    		if (veteranPerson != null) {
	    			result.setFirstName(veteranPerson.getFirstName());
	    			result.setLastName(veteranPerson.getLastName());
	    			result.setMiddleName(veteranPerson.getMiddleName());
	    			result.setSuffix(veteranPerson.getSuffixName());
	    			result.setSocialSecurityNum(veteranPerson.getSocialSecurityNumber());
	    			birthDt = veteranPerson.getBirthDt();
	    			deathDt = veteranPerson.getDeathDt();
	    			fnodCreatedDt = rec.getEntryDt();
	    		}
				
				if (birthDt != null) {
					result.setDateOfBirth(new Timestamp(birthDt.getTime()));
				}
				if (deathDt != null) {
					result.setDateOfDeath(new Timestamp(deathDt.getTime()));
				}
				if (fnodCreatedDt != null) {
					result.setFnodCreatedDt(fnodCreatedDt);
				}
				
				result.setLockedStatus(rec.getCaseLink().getLockedStatus().toString());
				
				SourceCaseTypeMap srcSystemCastType = sourceCaseTypeMapSession.getSourceCaseTypeMap(rec.getCaseLink().getSourceSystem(), rec.getCaseLink().getCaseType());
				
				result.setCaseType(srcSystemCastType.getDescription());
				result.setMilitaryServiceNum(rec.getVeteran().getMilitaryServiceNum());
				result.setVeteranClaimNum(rec.getVeteran().getClaimNum());
				
				results.add(result);
	    	}
    	}
    	return results;
    }
    
    /**
     * Checks to see if the fields contains a wildcard
     * 
     * @param s
     * @return
     */
    private boolean containsWildcard(String s) {
    	boolean result = false;
    	if (s != null && s.contains("%")) {
    		result = true;
    	}
    	return result;
    }
}
