package gov.va.fnod.service;

import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.SampleCriteria;
import gov.va.fnod.model.SampleItem;
import gov.va.fnod.util.DatabaseUtil;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.text.SimpleDateFormat;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ejb.EJBException;
import javax.ejb.Stateless;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import oracle.jdbc.OracleTypes;


/**
 * Session Bean used for generating/getting sample data
 */
@Stateless
@TransactionManagement(TransactionManagementType.BEAN)
public class SampleSessionBean implements SampleSession {
	private static final Logger logr = Logger.getLogger(SampleSessionBean.class.getName());
	private static final String CALL_FUNC_PREFIX = String.format("{ call ? := %s.sample_pkg.", FNODModelConstants.FNOD_DATA_SCHEMA);
	private static final String CALL_PROC_PREFIX = String.format("{ call %s.sample_pkg.", FNODModelConstants.FNOD_DATA_SCHEMA);
	
	
	SimpleDateFormat dateFormat = new SimpleDateFormat();

	/**
	 * generate a case sample set
	 * 
	 * @return
	 */
	@Override
	public int generateCaseSample(String owner, SampleCriteria sampleCriteria) {
		int sampleCount = -1;

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;

			try {
				callableStatement = connection.prepareCall(CALL_FUNC_PREFIX + "generate_case_sample(?,?,?,?,?,?,?,?,?) }");
				callableStatement.registerOutParameter(1, OracleTypes.INTEGER);
				callableStatement.setString(2, owner);
				callableStatement.setTimestamp(3, sampleCriteria.getFromDate());
				callableStatement.setTimestamp(4, sampleCriteria.getToDate());

				Long longParam = sampleCriteria.getSourceSystemId();
				if (longParam == null) {
					callableStatement.setNull(5, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(5, longParam);
				}

				longParam = sampleCriteria.getCaseTypeId();
				if (longParam == null) {
					callableStatement.setNull(6, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(6, longParam);
				}

				longParam = sampleCriteria.getRegionId();
				if (longParam == null) {
					callableStatement.setNull(7, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(7, longParam);
				}

				longParam = sampleCriteria.getRegionalOfficeId();
				if (longParam == null) {
					callableStatement.setNull(8, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(8, longParam);
				}

				callableStatement.setInt(9, sampleCriteria.getMaxRows());
				callableStatement.setDouble(10, sampleCriteria.getSamplePercent());
				callableStatement.execute();
				sampleCount = callableStatement.getInt(1);
				
				connection.commit();
				
			} catch (Exception e) {				
				logr.log(Level.FINE, null, e);
				connection.rollback();
				throw new EJBException(e);
			} finally {				
				DatabaseUtil.releaseConnection(connection);
			}
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to generate case sample", e);
		}

		return sampleCount;
	}

	/**
	 * generate an fnod sample set
	 * 
	 * @return
	 */
	@Override
	public int generateFnodSample(String owner, SampleCriteria sampleCriteria) {
		int sampleCount = -1;

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;

			try {
				callableStatement = connection.prepareCall(CALL_FUNC_PREFIX + "generate_fnod_sample(?,?,?,?,?,?,?,?,?,?) }");
				callableStatement.registerOutParameter(1, OracleTypes.INTEGER);
				callableStatement.setString(2, owner);
				callableStatement.setTimestamp(3, sampleCriteria.getFromDate());
				callableStatement.setTimestamp(4, sampleCriteria.getToDate());

				Long longParam = sampleCriteria.getSourceSystemId();
				if (longParam == null) {
					callableStatement.setNull(5, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(5, longParam);
				}

				longParam = sampleCriteria.getCaseTypeId();
				if (longParam == null) {
					callableStatement.setNull(6, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(6, longParam);
				}

				longParam = sampleCriteria.getRegionId();
				if (longParam == null) {
					callableStatement.setNull(7, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(7, longParam);
				}

				longParam = sampleCriteria.getRegionalOfficeId();
				if (longParam == null) {
					callableStatement.setNull(8, OracleTypes.NUMBER);
				} else {
					callableStatement.setLong(8, longParam);
				}

				callableStatement.setInt(9, sampleCriteria.getMaxRows());
				callableStatement.setDouble(10, sampleCriteria.getSamplePercent());
				callableStatement.setString(11, sampleCriteria.getUsername());
				callableStatement.execute();
				sampleCount = callableStatement.getInt(1);
				
				connection.commit();
				
			} catch (Exception e) {
				logr.log(Level.FINE, null, e);
				connection.rollback();				
				throw new EJBException(e);
			} finally {
				DatabaseUtil.releaseConnection(connection);
			}
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to generate fnod record sample", e);
		}

		return sampleCount;
	}

	/**
	 * get the next case from the sample set
	 * 
	 * @return
	 */
	@Override
	public SampleItem getNext(String owner, SampleItem lastSampleItem) {
		
		SampleItem sampleItem = null;

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;
 
			try {
				callableStatement = connection.prepareCall(CALL_PROC_PREFIX + "get_next( ?, ?, ?) }");
				
				callableStatement.setString(1, owner);
				callableStatement.setLong(2, lastSampleItem.getSampleId());
				
				callableStatement.registerOutParameter(2, OracleTypes.NUMBER);
				callableStatement.registerOutParameter(3, OracleTypes.NUMBER);
								
				callableStatement.execute();
				
				sampleItem = new SampleItem();
				sampleItem.setSampleId(callableStatement.getLong(2));
				sampleItem.setCaseId(callableStatement.getLong(3));				
	 				
				connection.commit();
				
			} catch (Exception e) {
				logr.log(Level.FINE, null, e);
				connection.rollback();
				throw new EJBException(e);
			} finally {				
				DatabaseUtil.releaseConnection(connection);
			}
			
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to get next case sample", e);
		}

		return sampleItem;
	}

	/**
	 * clear the existing sample set
	 * 
	 * @return
	 */
	@Override
	public int clearSample(String owner) {
		int sampleCount = -1;

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;

			try {
				callableStatement = connection.prepareCall(CALL_FUNC_PREFIX + "clear_sample(?) }");
				callableStatement.registerOutParameter(1, OracleTypes.NUMBER);
				callableStatement.setString(2, owner);
				callableStatement.execute();
				sampleCount = callableStatement.getInt(1);
				
				connection.commit();
				
			} catch (Exception e) {
				logr.log(Level.FINE, null, e);
				connection.rollback();
				throw new EJBException(e);
			} finally {
				DatabaseUtil.releaseConnection(connection);
			}
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to clear sample", e);
		}

		return sampleCount;
	}

	/**
	 * add a case to the award case sample set
	 * 
	 * @return
	 */
	@Override
	public int addAwardCase(long caseId) {
		int sampleCount = -1;

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;

			try {
				callableStatement = connection.prepareCall(CALL_FUNC_PREFIX + "add_award_case(?) }");
				callableStatement.registerOutParameter(1, OracleTypes.NUMBER);
				callableStatement.setLong(2, caseId);
				callableStatement.execute();
				sampleCount = callableStatement.getInt(1);
				
				connection.commit();
				
			} catch (Exception e) {
				logr.log(Level.FINE, null, e);
				connection.rollback();
				throw new EJBException(e);
			} finally {				
				DatabaseUtil.releaseConnection(connection);
			}
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to add award sample", e);
		}

		return sampleCount;
	}

	/**
	 * get the next case from the award case sample set
	 * 
	 * @return
	 */
	@Override
	public SampleItem getNextAwardCase(String owner, SampleItem lastSampleItem) {
		
		SampleItem sampleItem = null;	

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;
 
			try {
				
				callableStatement = connection.prepareCall(CALL_PROC_PREFIX + "get_next_award_case(?, ?, ?) }");
				
				callableStatement.setString("P_USERNAME", owner);
				callableStatement.setLong("P_LAST_SAMPLE_ID", lastSampleItem.getSampleId());
				
				callableStatement.registerOutParameter("P_LAST_SAMPLE_ID", OracleTypes.NUMBER);
				callableStatement.registerOutParameter("P_CASE_ID", OracleTypes.NUMBER);				
				
				callableStatement.execute();
				
				sampleItem = new SampleItem();
				sampleItem.setSampleId(callableStatement.getLong("P_LAST_SAMPLE_ID"));
				sampleItem.setCaseId(callableStatement.getLong("P_CASE_ID"));
				
				connection.commit();
				
			} catch (Exception e) {
				logr.log(Level.FINE, null, e);
				connection.rollback();
				throw new EJBException(e);
			} finally {
				
				DatabaseUtil.releaseConnection(connection);
			}
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to get next award sample", e);
		}

		return sampleItem;
	}
	
	@Override
	public long cancelAwardAudit( SampleItem sampleItem ) {
		long count = 0;

		try {
			Connection connection = DatabaseUtil.getConnection();
			CallableStatement callableStatement = null;

			try { 
				callableStatement = connection.prepareCall(CALL_FUNC_PREFIX + "cancel_award_audit(?) }");
				callableStatement.registerOutParameter(1, OracleTypes.NUMBER);
				callableStatement.setLong(2, sampleItem.getSampleId());
				callableStatement.execute();
				
				count = callableStatement.getLong(1);
				
				connection.commit();
				
			} catch (Exception e) {
				logr.log(Level.FINE, "Failed to cancelAwardAudit" , e);
				connection.rollback();
				throw new EJBException(e);
			} finally {				
				DatabaseUtil.releaseConnection(connection);
			}
		} catch (Exception e) {
			logr.log(Level.FINE, null, e);
			throw new RuntimeException("Failed to cancel and clear award audit record", e);
		}

		return count;
	}
	
}
