package gov.va.fnod.service;

import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.fnoddata.Report;
import gov.va.fnod.util.DatabaseUtil;

import java.io.BufferedWriter;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.util.HashMap;
import java.util.List;

import javax.ejb.EJB;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.util.JRLoader;
import oracle.jdbc.OracleTypes;

import org.apache.log4j.Logger;

@Stateless
@TransactionAttribute(TransactionAttributeType.NOT_SUPPORTED)
public class ReportSessionBean implements ReportSession {

	@PersistenceContext(unitName = FNODModelConstants.DEFAULT_PERSISTENCE_UNIT)
	private EntityManager em;

	private static final Logger log = Logger.getLogger(ReportSessionBean.class);
	
	@EJB
	FnodRecordUserSession fnodRecordSession;

	/*
	 * The SuppressWarning is because JPA returns raw type of List<?> and the compiler warns with the casting to
	 * List<Report>, which is the correct mapping to List of Report objects.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<Report> getReportSelectList() {
		return (List<Report>) em.createNamedQuery(Report.GET_ALL_REPORTS).getResultList();
	}

	@Override
	public Report getReport(String reportName) {
		return (Report) em.find(Report.class, reportName);
	}

	@Override
	public List<String> getFnodRecordUsers() {
		return fnodRecordSession.getFnodRecordCompletedUserNames();
	}

	@Override
	public void runReport(String pathToReports, String reportName, HashMap<String, Object> parameterValues,
			OutputStream reportOut, OutputStream csvOut) {

		Connection connection = null;
		try {
			connection = DatabaseUtil.getConnection();
			log.info("Opening report: path: " + pathToReports + ", report name: " + reportName  );
			System.out.println("Opening report: path: " + pathToReports + ", report name: " + reportName  );
			getReport(connection, pathToReports, reportName, parameterValues, reportOut, csvOut);
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e.getMessage(), e.fillInStackTrace());
			throw new RuntimeException("A serious error occured duuring the Report creation.");
		} finally {
			if (connection != null) {
				try {
					connection.close();
				} catch (Exception e) {
					e.printStackTrace();
					log.error(e.getMessage(), e.fillInStackTrace());
				}
			}
		}
	}

	private void getReport(Connection connection, String pathToReports, String reportName,
			HashMap<String, Object> parameterValues, OutputStream reportOut, OutputStream csvOut) {
		PrintWriter writer = null;
		CallableStatement callableStatement = null;

		try {
			writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(csvOut)));
			connection.setAutoCommit(false);
			Report report = em.find(Report.class, reportName);
			int parameterCode = report.getParameterCode();
			StringBuilder functionBuilder = new StringBuilder(String.format("{ call ? := %s.report_pkg.%s(",
					FNODModelConstants.FNOD_DATA_SCHEMA, reportName));

			switch (parameterCode) {
			case 1:
				functionBuilder.append("?");
				break;
			case 2:
			case 3:
				functionBuilder.append("?,?");
				break;
			default:
				throw new Exception(parameterCode + " is not a valid parameter code");
			}

			functionBuilder.append(") }");
			callableStatement = connection.prepareCall(functionBuilder.toString());
			callableStatement.registerOutParameter(1, OracleTypes.CURSOR);

			switch (parameterCode) {
			case 1:
				callableStatement.setDate(2, new java.sql.Date(((java.util.Date) parameterValues.get("start_date"))
						.getTime()));
				break;
			case 2:
				callableStatement.setDate(2, new java.sql.Date(((java.util.Date) parameterValues.get("start_date"))
						.getTime()));
				callableStatement.setDate(3, new java.sql.Date(((java.util.Date) parameterValues.get("end_date"))
						.getTime()));
				break;
			case 3:
				callableStatement.setString(2, (String) parameterValues.get("username"));
				callableStatement.setDate(3, new java.sql.Date(((java.util.Date) parameterValues.get("start_date"))
						.getTime()));
				break;
			default:
				throw new Exception(parameterCode + " is not a valid parameter code");
			}

			callableStatement.execute();
			ResultSet resultSet = (ResultSet) callableStatement.getObject(1);
			ResultSetMetaData metaData = resultSet.getMetaData();
			int columnCount = metaData.getColumnCount();

			for (int i = 1; i <= columnCount; i++) {
				if (i > 1) {
					writer.print(",");
				}
				writer.print(formatField(metaData.getColumnName(i)));
			}

			while (resultSet.next()) {
				writer.println();
				for (int i = 1; i <= columnCount; i++) {
					if (i > 1) {
						writer.print(",");
					}
					writer.print(formatField(resultSet.getString(i)));
				}
			}

			parameterValues.put("SUBREPORT_DIR", pathToReports);
			JasperReport jasperReport = getJasperReport(pathToReports, report.getReportId());
			JasperPrint jasperPrint = null;
			jasperPrint = JasperFillManager.fillReport(jasperReport, parameterValues, connection);
			JasperExportManager.exportReportToPdfStream(jasperPrint, reportOut);
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e.getMessage(), e.fillInStackTrace());
			throw new RuntimeException("A serious error occured during the Report creation.");
		} finally {
			try {
				callableStatement.close();
				reportOut.flush();
				reportOut.close();
				writer.close();
				csvOut.close();
				connection.commit();
				connection.close();
			} catch (Exception e) {
				e.printStackTrace();
				log.error(e.getMessage(), e.fillInStackTrace());
			}
		}
	}

	private String formatField(String fldValue) {
		StringBuilder field = new StringBuilder();
		boolean quoteField = false;
		field.setLength(0);

		if (fldValue != null && fldValue.length() > 0) {
			char[] fldChar = fldValue.toCharArray();

			if (fldChar[0] == ' ' || fldChar[0] == '\t' || fldChar[fldChar.length - 1] == ' '
					|| fldChar[fldChar.length - 1] == '\t') {
				quoteField = true;
			}
			for (char ch : fldChar) {
				if (ch == '\n' || ch == '\r' || ch == ',' || ch == '"') {
					quoteField = true;
				}
				if (ch == '"') {
					field.append(ch).append(ch);
				} else {
					field.append(ch);
				}
			}
		}
		if (quoteField) {
			field.insert(0, '"').append('"');
		}
		return field.toString();
	}

	// Returns a Jasper report template from the path to reports and report Id parameters.
	private JasperReport getJasperReport(String pathToReports, String reportId) {
		JasperReport jasperReport = null;
		try {
			jasperReport = (JasperReport) JRLoader.loadObjectFromFile(pathToReports + reportId + ".jasper");
		} catch (JRException e) {
			log.error(e.getMessage(), e.fillInStackTrace());
			throw new RuntimeException("A serious error occured duuring the Report creation.");
		}
		return jasperReport;
	}
}