package gov.va.fnod.service;

import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.SearchCriteria;
import gov.va.fnod.model.SourceCaseType;
import gov.va.fnod.model.fnoddata.ActivityType;
import gov.va.fnod.model.fnoddata.ActivityTypeCode;
import gov.va.fnod.model.fnoddata.AuditActivity;
import gov.va.fnod.model.fnoddata.CaseData;
import gov.va.fnod.model.fnoddata.CaseFlagApp;
import gov.va.fnod.model.fnoddata.CaseLink;
import gov.va.fnod.model.fnoddata.FnodRecord;
import gov.va.fnod.model.fnoddata.LockedStatus;
import gov.va.fnod.model.fnoddata.Person;
import gov.va.fnod.model.fnoddata.RegionalOffice;
import gov.va.fnod.model.fnoddata.SourceCaseTypeMap;
import gov.va.fnod.model.fnoddata.Veteran;
import gov.va.fnod.service.exception.StatefulContextRuntimeException;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.ejb.EJB;
import javax.ejb.Stateful;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.PersistenceContextType;
import javax.persistence.Query;
import javax.persistence.TypedQuery;

import org.apache.log4j.Logger;

/**
 * Session bean to handle the Case
 */

/*
 * Overriding the default transaction type from REQUIRED to NOT_SUPPORTED.  This will have
 * the effect of causing methods to execute inside their own transactions unless
 * TransactionAttributeType is overridden on the method (This parallels a @Stateless bean). 
 */
@Stateful(name = "CaseSessionBean")
@TransactionAttribute(TransactionAttributeType.NOT_SUPPORTED)
public class CaseSessionBean implements CaseSession {

	private static final Logger log = Logger.getLogger(CaseSessionBean.class);
	                                                  
	@PersistenceContext(unitName = FNODModelConstants.DEFAULT_PERSISTENCE_UNIT, 
			            type = PersistenceContextType.EXTENDED)  // EXTENDED cause context to be maintained 
	private EntityManager em;
	
	private CaseLink caseLink;
	
	@EJB
	SuffixNameSession suffixNameSession;
	
	@EJB
	FnodRecordSession fnodRecordSession;

	public CaseLink getCaseLink() {
		return caseLink;
	}

	/**
	 * get the CaseLink given a Case Id
	 * 
	 * @param caseId
	 * @return
	 */
	@Override
	public CaseLink openCaseLinkById(long caseId) {
		
		// Default: Transaction not supported - also not needed
		
		caseLink = null;

		try {
			caseLink = em.find(CaseLink.class, caseId);
		} catch (Exception e) {
			log.error("openCase",e);			
		}

		return caseLink;
	}

	@TransactionAttribute(TransactionAttributeType.REQUIRES_NEW)
	public void cancel() {
		em.clear();
		caseLink = null;
	}
	
	
	/**
	 *  This method is called from the SOA interface and is responsible for saving the
	 *  initial CaseLink and (CaseData or CaseInsuranceData or CaseFlagAppData) to the
	 *  database.
	 *  
	 *  Each call to this method results in a single database transaction which is either
	 *  committed or rolled back by the container.
	 */
	@Override
	@TransactionAttribute(TransactionAttributeType.REQUIRES_NEW)
	public long addPendingCase(CaseLink caseLink) {
		long fnodCaseId = 0;
		em.persist(caseLink);

		fnodCaseId = caseLink.getCaseId();
		return fnodCaseId;
	}

	/**
	 * This method call a Stored Procedure from the database, which updates the locked status
	 * of a CASE_LINK record from PENDING to PARKED
	 */
	@Override
	@TransactionAttribute(TransactionAttributeType.REQUIRES_NEW)
	public Long searchNextCase(String username, SearchCriteria searchCriteria) {
		Long result = null;
		CaseLink cl = null;
		Long nextCaseLinkID = null;

		SourceCaseTypeMap sourceCaseTypeMap = em.find(SourceCaseTypeMap.class, searchCriteria.getSourceCaseTypeMapId());
		
		// Clear last case that was being worked on
		caseLink = null;
	
		try {
			// Prepare criteria for getting next case
			Date dateRangeFrom = searchCriteria.getDateRangeFrom();
			Date dateRangeTo = searchCriteria.getDateRangeTo();
			
			// set default dates if none provided
			if (dateRangeFrom == null) {
				dateRangeFrom = new SimpleDateFormat("MM/dd/yyyy")
						.parse("07/04/1776");
			}
			if (dateRangeTo == null) {
				dateRangeTo = new SimpleDateFormat("MM/dd/yyyy")
						.parse("12/31/2099");
			}

			// make the stored procedure call to lock next case for processing
			// this call will begin a new transaction.
			
			if (searchCriteria.getSourceCaseTypeMapId() == SourceCaseType.FLAG_APPLICATION
					.getSourceCaseTypeId()) {
		
				Long regionId = null;
				if (searchCriteria.getRegionId() != null && searchCriteria.getRegionId() != 0) {
					regionId = searchCriteria.getRegionId();
				}
				
				Long regOffId = null;
				if (searchCriteria.getRegionalOfficeId() != null && searchCriteria.getRegionalOfficeId() != 0) {
					regOffId = searchCriteria.getRegionalOfficeId();
					if ( regionId == null ) {
						RegionalOffice ro = em.find(RegionalOffice.class, regOffId);
						if ( ro != null ) {
							regionId = ro.getRegion().getRegionId();
						}
					}
				}
										
				if (regionId == null && regOffId == null) {
					nextCaseLinkID = getAndLockNextCase(
							searchCriteria.getSourceCaseTypeMapId(), dateRangeFrom,
							dateRangeTo);
				}
				else {
					// Lookup the next flag application for processing
					nextCaseLinkID = getAndLockNextCase(		
							searchCriteria.getSourceCaseTypeMapId(), dateRangeFrom,
							dateRangeTo, regionId, regOffId);
				}
				
			} else {
				
				// Lookup any other case type for processing 
				nextCaseLinkID = getAndLockNextCase(
						searchCriteria.getSourceCaseTypeMapId(), dateRangeFrom,
						dateRangeTo);
				
			}

			
			// If a case is returned, created a new FNOD Record to be saved
			// with the now PARKED case.
			if (nextCaseLinkID != null) {
				log.debug("Locked Next CaseLink: ID=" + nextCaseLinkID);
				
				// Pull the case link record into context
				cl = em.find(CaseLink.class, nextCaseLinkID);
				cl.setLockedStatus(LockedStatus.PARKED);
				
				// Add the FNOD record with defaulted values
				FnodRecord fnodRecord = em.find(FnodRecord.class, nextCaseLinkID.longValue());
				if(fnodRecord == null) {
					fnodRecord = new FnodRecord();
				}
				
				fnodRecord.setUsername(username);
				
				// set Eligibility Verification
				fnodRecord.setEligibilityVerification(sourceCaseTypeMap.getDefaultEligibilityVerification());
		
				mapCaseDataToFnod(cl,fnodRecord);
				cl.setFnodRecord(fnodRecord);
				
				// And save back to the database
				cl = em.merge(cl);
				result = cl.getCaseId();
				
			} else {
				log.warn("No CaseLink could be locked per request.");
			}
		} catch (Exception e) {
			log.error("CaseSessionBean.searchNextCase() encountered an error!", e);
			throw new StatefulContextRuntimeException(e);
		}
		
		// Once we return, the context has been cleared, making the returned record
		// a detached record (side-effect of calling merge).
		// The returned record should be opened by Id to pull the record back into
		// context.  This will have to be done after this call has completed since the
		// transaction is managed by the container.
		return result;
	}
	
	
	/* 
	 * (non-Javadoc)
	 * @see gov.va.fnod.service.CaseSession#addFnodRecordForPendingCase(java.lang.Long, java.lang.String)
	 * 
	 * this method is new, and as of 2/24/2015, this method is used on the Search FNOD Record page. 
	 * The way the system works is that it displays data from the FNOD_Record table, the only way to 
	 * get that data into the table is via the process case, for them to be able to view the data otherwise 
	 * we had to add the record 
	 */
	
	public boolean addFnodRecordForPendingCase(Long caseId, String username) {
		boolean ok = true; 
		
		try { 
			
			boolean caseDoesNotExist = true;
			// Clear last case that was being worked on
			caseLink = null;
			

			// Pull the case link record into context
			caseLink = em.find(CaseLink.class, caseId);
			

			// Add the FNOD record with defaulted values
			FnodRecord fnodRecord = em.find(FnodRecord.class,caseId);
			if(fnodRecord != null) {
				caseDoesNotExist = false;
			}else { 
				fnodRecord = new FnodRecord();
			}
			
			fnodRecord.setUsername(username);

			// set Eligibility Verification
			fnodRecord.setEligibilityVerification("N");

			mapCaseDataToFnod(caseLink,fnodRecord);
			caseLink.setFnodRecord(fnodRecord);
			
			// Create record if not exist
			if( caseDoesNotExist ) {  
				ok = fnodRecordSession.createFnodRecord(fnodRecord);
				System.out.println("CaseSessionBean.addFnodRecordForPendingCase: something went wrong saving FNOD record");
			}

		} catch (Exception e) { 
			System.out.println(e.getMessage());
			ok = false;
		}
		return ok;
		
	}
	
	private String toUpperCase(String value) {
		if ( value != null ) {
			return value.toUpperCase();
		} else {
			return null;
		}
	}
	
	private void mapCaseDataToFnod(CaseLink result, FnodRecord fnodRecord) {
		
		fnodRecord.setCaseLockedDt(new Timestamp(System.currentTimeMillis()));
		
		Person p = fnodRecord.getVeteranPerson();
		if (p == null) {
			p = new Person();
			fnodRecord.setVeteranPerson(p);
		}

		if ( result.getCaseData() != null ) {
			CaseData caseData = result.getCaseData();
			Person veteranPerson = caseData.getVeteranPerson();
			if (veteranPerson != null) {
				p.setFirstName(toUpperCase(veteranPerson.getFirstName()));
				p.setLastName(toUpperCase(veteranPerson.getLastName()));
				p.setMiddleName(toUpperCase(veteranPerson.getMiddleName()));
				
				String suffixName = toUpperCase(veteranPerson.getSuffixName());
				if ( suffixNameSession.isValidSuffix(suffixName) ) {
					p.setSuffixName(suffixName);
				} else {
					p.setSuffixName(null);
				}
				p.setBirthDt(veteranPerson.getBirthDt());
				p.setDeathDt(veteranPerson.getDeathDt());
				p.setSocialSecurityNumber(veteranPerson.getSocialSecurityNumber());
			}

			Veteran veteran = caseData.getVeteran();
			if (veteran != null) {
				fnodRecord.setVeteranClaimNum(veteran.getClaimNum());
				fnodRecord.setMilitaryServiceNum(veteran.getMilitaryServiceNum());
			}
			
		} else if ( result.getCaseFlagApp() != null ) {
			CaseFlagApp caseFlagAppData = result.getCaseFlagApp();
			if ( caseFlagAppData.getRegionalOfficeId() > 0 ) {
				RegionalOffice ro = em.find(RegionalOffice.class, caseFlagAppData.getRegionalOfficeId());
				fnodRecord.setRegionalOffice(ro);
			}
		} 
			
	}


	/**
	 * Retrieves and locks the next available case as an ID.
	 * 
	 * @param sourceCaseTypeMapId
	 * @param dateRangeFrom
	 * @param dateRangeTo
	 * @return
	 */
	@TransactionAttribute(value = TransactionAttributeType.MANDATORY)
	private Long getAndLockNextCase(long sourceCaseTypeMapId,
			java.util.Date dateRangeFrom, java.util.Date dateRangeTo) {
		
		Long nextCaseId;
		
		// This query actually begins a transaction
		Query query = em.createNamedQuery(CaseLink.GET_NEXT_CASE);		
		SourceCaseTypeMap sourceCaseTypeMap = em.find(SourceCaseTypeMap.class, sourceCaseTypeMapId);
		
		query.setParameter("p_system_id", sourceCaseTypeMap.getSourceSystem().getSourceSystemId());
		query.setParameter("p_case_type_id", sourceCaseTypeMap.getCaseType().getCaseTypeId());
		query.setParameter("p_load_from", dateRangeFrom);
		query.setParameter("p_load_to", dateRangeTo);
		System.out.println("getAndLockNextCase: " + query.toString());
		Integer singleResult = (Integer) query.getSingleResult();
		if (singleResult == null) {
			return null;
		}
		nextCaseId = singleResult.longValue();
		return nextCaseId;
	}

	/**
	 * Retrieves and locks the next available case as an ID.
	 * 
	 * @param sourceCaseTypeMapId
	 * @param dateRangeFrom
	 * @param dateRangeTo
	 * @param regionId
	 * @param regionalOfficeId
	 * @return
	 */
	@TransactionAttribute(value = TransactionAttributeType.MANDATORY)
	private Long getAndLockNextCase(long sourceCaseTypeMapId,
			java.util.Date dateRangeFrom, java.util.Date dateRangeTo,
			Long regionId, Long regionalOfficeId) {
		Long nextCaseId = null;
		Query query = em.createNamedQuery(CaseLink.GET_NEXT_FLAG_APP_CASE);
		SourceCaseTypeMap sourceCaseTypeMap = em.find(SourceCaseTypeMap.class, sourceCaseTypeMapId);
		
		query.setParameter("p_system_id", sourceCaseTypeMap.getSourceSystem().getSourceSystemId());
		query.setParameter("p_case_type_id", sourceCaseTypeMap.getCaseType().getCaseTypeId());
		query.setParameter("p_load_from", dateRangeFrom);
		query.setParameter("p_load_to", dateRangeTo);
		query.setParameter("p_region_id", regionId);
		query.setParameter("p_regional_office_id", regionalOfficeId);
		Integer singleResult = (Integer) query.getSingleResult();
		if (singleResult != null) {
			nextCaseId = singleResult.longValue();
		}
		return nextCaseId;
	}

	@Override
	@TransactionAttribute(TransactionAttributeType.REQUIRES_NEW)
	public void saveCaseLink(boolean awardAuditRequired, String username, ActivityTypeCode activityType ) {
		
		em.merge(caseLink);
		
		if ( awardAuditRequired ) {
			Query qry = em.createNamedQuery(AuditActivity.ADD_AWARD_CASE).setParameter("p_case_id", caseLink.getCaseId());
			qry.getSingleResult();
		}
		
		if ( activityType != null && activityType.isAuditActivity() ) {
			if ( ActivityTypeCode.AUDITED_AWARD.equals(activityType)) {
				deleteSample(caseLink.getCaseId(),"~~AWARD~~");	
			} else {
				deleteSample(caseLink.getCaseId(), username);
			}
		}
		
		//caseLink = null;
	}
			
	@TransactionAttribute(TransactionAttributeType.MANDATORY)
	private int deleteSample(Long caseId, String username ) {
		int sampleCount = -1;

		try {
			Query query = em.createNamedQuery(AuditActivity.DELETE_SAMPLE);
			query.setParameter("p_case_id", caseId);
			query.setParameter("p_username", username);
			sampleCount = (Integer) query.getSingleResult();
		}

		catch (Exception e) {
			e.printStackTrace();
		}

		return sampleCount;
	}
	
	public ActivityType getActivityType(ActivityTypeCode activityTypeCode) {
		TypedQuery<ActivityType> qry = em.createNamedQuery(ActivityType.GET_ACTIVITY_TYPE_BY_ACTIVITY_CODE,ActivityType.class);
		qry.setParameter("activityCd", activityTypeCode);
		return qry.getSingleResult();		
	}
	
	public void setCaseLinkNull() { 
		caseLink = null;
	}

	public void setCaseLink(CaseLink caseLink) {
		this.caseLink = caseLink;
	}
}
