package gov.va.fnod.model.fnoddata;

import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.util.OracleTimestampTimeZoneConverter;

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;
import javax.persistence.Transient;

import org.eclipse.persistence.annotations.NamedStoredFunctionQueries;
import org.eclipse.persistence.annotations.NamedStoredFunctionQuery;
import org.eclipse.persistence.annotations.StoredProcedureParameter;

/**
 * The persistent class for the AUDIT_ACTIVITY database table.
 * 
 */
@Entity
@Table(name = "AUDIT_ACTIVITY")
@NamedQueries({
	@NamedQuery(name=AuditActivity.GET_AUDIT_ACTIVITY_BY_CASE_ID, 
				query="select a from AuditActivity a " + 
					  "where a.caseLink.caseId = :caseId")
})
@NamedStoredFunctionQueries({
	@NamedStoredFunctionQuery(   
		name=AuditActivity.DELETE_SAMPLE,   
	    functionName=FNODModelConstants.FNOD_DATA_SCHEMA+".sample_pkg.delete_sample",   
	    returnParameter=@StoredProcedureParameter(queryParameter="returnVal", name="returnVal", type=Integer.class),   
	    parameters={   
	        @StoredProcedureParameter(queryParameter="p_case_id", name="p_case_id", type=Long.class),   
	        @StoredProcedureParameter(queryParameter="p_username", name="p_username", type=String.class)
	    }   
	),
	@NamedStoredFunctionQuery(   
		name=AuditActivity.ADD_AWARD_CASE,   
	    functionName=FNODModelConstants.FNOD_DATA_SCHEMA+".sample_pkg.add_award_case",   
	    returnParameter=@StoredProcedureParameter(queryParameter="returnVal", name="returnVal", type=Integer.class),   
	    parameters={   
	        @StoredProcedureParameter(queryParameter="p_case_id", name="p_case_id", type=Long.class),   
	    }   
	) 
})             
public class AuditActivity implements Serializable {
	private static final long serialVersionUID = 1L;
	
	public static final String GET_AUDIT_ACTIVITY_BY_CASE_ID = "GET_AUDIT_ACTIVITY_BY_CASE_ID";
	public static final String DELETE_SAMPLE = "DELETE_SAMPLE";
	public static final String ADD_AWARD_CASE = "ADD_AWARD_CASE";

	@Id
	@SequenceGenerator(name = "AUDIT_ACTIVITY_TRANSACTIONID_GENERATOR", sequenceName = "AUDIT_ACTIVITY_SEQ", allocationSize = 1)
	@GeneratedValue(strategy = GenerationType.SEQUENCE, generator = "AUDIT_ACTIVITY_TRANSACTIONID_GENERATOR")
	@Column(name = "TRANSACTION_ID", unique = true, nullable = false, precision = 9)
	private long transactionId;

	//bi-directional many-to-one association to CaseLink
	@ManyToOne
	@JoinColumn(name="CASE_ID", nullable=false)
	private CaseLink caseLink;

	@ManyToOne
	@JoinColumn(name = "ACTIVITY_TYPE_ID", nullable = false)
	private ActivityType activityType;

	@Column(name = "ACTIVITY_DT", nullable = false)
	private Timestamp activityDt;

	@Column(nullable = false, length = 32)
	private String username;

	@OneToMany(fetch = FetchType.EAGER, cascade = {CascadeType.PERSIST,CascadeType.MERGE}, mappedBy = "auditActivity")
	private List<AuditChange> auditChanges;
	
	// this field is for display purposes only, with oracle timestamp you get GMT-0500 for EST time, and customer 
	// wants EST, CST not the GMT-#### format.
	@Transient
	private String activityDate ;
	
	public AuditActivity() {
	}
	
	public AuditActivity(CaseLink caseLink, ActivityType activityType, Timestamp activityDt, String username, List<AuditChange> auditChanges) {
		this.caseLink = caseLink;
		this.activityType = activityType;
		this.activityDt = activityDt;
		this.username = username;
		this.auditChanges = auditChanges;
		this.activityDate = OracleTimestampTimeZoneConverter.formatDateWithTimeZone(new Date(activityDt.getTime()));
	}

	public long getTransactionId() {
		return this.transactionId;
	}

	public void setTransactionId(long transactionId) {
		this.transactionId = transactionId;
	}

	public CaseLink getCaseLink() {
		return this.caseLink;
	}

	public void setCaseLink(CaseLink caseLink) {
		this.caseLink = caseLink;
	}

	public ActivityType getActivityType() {
		return activityType;
	}

	public void setActivityType(ActivityType activityType) {
		this.activityType = activityType;
	}
	
	public String getActivityDescription() {
		return activityType.getDescription();
	}
	
	public Timestamp getActivityDt() {
		return this.activityDt;
	}

	public void setActivityDt(Timestamp activityDt) {
		this.activityDt = activityDt;
		this.activityDate = OracleTimestampTimeZoneConverter.formatDateWithTimeZone(new Date(activityDt.getTime()));
	}

	public String getUsername() {
		return this.username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public List<AuditChange> getAuditChanges() {
		return this.auditChanges;
	}

	public void setAuditChanges(List<AuditChange> auditChanges) {
		if ( auditChanges != null ) {
			for(AuditChange auditChange: auditChanges) {
				auditChange.setAuditActivity(this);
			}
		}
		this.auditChanges = auditChanges;
	}
	
	public void addAuditChange(AuditChange auditChange) {
		auditChange.setAuditActivity(this);
		auditChanges.add(auditChange);
	}
	
	public boolean getHasFieldChanges() {
		return auditChanges != null && auditChanges.size() > 0;
	}

	public String getActivityDate() {
		return activityDate;
	}

	public void setActivityDate(String activityDate) {
		this.activityDate = activityDate;
	}
}