package gov.va.fnod.model;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

public class PasswordValidator implements Validator {

	private static final Logger log = Logger.getLogger(PasswordValidator.class);

	private PasswordHashable passwordHasher;

	@Override
	public void setPasswordHasher(PasswordHashable passwordHasher) {
		this.passwordHasher = passwordHasher;
	}

	@Override
	public Map<ErrorCode, String> validatePassword(String password,
			PasswordData passwordData) {
		Map<ErrorCode, String> errors = new HashMap<ErrorCode, String>();
		boolean isPasswordValid = false;
		isPasswordValid = validatePasswordChars(password);// check for valid
															// characters
		if (isPasswordValid) {
			isPasswordValid = passwordHasher.comparePasswords(
					passwordData.getPassword(), password);// check for password
															// match
			if (!isPasswordValid) {
				log.error(ErrorCode.PASSWORD_NOT_MATCH.getErrorMsg());
				errors.put(ErrorCode.PASSWORD_NOT_MATCH,
						ErrorCode.PASSWORD_NOT_MATCH.getErrorMsg());
			}
		} else {
			log.error(ErrorCode.INVALID_CHARS.getErrorMsg());
			errors.put(ErrorCode.INVALID_CHARS,
					ErrorCode.INVALID_CHARS.getErrorMsg());
		}
		if (passwordData.getExpiryDate() != null) {
			isPasswordValid = checkPasswordExpiry(passwordData.getExpiryDate());// check
																				// if
																				// the
																				// password
																				// is
																				// expired
			if (!isPasswordValid) {
				int numWarningDays = checkPasswordWarning(
						passwordData.getExpiryDate(),
						passwordData.getPasswordWarningDays());
				if (numWarningDays > 0) {
					log.error(ErrorCode.PASSWORD_EXPIRY_WARNING.getErrorMsg()
							+ numWarningDays + " days ");
					errors.put(ErrorCode.PASSWORD_EXPIRY_WARNING,
							ErrorCode.PASSWORD_EXPIRY_WARNING.getErrorMsg()
									+ numWarningDays + " days ");
				}
			} else {
				log.error(ErrorCode.PASSWORD_EXPIRED.getErrorMsg());
				errors.put(ErrorCode.PASSWORD_EXPIRED,
						ErrorCode.PASSWORD_EXPIRED.getErrorMsg());
			}
		}
		return errors;
	}

	@Override
	public Map<ErrorCode, String> validateChangePassword(String newPassword,
			PasswordData passwordData) {
		Map<ErrorCode, String> errors = new HashMap<ErrorCode, String>();
		boolean passwordCheck = false;
		if (DateUtil.getDateDiffDays(new Date(),
				passwordData.getPasswordChangedDate()) >= 1) {
			if (validateLength(newPassword,
					passwordData.getPasswordMinLength(),
					passwordData.getPasswordMaxLength())) { // validate the
															// length
				passwordCheck = validatePasswordChars(newPassword);// check for
																	// valid
																	// characters
				if (passwordCheck) {
					List<String> passwordHistoryList = new ArrayList<String>();
					passwordHistoryList.add(passwordData.getPassword());
					if (passwordData.getPasswordHistory() != null
							&& !passwordData.getPasswordHistory().trim()
									.isEmpty()) {
						StringTokenizer st = new StringTokenizer(
								passwordData.getPasswordHistory(), "|");
						while (st.hasMoreElements()) {
							passwordHistoryList.add((String) st.nextElement());
						}
					}
					passwordCheck = validatePasswordHistory(newPassword,
							passwordHistoryList);// check if the password exists
													// in history
					if (passwordCheck) {
						log.error(ErrorCode.INVALID_PASSWORD
								.getErrorMsg());
						errors.put(ErrorCode.INVALID_PASSWORD,
								ErrorCode.INVALID_PASSWORD.getErrorMsg());
					}
				} else {
					log.error(ErrorCode.INVALID_PASSWORD.getErrorMsg());
					errors.put(ErrorCode.INVALID_PASSWORD,
							ErrorCode.INVALID_PASSWORD.getErrorMsg());
				}
			} else {
				log.error(ErrorCode.INVALID_PASSWORD.getErrorMsg());
				errors.put(ErrorCode.INVALID_PASSWORD,
						ErrorCode.INVALID_PASSWORD.getErrorMsg());
			}
		} else {
			log.error(ErrorCode.PASSWORD_CHANGE_PER_DAY.getErrorMsg());
			errors.put(ErrorCode.PASSWORD_CHANGE_PER_DAY,
					ErrorCode.PASSWORD_CHANGE_PER_DAY.getErrorMsg());
		}
		return errors;
	}

	/**
	 * check if the password is expired
	 * 
	 * @param expiryDate
	 * @param passwordWarningDays
	 * @return
	 */
	private int checkPasswordWarning(Date expiryDate, int passwordWarningDays) {
		int numWarningDays = 0;
		long dateDiff = DateUtil.getDateDiffDays(expiryDate, new Date());
		if (dateDiff <= passwordWarningDays) {
			numWarningDays = (int) dateDiff;
		}
		return numWarningDays;
	}

	/**
	 * check if the password is expired
	 * 
	 * @param expiryDate
	 * @return
	 */
	private boolean checkPasswordExpiry(Date expiryDate) {
		if (new Date().compareTo(expiryDate) > 0) {
			return true;
		}
		return false;
	}

	/**
	 * validates if the password exists in the password history
	 * 
	 * @param newPassword
	 * @param passwordHistoryList
	 * @return
	 */
	private boolean validatePasswordHistory(String newPassword,
			List<String> passwordHistoryList) {
		boolean isPasswordExists = false;
		for (String oldPasswordHash : passwordHistoryList) {
			if (passwordHasher.comparePasswords(oldPasswordHash, newPassword)) {
				isPasswordExists = true;
				break;
			}
		}
		return isPasswordExists;
	}

	/**
	 * validates the password with regular expression ( (?=.*\\d) - must contain
	 * one digit from 0-9 (?=.*[a-z]) - must contain one lowercase character
	 * (?=.*[A-Z]) - must contain one uppercase character (?=.*\\W) - must
	 * contain one special character from the list (a non-word character) . -
	 * match anything with previous condition checking {0, 40} - be 40
	 * characters long ) 
	 * 
	 * @param password
	 * 
	 * @return
	 */
	public boolean validatePasswordChars(String password) {
		if (password == null) {
			return false;
		}
		Pattern pattern = Pattern
				.compile("((?=.*\\d)(?=.*[a-z])(?=.*[A-Z])(?=.*\\W).{0,40})");
		Matcher matcher = pattern.matcher(password);
		return matcher.matches();
	}

	/**
	 * 
	 * @param username
	 * @return
	 */
	@Override
	public Map<ErrorCode, String> validateCreateUser(String username,
			String password, PasswordParameters passwordParams) {
		if (username == null || username.isEmpty() || password == null
				|| password.isEmpty()) {
			throw new IllegalArgumentException(
					"username and password must not be null or empty");
		}
		Map<ErrorCode, String> errors = new HashMap<ErrorCode, String>();
		boolean isUsernameValid = false;
		Pattern pattern = Pattern.compile("^[a-zA-Z0-9]*$");
		Matcher matcher = null;
		if (!validateLength(username, passwordParams.getUsernameMinLength(),
				passwordParams.getUsernameMaxLength())) {
			log.error(ErrorCode.INVALID_USERNAME.getErrorMsg());
			errors.put(ErrorCode.INVALID_USERNAME,
					ErrorCode.INVALID_USERNAME.getErrorMsg());
		} else {
			matcher = pattern.matcher(username);
			isUsernameValid = matcher.matches();
			if (!isUsernameValid) {
				log.error(ErrorCode.INVALID_USERNAME.getErrorMsg());
				errors.put(ErrorCode.INVALID_USERNAME,
						ErrorCode.INVALID_USERNAME.getErrorMsg());
			}
		}
		if (!validateLength(password, passwordParams.getPasswordMinLength(),
				passwordParams.getUsernameMaxLength())) {
			log.error(ErrorCode.INVALID_PASSWORD.getErrorMsg());
			errors.put(ErrorCode.INVALID_PASSWORD,
					ErrorCode.INVALID_PASSWORD.getErrorMsg());
		} else {
			if (!validatePasswordChars(password)) {
				log.error(ErrorCode.INVALID_PASSWORD.getErrorMsg());
				errors.put(ErrorCode.INVALID_PASSWORD,
						ErrorCode.INVALID_PASSWORD.getErrorMsg());
			}
		}
		return errors;
	}

	/**
	 * validate the length constraints
	 * 
	 * @param vaidateString
	 * @param minLength
	 * @param maxLength
	 * @return
	 */
	private boolean validateLength(String vaidateString, int minLength,
			int maxLength) {
		if (vaidateString.trim().length() < minLength
				|| vaidateString.trim().length() > maxLength) {
			return false;
		}
		return true;
	}

	/**
	 * validates reset password
	 * 
	 * @param password
	 * @param passwordParams
	 * @return
	 */
	@Override
	public Map<ErrorCode, String> validateResetPassword(String password,
			PasswordParameters passwordParams) {
		Map<ErrorCode, String> errors = new HashMap<ErrorCode, String>();
		if (password == null || password.isEmpty()) {
			throw new IllegalArgumentException(
					"password must not be null or empty");
		}
		if (!validateLength(password, passwordParams.getPasswordMinLength(),
				passwordParams.getUsernameMaxLength())) {
			log.error(ErrorCode.INVALID_PASSWORD.getErrorMsg());
			errors.put(ErrorCode.INVALID_PASSWORD,
					ErrorCode.INVALID_PASSWORD.getErrorMsg());
		} else if (!validatePasswordChars(password)) {
			log.error(ErrorCode.INVALID_PASSWORD.getErrorMsg());
			errors.put(ErrorCode.INVALID_PASSWORD,
					ErrorCode.INVALID_PASSWORD.getErrorMsg());
		}
		return errors;
	}
}
