package gov.va.fnod.model;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;

/**
 * contains the password hashing functionality 
 *
 */
public class PasswordHasher implements PasswordHashable {

	private static final int ITERATION_NUM = 3;

	/**
	 * compare the password hash to the password entered by the user
	 * 
	 * @param hashedPasswordHex
	 * @param password
	 * @return
	 */
	@Override
	public boolean comparePasswords(String hashedPasswordHex, String password) {
		boolean isPasswordMatch = false;
		byte[] seed = extractSeed(hashedPasswordHex);
		byte[] passwordHash = extractPasswordHash(hashedPasswordHex);
		byte[] newPasswordHash = hashPassword(password, seed);
		if (MessageDigest.isEqual(passwordHash, newPasswordHash)) {
			isPasswordMatch = true;
		}
		return isPasswordMatch;
	}

	/**
	 * generate the seed using the SecureRandom by calling nextBytes method to
	 * generate Random Bytes
	 * 
	 * @return
	 */
	@Override
	public byte[] generateSeed() {
		SecureRandom random;
		byte[] seed = null;
		try {
			random = SecureRandom.getInstance("SHA1PRNG");
			seed = new byte[4];
			random.nextBytes(seed);
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException("NoSuchAlgorithmException ");
		}
		return seed;
	}

	/**
	 * extract seed from the hashed password
	 * 
	 * @param hashedPassword
	 * @return
	 */
	@Override
	public byte[] extractSeed(String hashedPasswordHex) {
		byte[] hashedPassword = hexStringToByteArray(hashedPasswordHex);
		byte[] extractedSeed = Arrays.copyOfRange(hashedPassword, 32, 36);
		return extractedSeed;
	}

	/**
	 * extract the password from the hashed password
	 */
	@Override
	public byte[] extractPasswordHash(String hashedPasswordHex) {
		byte[] hashedPassword = hexStringToByteArray(hashedPasswordHex);
		byte[] extractedPwdHash = Arrays.copyOfRange(hashedPassword, 0, 32);
		return extractedPwdHash;
	}

	/**
	 * hash the password - generate the seed and call hashpassword() passing the
	 * seed
	 * 
	 * @param password
	 * @param seed
	 * @return
	 */
	@Override
	public String hashPassword(String password) {
		byte[] seed = generateSeed();
		return byteArrayToHexString(concatenate(hashPassword(password, seed),
				seed));
	}

	/**
	 * 
	 * @param password
	 * @param seed
	 * @return
	 */
	@Override
	public byte[] hashPassword(String password, byte[] seed) {
		MessageDigest md;
		byte[] input = null;

		try {
			md = MessageDigest.getInstance("SHA-256");
			md.reset();
			md.update(seed);
			input = md.digest(password.getBytes("UTF-8"));
			for (int i = 0; i < ITERATION_NUM; i++) {
				md.reset();
				input = md.digest(input);
			}
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException("NoSuchAlgorithmException ");
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("UnsupportedEncodingException ");
		}
		return input;
	}

	/**
	 * combine the saltedPasswordHash with the seed
	 * 
	 * @param saltedPasswordHash
	 * @param salt
	 * @return
	 */
	private byte[] concatenate(byte[] saltedPasswordHash, byte[] seed) {
		byte[] resultArray = new byte[saltedPasswordHash.length + seed.length];
		System.arraycopy(saltedPasswordHash, 0, resultArray, 0,
				saltedPasswordHash.length);
		System.arraycopy(seed, 0, resultArray, saltedPasswordHash.length,
				seed.length);
		return resultArray;
	}

	/**
	 * convert from byte array to hex string
	 * 
	 * @param password
	 * @return
	 */
	private String byteArrayToHexString(byte[] password) {
		if (password == null) {
			throw new IllegalArgumentException("password must not be null");
		}
		return new String(Hex.encodeHex(password));
	}

	/**
	 * convert from hex string to byte array
	 * 
	 * @param s
	 * @return
	 */
	private byte[] hexStringToByteArray(String hexString) {
		if (hexString == null || hexString.isEmpty()) {
			throw new IllegalArgumentException(
					"hexString must not be null or empty");
		}
		byte[] originalBytes = null;
		try {
			originalBytes = Hex.decodeHex(hexString.toCharArray());
		} catch (DecoderException e) {
			throw new RuntimeException("DecoderException ");
		}
		return originalBytes;
	}

}