package gov.va.cem.docstorage.service;



import java.io.File;

import java.util.UUID;



import gov.va.cem.common.fileutils.FileHelper;

import gov.va.cem.docstorage.exceptions.ZeroByteFileException;

import gov.va.cem.docstorage.model.DocCatalog;

import gov.va.cem.docstorage.model.StorageSegment;

import gov.va.cem.docstorage.model.SystemAccount;

import gov.va.cem.docstorage.model.UnitName;



import javax.ejb.EJB;

import javax.ejb.Stateless;

import javax.persistence.EntityManager;

import javax.persistence.PersistenceContext;



/**

 * Session Bean implementation class DocumentStorageSessionBean

 */



@EJB(name="documentStorage", beanInterface=DocumentStorageSession.class)

@Stateless

public class DocumentStorageSessionBean  implements DocumentStorageSession {

	

	

	@PersistenceContext(unitName=UnitName.DOC_STORAGE)

	EntityManager em;

	

	@EJB 

	private SystemAccountSession systemAccountSession;

	

	@EJB

	private StorageSegmentSession storageSegmentSession;

	

	private FileHelper fileHelper = new FileHelper();

	

	private static final String DIGEST_SHA1 = "SHA1";

	

    /**

     * Default constructor. 

     */

    public DocumentStorageSessionBean() {

    }



    /**

     * This method is responsible for adding document into the document storage

     * system.

     * 

     *  @param systemName - name of system that owns the document

     *  @param filename - name to be stored for the file.

     *  @param filePath - location of existing source file

     *  @return the document id

     */

	@Override

	public long addDocument(String systemName, String filename, String filePath) {

		

		DocCatalog docCatalog = new DocCatalog();

		String cacheFilePath;

		

		// Verify the adding system

		docCatalog.setSystemAccount(lookupSystemAccount(systemName));

		

		// The file name is broken down into the "name" and the "extension".  The extension

		// will be added the disk image of the document to make it easier on users to view

		// files outside of the document storage system.

		docCatalog.setFilename(filename);

		docCatalog.setFileExt(parseFileExt(filename));

			

		// This will determine the segment and copy the file there

		// updating the document catalog record with its temporary location

		// and name (a UUID value).

		copySrcFileToSegmentCache(filePath, docCatalog);

		

		// Location is currently set to cache since documentId is unassigned

		saveDocCatalog(docCatalog);

		

		// Get the current path so we can delete the file later

		cacheFilePath = docCatalog.getDocumentPath();

		

		// Now that we have a documentId, copy the file to the correct directory, the file is in the 

		// database, and could be referenced from cache location, so we must copy, not move.

		copyDocFromCacheToStorage(docCatalog);

		

		// Save the new location

		saveDocCatalog(docCatalog);

		

		// Remove the entry from the segment cache

		removeItemFromCache(docCatalog.getStorageSegment(), cacheFilePath);

		

		// Finally, return the document Id to the calling routine.

		return docCatalog.getDocCatalogId();

	}



	/**

	 * The file extension is last segment of a file name delimited by a period following

	 * the first sequence of periods at the beginning of the file name. 

	 * @param filename

	 * @return

	 */

	private String parseFileExt(String filename) {

		int beginIndex=0;

		int lastIndex;

		String retval="";

		

		// Locate the last leading period - this eliminates relative path periods

		while( filename.charAt(beginIndex) == '.' ) {

			beginIndex++;

		}

		

		lastIndex = filename.lastIndexOf('.');

		

		// now if the last index is greater than the first index, we have an extension

		if ( beginIndex < lastIndex ) {

			retval = filename.substring(lastIndex + 1) ;

		}

		

		return retval;

	}

	

	/**

	 * The file is copied from the source system into cache.

	 * 

	 * @param srcPath

	 * @param docCatalog

	 */

	private void copySrcFileToSegmentCache(String srcPath,

			DocCatalog docCatalog) {	

		

		// TODO this implementation does not handle a full segment



		// Assign to Storage segment

		docCatalog.setStorageSegment(storageSegmentSession.getNextAvailableSegement());

		

		// Build temporary filename and path

		StringBuilder destPath = new StringBuilder();

		destPath.append("cache").append(File.separator);

		destPath.append(UUID.randomUUID().toString());

		if (docCatalog.getFileExt() != null ) {

			destPath.append('.');

			destPath.append(docCatalog.getFileExt());

		}

		docCatalog.setDocumentPath(destPath.toString());

		

		// Copy file, computing SHA1 digest while the file is being copied.

		File srcFile = new File(srcPath);

		File destFile = getDocFile(docCatalog);

		

		if ( srcFile.length() > 0 ) {

			String digest = fileHelper.copyFile(srcFile, destFile, DIGEST_SHA1);

		

			// Save the digest to the document record, this will be our baseline to detect change

			docCatalog.setDigestSha1(digest);

		} else {

			throw new ZeroByteFileException("Cannot load zero byte file to document storage.");

		}

		

	}

	

	private File getDocFile(DocCatalog docCatalog) {

		File dir = new File(docCatalog.getStorageSegment().getSystemPath());

		return new File( dir, docCatalog.getDocumentPath());

	}



	/**

	 * Saves the document to the database, assigning the document id.

	 * @param docCatalog

	 */

	private void saveDocCatalog(DocCatalog docCatalog) {

		if ( docCatalog.getDocCatalogId() == 0 ) {

			em.persist(docCatalog);

		} else {

			em.merge(docCatalog);

		}

	}



	/**

	 * 

	 * @param docCatalog

	 */

	private void copyDocFromCacheToStorage(DocCatalog docCatalog) {

	      

        File srcFile = getDocFile(docCatalog);

        

        String docId = String.format("%09d", docCatalog.getDocCatalogId() );    

        StringBuilder docPath = new StringBuilder();

        

        docPath.append(docId.substring(0,3)).append(File.separator);

        docPath.append(docId.substring(3,6)).append(File.separator);            

        docPath.append(docId);

        

        if ( docCatalog.getFileExt() != null  && ! docCatalog.getFileExt().isEmpty() ) {

              docPath.append('.').append(docCatalog.getFileExt());              

        }

        docCatalog.setDocumentPath(docPath.toString());

        

        String digest = fileHelper.copyFile(srcFile, getDocFile(docCatalog), DIGEST_SHA1);

        if ( !digest.equals(docCatalog.getDigestSha1()) ) {

              throw new RuntimeException("Checksum failure - document not added to storage.");

        }

  }





	/**

	 * 

	 * @param storageSegment

	 * @param cacheFilePath

	 */

	private void removeItemFromCache(StorageSegment storageSegment, String cacheFilePath) {

		

		// TODO consider file lock. 

		File removeFile = new File(new File(storageSegment.getSystemPath()),cacheFilePath);

		if ( removeFile.isFile() ) {

			removeFile.delete();

		}

		

	}



	/**

	 * 

	 * @param systemName

	 * @return

	 */

	private SystemAccount lookupSystemAccount(String systemName) {

		return systemAccountSession.getSystemAccount(systemName);

	}



	/**

	 * 	

	 */

	@Override

	public void removeDocument(String systemName, long documentId) {

		

		DocCatalog docCatalog = em.find(DocCatalog.class, documentId);

		

		if ( ! docCatalog.getSystemAccount().getAccountName().equals(systemName) ) {

			throw new RuntimeException("Document not owned by "+systemName);

		}

		

		// Remove it from the database

		em.remove(docCatalog);

		

		// Then the file system 

		getDocFile(docCatalog).delete();

	}



	/**

	 * 

	 */

	@Override

	public String getDocument(String systemName, long documentId, String destPath) {

		

		DocCatalog docCatalog = em.find(DocCatalog.class, documentId);

		File srcFile = getDocFile(docCatalog);

		File destFile = new File(destPath);

		

		String digest = fileHelper.copyFile(srcFile, destFile,DIGEST_SHA1);

		if ( ! digest.equals(docCatalog.getDigestSha1()) ) { 

			destFile.delete();

			throw new RuntimeException("Failed to create target file");

		}

		

		return docCatalog.getFilename();

	}

	



}

