package gov.va.cem.docstorage.service;

/**
 * Provides basic methods for saving and retrieving documents in a centralized repository.
 *
 */
public interface DocumentStorageSession {
	
	/**
	 * This method is used to add a document into the document repository.  A registered 
	 * system must be provided along with the document name to be associated with the document 
	 * and the current fully qualified file system path.  After successfully adding the document 
	 * to the repository, a document ID will be returned allowing later access to the document.
	 * 
	 * The method will throw a ... exception if the source cannot be found/read.  Additionally,
	 * a ... exception will be thrown if the document cannot be added to the repository.
	 * 
	 * @param systemName - name/token representing a registered system.
	 * @param documentName - name of document to be stored
	 * @param srcPath - current, fully qualified file system path to document
	 * @return - documentId assigned to document for later retrieval.
	 * 
	 */
	public long addDocument(String systemName, String documentName, String srcPath );
	
	/**
	 * Given a documentID, the requested document will be copied to the specified
	 * destination path, and the filename stored with the document catalog will be returned. The
	 * system is used to enforce system access.  An ... exception will be thrown if the system is 
	 * not know.  A ... exception will be thrown if the requested document does not exist.  A ... exception
	 * document will be thrown if the document can not be written to the destination.  Finally, a ... exception 
	 * will be thrown if the target file fails to match the digest check.
	 * 
	 * @param systemName - name/token representing a registered system.
	 * @param documentId - documentId returned when document was added.
	 * @param destPath - fully qualified path, including filename and extension where the file should be placed.
	 * @return - the document name of the document retrieved.
	 */
	public String getDocument(String systemName, long documentId, String destPath);
	
	
	/**
	 * This routine will delete a document from the document storage system.  Documents can
	 * only be removed by the system that added the document.
	 * 
	 * @param systemName - name/token representing a registered system.
	 * @param documentId 
	 */
	public void removeDocument(String systemName, long documentId);

}
