package gov.va.fnod.util;

import java.io.File;

import java.util.LinkedList;
import java.util.List;

public class FileHelper {


    /**
     * Deletes the specified directory, and all child files/directories.  This
     * implementation does not use recursion; this is a trade of resulting in
     * better processing memory utilization for slightly more process expense.
     *
     * The parameter dir is required, if a null value is passed a
     * NullPointerException will be throw.  Additionally, the input File object
     * expects the value to represent a directory. If the referenced File
     * exists and is not a directory, and InvalidArgumentException will be
     * throw.  If the directory does not exist, the delete will effectively
     * be null processed.
     *
     * @param dir File representing directory to be deleted.
     */
    public static void deleteDir(File dir) {

        if (dir == null) {
            throw new NullPointerException("non-null value required.");
        }

        if (dir.exists()) {
            if (!dir.isDirectory()) {
                throw new IllegalArgumentException("expected a directory");
            }
        } else {
            // There is nothing to do.
            return;
        }

        // Push input dir onto stack
        List<File> dirStack = new LinkedList<File>();
        dirStack.add(dir);

        while (!dirStack.isEmpty()) {

            // Get the current directory to be worked on from the top of the stack
            File curDir = dirStack.get(0);

            // Process all files and directories in the current directory
            File[] files = curDir.listFiles();
            for (File file : files) {
                // Deleting files as they are encountered
                if (file.isFile()) {
                    // the file.exists is to guard against failures caused by
                    // files being deleted by other processes between list and now
                    if (!file.delete() && file.exists()) {
                        throw new FileDeleteException(file.getAbsolutePath());
                    }
                } else {
                    // and stack up the directories for later processing
                    dirStack.add(0, file);
                }
            }

            // Determine if we've reached a terminal directory
            if (curDir == dirStack.get(0)) {
                dirStack.remove(0);
                // the file.exists is to guard against failures caused by
                // files being deleted by other processes between list and now
                if (!curDir.delete() && curDir.exists()) {
                    throw new FileDeleteException(curDir.getAbsolutePath());
                }
            }
        }
    }

    private FileHelper() {
        super();
    }
}
