package gov.va.fnod.bl;

import gov.va.fnod.be.ReportConfigEntity;
import gov.va.fnod.be.SourceReportType;
import gov.va.fnod.soa_common.model.ServiceConnectionParams;
import gov.va.fnod.soa_common.model.fnod.CaseTypeCd;
import gov.va.fnod.soa_common.model.fnod.FnodCase;
import gov.va.fnod.soa_common.model.fnod.FnodPayload;
import gov.va.fnod.soa_common.model.fnod.SourceSystemCd;
import gov.va.fnod.soa_common.service.FnodClientServiceBean;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * This class is main process controller for the FNOD Data Load process The
 * general process flow is as follows:
 * <ol>
 * 		<li>For each Report config
 * 			<ol>
 * 				<li>Find the reports in the inputDir matching the filePattern from the Report config</li>
 * 				<li>For each Filename
 * 					<ol>
 * 						<li>Rename the report to add .splitting.txt extension and move to the working directory</li>
 * 						<li>Open the Report for processing</li>
 * 						<li>Split Report into records and place them in working directory</li>
 * 						<li>Find records to parse from the working directory</li>
 * 						<li>For each record
 * 							<li>Parse the record and delete it</li>
 * 						</li> 						
 * 					</ol>
 * 				</li>
 * 			</ol>
 * 			<li>On successful loading, rename the file to add .complete.txt extension and move it to the completed directory </li>
 * 		</li> 		
 * </ol>
 */
public class ProcessManager extends ProcessControlBase {
	
	
	/**
	 * Constructs the ProcessManager passing it the data required for processing FNOD Reports
	 * @param reportConfigs configuration parameters
	 * @param connectionParams connection parameters
	 */
	public ProcessManager(
			Map<SourceReportType, ReportConfigEntity> reportConfigs,
			ServiceConnectionParams connectionParams) {
		super(reportConfigs, connectionParams);
	}

	/**
	 * Returns the List of filenames from the input directory matching the file
	 * pattern from the configuration provided
	 * @param reportConfig
	 * @return
	 */
	@Override
	protected List<String> findReports(ReportConfigEntity reportConfig) {
		log.debug("In findReports");
		List<String> filenames = new ArrayList<String>();
		File inputDir = new File(reportConfig.getInputDirPath());
		if (!inputDir.exists()) {
			throw new IllegalArgumentException("inputDir does not exists: " + inputDir.getAbsolutePath());
		}
		if (!inputDir.isDirectory()) {
			throw new IllegalArgumentException("inputDir is not a directory: " + inputDir.getAbsolutePath());
		}
		File[] files = inputDir.listFiles();
		for (File file : files) {
			if (file.getName().matches(reportConfig.getFilenamePattern())) {
				filenames.add(file.getAbsolutePath());
			}
		}
		log.debug("Number of Reports found: " + filenames.size());
		return filenames;
	}

	/**
	 * opens the Report for processing
	 * @param filepath
	 * @return
	 */
	@Override
	protected Reader openReportRecord(String filepath) {
		log.debug("In openReportRecord, opening file: " + filepath);
		FileInputStream fin;
		DataInputStream in;
		BufferedReader br;
		try {
			fin = new FileInputStream(filepath);
			in = new DataInputStream(fin);
			br = new BufferedReader(new InputStreamReader(in));
		} catch (FileNotFoundException e) {
			throw new RuntimeException("FileNotFoundException " + filepath);
		}
		return br;
	}
	
	/**
	 * Splits the Report based on the SourceReportType and places the records in
	 * the current working directory
	 * 
	 * @param reader
	 */
	@Override
	protected void splitReport(Reader reader) {
		log.debug("In splitReport");
		RecordSplittable recordSplitter = RecordSplitterFactory
				.getInstance(currentSourceReportType);
		recordSplitter.splitRecords(reader, this);
	}
	
	/**
	 * Splits the Report based on the SourceReportType and places the records in
	 * the current working directory
	 * 
	 * @param reader
	 */
	@Override
	protected void splitBossSpouseReport(Reader reader) {
		log.debug("In splitBossSpouseReport");
		RecordSplittable recordSplitter = RecordSplitterFactory
				.getInstance(currentSourceReportType);
		recordSplitter.splitBossSpouseRecords(reader, this);
	}
	
	/**
	 * find the records for parsing from the current working directory
	 * 
	 * @return
	 */
	@Override
	protected List<String> findRecords() {
		log.debug("In findRecords");
		List<String> recordnames = new ArrayList<String>();
		File currWorkDir = new File(currentWorkingDir);
		if (!currWorkDir.exists()) {
			throw new IllegalArgumentException("currWorkDir does not exists" + currWorkDir.getPath());
		}
		if (!currWorkDir.isDirectory()) {
			throw new IllegalArgumentException("currWorkDir is not a directory" + currWorkDir.getPath());
		}
		File[] records = currWorkDir.listFiles();
		for (File record : records) {
			if (!record.getName().matches(
					".*" + FNODConstants.FILE_EXT_SPLITTING + ".*")) {
				recordnames.add(record.getAbsolutePath());
			}
		}
		return recordnames;
	}
	
	/**
	 * parses the record base on the current SourceReportType
	 * @param reader
	 */
	@Override
	protected void parseReportRecord(Reader reader) {
		log.debug("parseReportRecord");
		RecordParsable recordParser = RecordParserFactory
				.getInstance(currentSourceReportType);
		FnodCase fnodCase = recordParser.parseRecord(reader);
		if(fnodCase != null) {
			FnodPayload fnodPayload = new FnodPayload();
			fnodCase.setCaseTypeCd(CaseTypeCd.valueOf(currentSourceReportType.getReportType()));
			fnodCase.setSourceSystemCd(SourceSystemCd.valueOf(currentSourceReportType.getSourceSystem()));
			fnodPayload.setFnodCase(fnodCase);
			saveRecord(fnodPayload, getConnectionParams());
		}		
	}
	
	/**
	 * writes the record to the disk - current working directory
	 */
	@Override
	public void writeRecord(String record, String recordNum) {
		log.debug("In writeRecord");
		BufferedWriter writer = null;
		try {
			File outputFile = new File(currentWorkingDir, currentReportFilename
					+ recordNum + FNODConstants.FILE_EXT_TXT);
			writer = new BufferedWriter(new FileWriter(outputFile));
			writer.append(record);
			close(writer);// close writer
		} catch (IOException e) {
			close(writer);
			throw new RuntimeException("Error writing the Record");
		} finally {
			close(writer);// close writer
		}
	}

	@Override
	public long saveRecord(FnodPayload fnodPayload, ServiceConnectionParams serviceConnectionParams) {
		long caseId = 0L;
		
		try { 
		log.debug("In saveRecord");
		this.numRecordsProcessed++;
		FnodClientServiceBean fnodClientService = new FnodClientServiceBean();
		caseId = fnodClientService.submit(fnodPayload, serviceConnectionParams);
		this.caseIdLst.add(caseId);
		System.out.println("Saved Case, new case id: " + caseId);
		log.debug("Saved Case, new case id: " + caseId);
		}catch (Exception e) { 
			log.error("Failed to save case: " + fnodPayload.getFnodCase().getFnod().getVeteran().getPerson().getName().getFirstName() + " " + fnodPayload.getFnodCase().getFnod().getVeteran().getPerson().getName().getLastName());
		}
		return caseId;
	}

	/**
	 * release resources
	 * 
	 * @param resource
	 */
	private void close(Closeable resource) {
		if (resource != null) {
			try {
				resource.close();
			} catch (IOException e) {
				throw new RuntimeException("Error releasing resources");
			}
		}
	}

	public int getNumRecordsProcessed() {
		return numRecordsProcessed;
	}

	public void setNumRecordsProcessed(int numRecordsProcessed) {
		this.numRecordsProcessed = numRecordsProcessed;
	}

	public List<Long> getCaseIdLst() {
		return caseIdLst;
	}

	public void setCaseIdLst(List<Long> caseIdLst) {
		this.caseIdLst = caseIdLst;
	}
}