package gov.va.fnod.bl;

import gov.va.fnod.be.ReportConfigEntity;
import gov.va.fnod.be.SourceReportType;
import gov.va.fnod.soa_common.model.ServiceConnectionParams;
import gov.va.fnod.util.DataLoadConstants;

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * This class is base class process controller for the FNOD Data Load process The
 * general process flow is as follows:
 * <ol>
 * 		<li>For each Report config
 * 			<ol>
 * 				<li>Find the reports in the inputDir matching the filePattern from the Report config</li>
 * 				<li>For each Filename
 * 					<ol>
 * 						<li>Rename the report to add .splitting.txt extension and move to the working directory</li>
 * 						<li>Open the Report for processing</li>
 * 						<li>Split Report into records and place them in working directory</li>
 * 						<li>Find records to parse from the working directory</li>
 * 						<li>For each record
 * 							<li>Parse the record and delete it</li>
 * 						</li> 						
 * 					</ol>
 * 				</li>
 * 			</ol>
 * 			<li>On successful loading, rename the file to add .complete.txt extension and move it to the completed directory </li>
 * 		</li> 		
 * </ol>
 */
public abstract class ProcessControlBase implements SplitterCallbackInterface,
		RecordParserReader, Runnable {
	
	/**
	 * Constructs the ProcessManager passing it the data required for processing FNOD Reports
	 * @param reportConfigs Report configuration parameters
	 * @param connectionParams connection parameters
	 */
	public ProcessControlBase(
			Map<SourceReportType, ReportConfigEntity> reportConfigs,
			ServiceConnectionParams connectionParams) {
		
		if (reportConfigs == null) {
            throw new NullPointerException("reportConfigs cannot be null");
        }

        if (connectionParams == null) {
            throw new NullPointerException("connectionParams cannot be null");
        }

        this.reportConfigs = reportConfigs;
		this.setConnectionParams(connectionParams);
	}
	
	public ProcessControlBase() {

	}
	protected int numRecordsProcessed = 0;
	protected List<Long> caseIdLst = new ArrayList<Long>();
	
	protected static Logger log = Logger.getLogger(ProcessManager.class);
	
	private Map<SourceReportType, ReportConfigEntity> reportConfigs;
	private ServiceConnectionParams connectionParams;

	protected SourceReportType currentSourceReportType;
	protected String currentWorkingDir;
	protected String currentCompleteDir;
	protected String currentReportFilename;
	protected String currentReportFilenameSplit;
	protected String currentFilenamePattern;
	protected Reader reader;
	
	/**
	 * Returns the List of filenames from the input directory matching the file
	 * pattern from the configuration provided
	 * 
	 * @param reportConfig
	 * @return
	 */
	protected abstract List<String> findReports(ReportConfigEntity reportConfig);
	
	/**
	 * opens the Report for processing
	 * @param filepath
	 * @return
	 */
	protected abstract Reader openReportRecord(String filepath);
	
	/**
	 * Splits the Report based on the SourceReportType and places the records in
	 * the current working directory
	 * 
	 * @param reader
	 */
	protected abstract void splitReport(Reader reader);
	
	/**
	 * Splits the Boss Spouse Report based on the SourceReportType and places the records in
	 * the current working directory
	 * 
	 * @param reader
	 */
	protected abstract void splitBossSpouseReport(Reader reader);
	
	/**
	 * find the records for parsing from the current working directory
	 * 
	 * @return
	 */
	protected abstract List<String> findRecords();
	
	/**
	 * parses the record 
	 * @param reader
	 */
	protected abstract void parseReportRecord(Reader reader);
	
	/**
     * This is main processing routine and controls the sequence of
     * process calls.
     */
	@Override
	public void run() {
		log.info("Begin Report Processing...");
	
		List<String> filenames;
		List<String> recordnames;
		File reportFile;
		File renameReportFileTo;
		
		for (ReportConfigEntity config : reportConfigs.values()) {	
			currentSourceReportType = config.getSourceReportType();
			currentWorkingDir = config.getWorkingDirPath();
			currentCompleteDir = config.getCompletedDirPath();
			currentFilenamePattern = config.getFilenamePattern();
			log.debug("Report Type: " + currentSourceReportType);
			filenames = findReports(config);
			if (filenames != null && !filenames.isEmpty()) {
				for (String filename : filenames) {
					try {
						log.debug("Processing File: " + filename);
						// System.out.println("=========================================");
						
						reportFile = new File(filename);
						currentReportFilename = reportFile.getName();
						renameReportFileTo = new File(currentWorkingDir,
								reportFile.getName()
										+ FNODConstants.FILE_EXT_SPLITTING);
						if(!reportFile.renameTo(renameReportFileTo)) {
							throw new RuntimeException("Rename Failed " + currentReportFilename);
						}
						currentReportFilenameSplit = renameReportFileTo.getAbsolutePath();
						reader = openReportRecord(renameReportFileTo.getAbsolutePath());
						log.debug("Report Type: " + currentSourceReportType.toString());
						if(currentSourceReportType.toString().equals(DataLoadConstants.BOSS_SPOUSE_RPT_TYPE)) { 
							log.debug("Splitting using method splitBossSpouseReport, report type = " + currentSourceReportType);
							splitBossSpouseReport(reader);
						}else { 
							log.debug("Splitting using method splitReport, report type = " + currentSourceReportType);
							splitReport(reader);
						}
						recordnames = findRecords();
						for (String record : recordnames) {
							parseReportRecord(openReportRecord(record));
							File rec = new File(record);
							if(rec.exists()) {
								rec.delete();
							}
						}
						if(!renameReportFileTo
								.renameTo(new File(
										currentCompleteDir,
										renameReportFileTo.getName()
												+ FNODConstants.FILE_EXT_SPLITTING_COMPLETE))) {
							throw new RuntimeException("Rename Failed " + currentReportFilenameSplit);
						}
					} catch (Exception e) {
						cleanup();
						log.error("Failed to load the report " + currentReportFilename, e);
					} 
				}
			} 
			log.debug(createRunSummary(this.numRecordsProcessed, this.caseIdLst, currentSourceReportType + ""));
			this.numRecordsProcessed = 0;
			this.caseIdLst = new ArrayList<Long>();
		}
		
		
	}
	
	/**
	 * delete all the files in the working directory and the rename the report
	 * to add .error.txt and move to the current working directory
	 */
	private void cleanup() {
		log.debug("In cleanup");
		try {
			if (reader != null) {
				reader.close();
			}
		} catch (IOException e) {
			throw new RuntimeException("Failed to close the reader");
		}	 
		File workDir = new File(currentWorkingDir);
		if (workDir.exists() && workDir.isDirectory()) {
			File[] files = workDir.listFiles();
			for (File f : files) {
				if (f.getAbsolutePath().equals(currentReportFilenameSplit)) {
					if(!f.renameTo(new File(currentWorkingDir, f.getName()
							+ FNODConstants.FILE_EXT_ERROR))) {
						throw new RuntimeException("Rename Failed " + currentReportFilenameSplit);
					}
				} else {
					if(!f.getAbsolutePath().contains(FNODConstants.FILE_EXT_ERROR)) {
						f.delete();
					}
				}
			}
		}
	}
	
	private String createRunSummary(int numRecs, List<Long> caseIdsLst, String rptType) { 
		StringBuffer sb = new StringBuffer();
		
		sb.append("\n-----------------------------------------\n");
		sb.append("----------- Report Summary --------------\n");
		sb.append("-----------------------------------------\n");
		sb.append("Run Date: " + new Date().toString() + "\n");
		sb.append("Report Type: " + rptType + "\n");
		sb.append("Number of Records Processed: " + numRecs + "\n");
		sb.append("Number of Case Ids: " + caseIdsLst.size() + "\n");
		sb.append("Case Ids: " + join(caseIdsLst) +  "\n");
		sb.append("--------- End Report Summary ------------\n");
		
		return sb.toString();
	
	}
	
	private String join(List lst) { 
		StringBuffer sb = new StringBuffer();
		for(int i = 0; i < lst.size(); i++) {
			if(i != (lst.size() -1)) { 
				sb.append(lst.get(i) + ",");
			}else { 
				sb.append(lst.get(i) );
			}
		}
		
		return sb.toString();		
	}
	

	public Map<SourceReportType, ReportConfigEntity> getReportConfigs() {
		return reportConfigs;
	}

	public void setReportConfigs(
			Map<SourceReportType, ReportConfigEntity> reportConfigs) {
		this.reportConfigs = reportConfigs;
	}

	public ServiceConnectionParams getConnectionParams() {
		return connectionParams;
	}

	public void setConnectionParams(ServiceConnectionParams connectionParams) {
		this.connectionParams = connectionParams;
	}

}