package gov.va.cem.common.pdfutils;

import com.lowagie.text.Image;
import com.lowagie.text.Rectangle;

public abstract class ImageSizeCalcBase implements PdfImagePageFormatter {

    private Image image;
    private PageMargin pageMargins;
    private Rectangle pageSize;

    public ImageSizeCalcBase() {
    }

    /**
     * Initialize the image, compute page size and margins and adjust image
     */
    public final void prepareImage(Image image) {
        this.image = image;
        pageSize = calculatePageSize();
        pageMargins = calculatePageMargins();
        adjustImage();
    }

    /**
     *  getter for Page Size
     *  Page Size calculated in prepareImage
     * @return Rectangle
     */
    public final Rectangle getPageSize() {
        return this.pageSize;
    }

    /**
     * getter for Page Margins
     * @return PageMargins
     */
    public final PageMargin getPageMargins() {
        return this.pageMargins;
    }

    protected final void rotateImage(Rotate rotate) {
        image.setRotation((float)rotate.getRadianAngle());    
    }
    
    /**
     * Does page size calculation.
     * This function ,ay requirre ovveride
     * @return default LETTER size
     */
    protected abstract Rectangle calculatePageSize();


    /**
     * Does page margins calculation.
     * @return default margins: 9 point for all borders
     */
    protected abstract PageMargin calculatePageMargins();

    /**
     * place holders for child, to do adjustments to the image, if needed
     * these w
     */
    protected abstract void adjustImage();

    /**
     * Helper function.
     * Scale and rotates image based on calculated size and margins
     */
    protected final void scaleImageToPageSize() {

        float width =
            pageSize.getWidth() - (pageMargins.getRight() + pageMargins.getLeft());

        float height =
            pageSize.getHeight() - (pageMargins.getTop() + pageMargins.getBottom());

        image.scaleToFit(width, height);

        image.setAbsolutePosition(pageMargins.getLeft(), pageMargins.getTop());
    }

    /**
     * Shorcut to calculate image Height in inches
     * @return inches
     */
    protected final float getHeightInInches() {
        return image.getHeight() / image.getDpiY();
    }

    /**
     * Shorcut to calculate image Width in inches
     * @return inches
     */
    protected final float getWidthInInches() {
        return image.getWidth() / image.getDpiX();
    }

    /**
     * give child access to image
     * @return
     */
    protected final Image getImage() {
        return image;
    }


}
