package gov.va.cem.common.fileutils;


import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;


public class FileHelper {

    private static final int MAX_MK_DIR_ATTEMPTS = 5;
    private static final int MAX_SLEEP_MS = 30;
    private static final boolean DEFAULT_OVER_WRITE = false;

    public FileHelper() {
        super();
    }

    /**
     *
     * @param srcPath
     * @param destPath
     */
    public void copyFile(String srcPath, String destPath) {
        copyFile(srcPath, destPath, DEFAULT_OVER_WRITE);
    }

    /**
     *
     * @param srcPath
     * @param destPath
     * @param overWrite
     */
    public void copyFile(String srcPath, String destPath, boolean overWrite) {

        if (srcPath == null || srcPath.isEmpty() || destPath == null || destPath.isEmpty()) {
            throw new IllegalArgumentException("srcPath and destPath must be non-null non-empty values.");
        }

        copyFile(new File(srcPath), new File(destPath), overWrite);
    }

    /**
     *
     * @param srcFile
     * @param destFile
     */
    public void copyFile(File srcFile, File destFile) {
        copyFile(srcFile, destFile, DEFAULT_OVER_WRITE);
    }


    private void validateFiles(File srcFile, File destFile, boolean overWrite ) {
        if (srcFile == null || destFile == null) {
            throw new IllegalArgumentException("srcFile and destFile must both be non-null values.");
        }

        if (!srcFile.isFile()) {
            throw new IllegalArgumentException("Source File either does not exist, or is a directory. ["+srcFile.getAbsolutePath()+"]");
        }

        if ( srcFile.equals(destFile) ) {
            throw new IllegalArgumentException("Cannot copy a file to itself. ["+srcFile.getAbsolutePath()+"]");    
        }
        
        if (destFile.exists() && !overWrite) {
            throw new IllegalArgumentException("Destination file exists, will not overwrite");
        }

        if (overWrite && destFile.isDirectory()) {
            throw new IllegalArgumentException("Cannot overwrite directory.["+destFile.getAbsolutePath()+"]");
        }
        
        if (!makeDirs(destFile.getParentFile())) {
            throw new RuntimeException("Failed to create destination directory: " + destFile.getParentFile());
        }
    }

    
    /**
     *
     * @param srcFile
     * @param destFile
     * @param overWrite
     */
    public void copyFile(File srcFile, File destFile, boolean overWrite) {

        validateFiles(srcFile,destFile,overWrite);

        InputStream is;
        try {
            is = new FileInputStream(srcFile);
            try {
                OutputStream os = new FileOutputStream(destFile);
                try {
                    copyStream(is, os);
                } finally {
                    try {
                        os.flush();
                        os.close();
                        os = null;
                    } catch (IOException e) {
                        throw new RuntimeException("Failed to close destination file. " + destFile.getAbsolutePath());
                    }
                }

            } finally {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new RuntimeException("Failed to close source file: " + srcFile.getAbsolutePath());
                }
            }
        } catch (FileNotFoundException e) {
            throw new RuntimeException("Source file not found.");
        }
    }
    
    
    public String copyFiile(String srcPath, String destPath, String digestAlgorithm) {
        return copyFile(srcPath, destPath, digestAlgorithm, DEFAULT_OVER_WRITE);
    }
    
    public String copyFile(String srcPath, String destPath, String digestAlgorithm, boolean overWrite) {
        
        if ( srcPath == null || srcPath.isEmpty() ) {
            throw new IllegalArgumentException("source path cannot be null");
        }
        
        if ( destPath == null || destPath.isEmpty() ) {
            throw new IllegalArgumentException("destination path cannot be null");
        }
        
        return copyFile(new File(srcPath), new File(destPath), digestAlgorithm, overWrite);        
    }
    
    public String copyFile(File srcFile, File destFile, String digestAlgorithm) {
        return copyFile(srcFile, destFile, digestAlgorithm, DEFAULT_OVER_WRITE);
    }
    
    public String copyFile(File srcFile, File destFile, String digestAlgorithm, boolean overWrite) {
        
        String retval;
        
        MessageDigest md;

        try {
            md = MessageDigest.getInstance(digestAlgorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("Unknown digest algorithm,",e);
        }
        
        validateFiles(srcFile, destFile, overWrite);
        
        InputStream is;
        try {
            is = new FileInputStream(srcFile);
            try {
                OutputStream os;
                os = new DigestOutputStream(new FileOutputStream(destFile), md);
                try {                    
                    copyStream(is,os);
                    retval = toHex(md.digest());            
                } finally {                    
                    try {
                        os.flush();
                        os.close();
                    } catch (IOException e) {
                        throw new IllegalStateException("Failed to close output file",e);
                    }
                }
            } finally {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new IllegalStateException("unable to close input file",e);
                }
            }

        } catch (FileNotFoundException e) {
            throw new IllegalArgumentException("Source file does not exist",e);
        }
        
        return retval;
    }
    
    public void copyStream( InputStream input, OutputStream output ) {
        byte[] bytes = new byte[1024 * 64];
        int bytesRead=0;

        try {
            while ( (bytesRead = input.read(bytes)) > 0 ) {
            try {
                output.write(bytes, 0, bytesRead);
            } catch (IOException e) {
                throw new RuntimeException("Failed to write output.",e);
            }
        }
        } catch (IOException e) {
            throw new RuntimeException("Error reading input.",e);
        }
    }

    /**
     *
     * @param dirPath
     * @return
     */
    public boolean makeDirs(String dirPath) {
        return makeDirs(new File(dirPath));
    }

    /**
     *
     * @param dirFile
     * @return
     */
    public boolean makeDirs(File dirFile) {

        int attempts = 0;

        // If a file exists (directory or file) a new directory cannot be created
        // so we won't try.
        if (!dirFile.exists()) {

            // We'll try to create the directory until a file exists with the desired path
            // or we successfully create the directory.
            while (attempts < MAX_MK_DIR_ATTEMPTS && !dirFile.exists() && !dirFile.mkdirs()) {

                // Record the number of time we failed to create the directory
                attempts++;

                // To break deadlocks with other processes, we'll pause for an variable about of time
                // before trying again.
                try {
                    Thread.sleep((long)(Math.random() * MAX_SLEEP_MS)+1);
                } catch (InterruptedException e) {
                    // This is extremely rare, but possible.  If thrown, there is
                    // nothing we can do.
                    throw new RuntimeException("InterruptedException caught while creating directory", e);
                }

            }
        }

        // A file could have been created that is not a directory, or another process may have
        // been successful, so we'll give it one more check before we return to the calling routine.
        return dirFile.isDirectory();

    }
    
    private String toHex(byte[] bytes) {                
        StringBuilder hex = new StringBuilder(bytes.length * 2);   
        for(byte b: bytes) {
            hex.append(String.format("%02x", b));   
        }
        return hex.toString(); 
    }
}

