package gov.va.cem.common.csvfile;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;

public class DataFileReader extends CSVSimpleReader {
	
    private int lastFldCount = -1;
    private boolean fieldCountMismatch = false;
    
    /**
     * Defaults field delimiter and quote delimiter to , and " respectively.
     * @param reader source of data to be parsed
     * @throws IOException
     */
    public DataFileReader(Reader reader) throws IOException {
      super(reader);
    }

    /**
     * Defaults quote delimiter ". 
     * @param reader source of data to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @throws IOException
     */
    public DataFileReader(Reader reader, 
                           CsvDelimiters.Field fieldDelimiter) throws IOException {
        super(reader, fieldDelimiter);
    }

    /**
     * Creates object with specified Field and Quote delimiters.
     * @param reader source of data to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @param quoteDelimiter enum representing char used to Quote field values
     * @throws IOException resulting for Reader
     */
    public DataFileReader(Reader reader, CsvDelimiters.Field fieldDelimiter, 
                           CsvDelimiters.Quote quoteDelimiter) throws IOException {
        super(reader,fieldDelimiter,quoteDelimiter);
    }

    /**
     * Creates object with default Field and Quote delimiters.
     * @param file source file to be parsed
     * @throws FileNotFoundException
     * @throws IOException
     */
    public DataFileReader(File file) throws FileNotFoundException, 
                                             IOException {
        super(file);
    }

    /**
     * Creates object with specified Field delimiter.
     * @param file source file to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @throws FileNotFoundException
     * @throws IOException
     */
    public DataFileReader(File file, 
                           CsvDelimiters.Field fieldDelimiter) throws FileNotFoundException, 
                                                                      IOException {
        this(file, fieldDelimiter, CsvDelimiters.Quote.Default);
    }

    /**
     * Creates object with specified Field and Quote delimiters.
     * @param file source file to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @param quoteDelimiter enum representing char used to Quote field values
     * @throws FileNotFoundException
     * @throws IOException
     */
    public DataFileReader(File file, CsvDelimiters.Field fieldDelimiter, 
                           CsvDelimiters.Quote quoteDelimiter) throws FileNotFoundException, 
                                                                      IOException {
        super(file,fieldDelimiter,quoteDelimiter);
    }
    
    @Override
    public boolean next() throws IOException, InvalidCSVFormatException {
        boolean retval = super.next();
        this.setFieldCountMismatch(false);
        if ( retval ) {
            int curFldCount=-2;
            try {
                curFldCount = getFieldCount();
            } catch (InvalidOperationException e) {
                throw new RuntimeException("This should never happen",e);
            }
            if ( lastFldCount == -1 ) {
                lastFldCount = curFldCount; 
            } else if ( lastFldCount != curFldCount ) {
            	 this.setFieldCountMismatch(true);
                //throw new RuntimeException(String.format("Different number of fields: expected %d got %d",lastFldCount,curFldCount));
            }
        } 
        
        return retval;
    }
    
    /**
     */
    public String[] readHeader() throws IOException {    
        next();
        return getValues();        
    }

	

	public boolean isFieldCountMismatch() {
		return fieldCountMismatch;
	}

	public void setFieldCountMismatch(boolean fieldCountMismatch) {
		this.fieldCountMismatch = fieldCountMismatch;
	}    

}
