package gov.va.cem.common.csvfile;

/**
 * Provides enumerations for Quote, Field and End Of Line delimiters.
 * 
 * These enumerations are used to limit the actual values that can be used
 * for each value while providing access to the underlying data values they
 * represent.
 */
public class CsvDelimiters {

    /**
     * Possible values for Quote delimiter.
     * <ul>
     *    <li>Default = DoubleQuote</li>
     *    <li>DoubleQuote = "</li>
     *    <li>SingleQuote = '</li>
     *    <li>Custom      = null</li>
     * </ul>
     */
    public enum Quote {
        Default('"',"\""),
        DoubleQuote('"',"\""),
        SingleQuote('\'',"'"),
        Custom('\0',null);
        private char quoteChar;
        private String quoteString;
        
        private Quote(char quoteChar, String quoteString) {
            this.quoteChar = quoteChar;
            this.quoteString = quoteString;
        }
        
        /**
         * Returns character value of QuoteDelimiter.
         * @return char value of QuoteDelimiter
         */
        char getCharValue() {
            return this.quoteChar;
        }        
        
        /**
         * Returns String value of QuoteDelimiter.
         * @return String value of QuoteDelimiter
         */
        String getStringValue() {
            return this.quoteString;
        }        
    }

    /**
     * Possible values for Field delimiter.
     * <ul>
     *    <li>Default = Comma</li>
     *    <li>Comma = ,</li>
     *    <li>Tab = \t</li>
     *    <li>SemiColon = ;</li>
     *    <li>Pipe = |</li>
     *    <li>Custom = null</li>
     * </ul> 
     */
    public enum Field {
        Default(',',","),
        Comma(',',","),
        Tab('\t',"\t"),
        SemiColon(';',";"),
        Pipe('|',"|"),
        Custom('\0',null);
        private char fieldChar;
        private String fieldString;
        private Field(char fieldChar, String fieldString) {
            this.fieldChar = fieldChar;
            this.fieldString = fieldString;
        }
        
        /**
         * Returns character value of FieldDelimiter.
         * @return char value of FieldDelimiter,
         */
        char getCharValue() {
            return fieldChar;
        }        
        
        /**
         * Returns String value of FieldDelimiter.
         * @return String value of FieldDelimiter.
         */
        String getStringValue() {
            return fieldString;
        }
        
    }

    /**
     * Possible values for Field delimiter.
     * <ul>
     *    <li>Default = CRLF</li>
     *    <li>CR = \r</li>
     *    <li>CRLF = \r\n</li>
     *    <li>SystemDefault = Sytem.getProperties("line.seperator","\r\n')</li>
     *    <li>Custom = null</li>
     * </ul> 
     */
    public enum EndOfLine {
        Default("\r\n"),
        CRLF("\r\n"),
        LF("\n"),
        SystemDefault(System.getProperties().getProperty("line.seperator","\r\n")),
        Custom(null);
        String eolDelimiter;
        
        private EndOfLine(String eolDelimiter) {
            this.eolDelimiter = eolDelimiter;
        }
        /**
         * Return String value of End Of Line Delimiter.
         * @return String value of End of Line Delimiter
         */
        String getValue() {
            return this.eolDelimiter;
        }
    }
    
}
