package gov.va.cem.common.csvfile;

 /**
  * <p>This class supports creating CSV (Character Separated Values) files.  The
  * field and quote delimiters are parameterized, allowing for the creation of
  * virtually any type of delimited file.</p>
  * <p></p>
  */

public class CSVWriter extends CSVBase {

    private StringBuilder record;
    private StringBuilder field;

    private static final int defRecBufSize = 1024;
    private boolean alwaysQuote = false;


    /**
     * creates a CSVWriter object, use the inherited delimiter, quote and record buffer size as default.
     */
    public CSVWriter() {
        this(CsvDelimiters.Field.Default, CsvDelimiters.Quote.Default, 
             defRecBufSize);
    }

    /**
     * creates a CSVWriter object, use the inherited delimiter, quote as default.
     * @param bufSize to initialize the internal buffer
     */
    public CSVWriter(int bufSize) {
        this(CsvDelimiters.Field.Default, CsvDelimiters.Quote.Default, 
             bufSize);
    }

    /**
     * creates a CSVWriter object, use the inherited delimiter, quote as default.
     * @param fieldDelimiter to separated the fields
     * @param quoteDelimiter to escape the embedded quote in the fields
     */
    public CSVWriter(CsvDelimiters.Field fieldDelimiter, 
                     CsvDelimiters.Quote quoteDelimiter) {
        this(fieldDelimiter, quoteDelimiter, defRecBufSize);
    }

    /**
     * creates a CSVWriter object, use the inherited delimiter as default.
     * @param fieldDelimiter to separate the fields
     */
    public CSVWriter(CsvDelimiters.Field fieldDelimiter) {
        this(fieldDelimiter, CsvDelimiters.Quote.Default, defRecBufSize);
    }

    /**
     * creates a CSVWriter object, use the inherited quote as default.
     * @param fieldDelimiter to separate the fields
     * @param bufSize to initialize the internal buffer
     */
    public CSVWriter(CsvDelimiters.Field fieldDelimiter, int bufSize) {
        this(fieldDelimiter, CsvDelimiters.Quote.Default, bufSize);
    }

    /**
     * creates a CSVWriter object.
     * @param fieldDelimiter to separate the fields
     * @param quoteDelimiter to escape the embedded quote in the fields
     * @param bufSize to initialize the internal buffer
     */
    public CSVWriter(CsvDelimiters.Field fieldDelimiter, 
                     CsvDelimiters.Quote quoteDelimiter, int bufSize) {
        super(fieldDelimiter, quoteDelimiter);
        this.record = new StringBuilder(bufSize);
        this.field = new StringBuilder(defRecBufSize);
    }

    /**
     * Provide means of specifying custom field and quote delimiters.
     * @param fieldDelimiterChar
     * @param quoteDelimiterChar
     */
    public CSVWriter( char fieldDelimiterChar, char quoteDelimiterChar ) {
        super(fieldDelimiterChar, quoteDelimiterChar);
        this.record = new StringBuilder(defRecBufSize);
        this.field = new StringBuilder(defRecBufSize);        
    }
    
    /**
     * Provide means of specifying custom field and quote delimiters.
     * @param fieldDelimiterChar
     * @param quoteDelimiterChar
     * @param bufSize 
     */
    public CSVWriter( char fieldDelimiterChar, char quoteDelimiterChar, int bufSize) {
        super(fieldDelimiterChar, quoteDelimiterChar);
        this.record = new StringBuilder(bufSize);
        this.field = new StringBuilder(defRecBufSize);        
    }
    
    /**
     * set boolean attribute alwaysQuote.
     * @param alwaysQuote to specify whether to enclose the field in quotes
     */
    public void setAlwaysQuote(boolean alwaysQuote) {
        this.alwaysQuote = alwaysQuote;
    }

    /**
     * return attribute alwaysQuote.
     * @return boolean
     */
    public boolean isAlwaysQuote() {
        return this.alwaysQuote;
    }

    /**
     * writes field name from object into a record, then returns the entire record.
     * @param obj Object that implements CSVWritable
     * @return a record in string format
     */
    public String writeHeader(CSVWritable obj) {
        return write(obj.getCSVFieldNames());
    }
    
    
    /**
     * writes values from object into a record, then returns the entire record.
     * @param obj Object that implements CSVWritable
     * @return a record in string format
     */
    public String writeValues(CSVWritable obj) {
        return write(obj.getCSVFieldValues());
    }
    
    /**
     * writes the passed in values into a record, then returns the entire record.
     * @param values representing fields in a string array 
     * @return a record in string format
     */
    public String write(String[] values) {
        // Clear previous value
        record.setLength(0);

        for (String fld : values) {
            record.append(formatField(fld));
            record.append(this.fieldDelimiterChar);
        }

        // Remove trailing fldDelimiter
        record.setLength(record.length() - 1);

        return record.toString();
    }

    /**
     * preserve white space, newline, carriage return characters in the passed in field value by surround the field with quotes.
     * preserve the quote character in the field by escape the quote character with another quote character.
     * @param fldValue
     * @return field in string
     */
    private String formatField(String fldValue) {
        boolean quoteField = alwaysQuote;
        field.setLength(0);

        if ( fldValue != null && fldValue.length() > 0 ) {
            char[] fldChar = fldValue.toCharArray();

            if (fldChar[0] == ' ' || fldChar[0] == '\t' || 
                fldChar[fldChar.length - 1] == ' ' || fldChar[fldChar.length - 1] == '\t') {
                quoteField = true;
            }

            for (char ch : fldChar) {
                if (ch == '\n' || ch == '\r' || 
                    ch == this.fieldDelimiterChar || 
                    ch == quoteDelimiterChar) {
                    quoteField = true;
                }

                if (ch == quoteDelimiterChar)
                    field.append(ch).append(ch);
                else
                    field.append(ch);
            }
        }

        if (quoteField) {
            field.insert(0, quoteDelimiterChar).append(quoteDelimiterChar);
        }

        return field.toString();
    }

    /**
     * return attribute fieldDelimiter.
     * @return CsvDelimiters.Field 
     */
    public CsvDelimiters.Field getFieldDelimiter() {
        return this.fieldDelimiter;
    }

    /**
     * return attribute quoteDelimiter.
     * @return CsvDelimiters.Quote
     */
    public CsvDelimiters.Quote getQuoteDelimiter() {
        return this.quoteDelimiter;
    }
}
