package gov.va.cem.common.csvfile;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

import java.util.List;

/**
 * <p>This class supports creating CSV (Character Seperatorated Values) files.  The
 * field and quote delimiters are parameterized, allowing for the creatation of
 * virtualy any type of delimitted file.</p>
 * <p></p>
 */
public class CSVSimpleWriter extends CSVBase {

    private String eolChar;
    private CsvDelimiters.EndOfLine eolDelimiter;

    private boolean alwaysQuote;
    private CSVWriter csv;
    private Writer output;

    /**
     * Takes the File object, and creates specified file for writing.  If file
     * already exists, it will be overwritten, and all existing data will be
     * lost.  Defaults fldQuote and fldDelimiter to defQuote and defDelimiter
     * respectively.
     *
     * @param outFile file to be written
     * @throws IOException raised if unable to create specified file
     */
    public CSVSimpleWriter(File outFile) throws IOException {
        this(outFile, CsvDelimiters.Field.Default,
             CsvDelimiters.Quote.Default);
    }

    /**
     * Takes the File object, and creates specified file for writing.  If file
     * already exists, it will be overwritten, and all existing data will be
     * lost.  Defaults fldQuote to defQuote.
     *
     * @param outFile file to be written
     * @param fieldDelimiter to seperate the fields
     * @throws IOException to be thrown when encounters disk operation error
     */
    public CSVSimpleWriter(File outFile,
                           CsvDelimiters.Field fieldDelimiter) throws IOException {
        this(outFile, fieldDelimiter, CsvDelimiters.Quote.Default);
    }

    /**
     * Takes the File object, and creates specified file for writing.  If file
     * already exists, it will be overwritten, and all existing data will be
     * lost.
     *
     * @param outFile file to be written
     * @param fieldDelimiter to seperate the fields
     * @param quoteDelimiter to escape the embedded quote in the fields
     * @throws IOException to be thrown when encounters disk operation error
     */
    public CSVSimpleWriter(File outFile, CsvDelimiters.Field fieldDelimiter,
                           CsvDelimiters.Quote quoteDelimiter) throws IOException {
        super(fieldDelimiter, quoteDelimiter);
        csv = new CSVWriter(this.fieldDelimiter, this.quoteDelimiter);
        setEolDelimiter(CsvDelimiters.EndOfLine.Default);
        output = new FileWriter(outFile);
    }

    public CSVSimpleWriter(File outFile, CsvDelimiters.Field fieldDelimiter,
                           CsvDelimiters.Quote quoteDelimiter,
                           CsvDelimiters.EndOfLine eol) throws IOException {
        super(fieldDelimiter, quoteDelimiter);
        csv = new CSVWriter(this.fieldDelimiter, this.quoteDelimiter);
        setEolDelimiter(eol);
        output = new FileWriter(outFile);
    }

    public CSVSimpleWriter(File outFile, char fieldDelimiterChar,
                           char quoteDelimiterChar) throws IOException {
        super(fieldDelimiterChar, quoteDelimiterChar);
        csv = new CSVWriter(this.fieldDelimiterChar, this.quoteDelimiterChar);
        setEolDelimiter(CsvDelimiters.EndOfLine.Default);
        output = new FileWriter(outFile);
    }

    public CSVSimpleWriter(File outFile, char fieldDelimiterChar,
            char quoteDelimiterChar, boolean appendToFile) throws IOException {
        super(fieldDelimiterChar, quoteDelimiterChar);

        csv = new CSVWriter(this.fieldDelimiterChar, this.quoteDelimiterChar);
        setEolDelimiter(CsvDelimiters.EndOfLine.Default);
        output = new FileWriter(outFile, appendToFile);
    }

    public CSVSimpleWriter(File outFile, char fieldDelimiterChar,
                           char quoteDelimiterChar,
                           CsvDelimiters.EndOfLine eolDelimiter) throws IOException {
        super(fieldDelimiterChar, quoteDelimiterChar);
        csv = new CSVWriter(this.fieldDelimiterChar, this.quoteDelimiterChar);
        setEolDelimiter(eolDelimiter);
        output = new FileWriter(outFile);
    }


    /**
     * Takes the Writer object, and uses it for writing output data.  Data
     * will be written starting at the current Writer position. Defaults
     * fldQuote and fldDelimiter to defQuote and defDelimiter respectively.
     *
     * @param writer to write the output data
     * @throws IOException to be thrown when encounters disk operation error
     */
    public CSVSimpleWriter(Writer writer) throws IOException {
        this(writer, CsvDelimiters.Field.Default, CsvDelimiters.Quote.Default);
    }

    /**
     * Takes the Writer object, and uses it for writing output data.  Data
     * will be written starting at the current Writer position. Defaults
     * fldQuote to defQuote.
     *
     * @param writer to write the output data
     * @param fieldDelimiter to seperate the fields
     * @throws IOException to be thrown when encounters disk operation error
     */
    public CSVSimpleWriter(Writer writer,
                           CsvDelimiters.Field fieldDelimiter) throws IOException {
        this(writer, fieldDelimiter, CsvDelimiters.Quote.Default);
    }

    /**
     * Takes the Writer object, and uses it for writing output data.  Data
     * will be written starting at the current Writer position.
     *
     * @param writer to write the output data
     * @param fieldDelimiter to seperate the fields
     * @param quoteDelimiter to escape the embedded quote in the fields
     * @throws IOException to be thrown when encounters disk operation error
     */
    public CSVSimpleWriter(Writer writer, CsvDelimiters.Field fieldDelimiter,
                           CsvDelimiters.Quote quoteDelimiter) throws IOException {
        super(fieldDelimiter, quoteDelimiter);
        csv = new CSVWriter(this.fieldDelimiter, this.quoteDelimiter);
        setEolDelimiter(CsvDelimiters.EndOfLine.Default);
        output = writer;
    }

    /**
     *
     * @param writer
     * @param fieldDelimiterChar
     * @param quoteDelimiterChar
     * @throws IOException
     */
    public CSVSimpleWriter(Writer writer, char fieldDelimiterChar,
                           char quoteDelimiterChar) throws IOException {
        super(fieldDelimiterChar, quoteDelimiterChar);
        csv = new CSVWriter(this.fieldDelimiterChar, this.quoteDelimiterChar);
        setEolDelimiter(CsvDelimiters.EndOfLine.Default);
        output = writer;
    }

    /**
     *
     * @param writer
     * @param fieldDelimiterChar
     * @param quoteDelimiterChar
     * @param eolDelimiter
     * @throws IOException
     */
    public CSVSimpleWriter(Writer writer, char fieldDelimiterChar,
                           char quoteDelimiterChar,
                           CsvDelimiters.EndOfLine eolDelimiter) throws IOException {
        super(fieldDelimiterChar, quoteDelimiterChar);
        csv = new CSVWriter(this.fieldDelimiterChar, this.quoteDelimiterChar);
        setEolDelimiter(eolDelimiter);
        output = writer;
    }

    /**
     * writeWithHeader
     * @param dataSet
     * @throws IOException
     */
    public void writeWithHeader(List<CSVWritable> dataSet) throws IOException {

        // if the object support a header, write the header
        if (dataSet.size() > 0 && dataSet.get(0).hasFieldNames()) {
            output.write(csv.writeHeader(dataSet.get(0)));
        }

        write(dataSet);
    }

    /**
     * write
     * @param dataSet
     * @throws IOException
     */
    public void write(List<CSVWritable> dataSet) throws IOException {
        // Now write the data
        for (CSVWritable data : dataSet) {
            output.write(csv.writeValues(data));
        }
        output.flush();
    }

    /**
     * Writes CSV formatted record to writer.
     *
     * @param values each element represents a field value
     * @throws IOException raised from Writer
     */
    public void write(String[] values) throws IOException {
        output.write(csv.write(values));
        output.write(eolChar);
        output.flush();
    }

    /**
     * Closes writer. If the object was created from a File object, this must
     * be called after final record is written.  If a Writer was passed in, then
     * don't call this, instead call the Writer.close().
     *
     * @throws IOException
     */
    public void close() throws IOException {
    	output.flush();
        output.close();
    }

    /**
     * When set to true, this value causes field values to always be enclosed in
     * fldQuotes.  If the value is false, field will only be quoted when required.
     *
     * @param alwaysQuote to specify whether to enclose the field in quotes
     */
    public void setAlwaysQuote(boolean alwaysQuote) {
        this.alwaysQuote = alwaysQuote;
        if (csv != null) {
            csv.setAlwaysQuote(alwaysQuote);
        }
    }

    /**
     * When true, this value causes field values to always be enclosed in
     * fldQuotes. If the value is false, field will only be quoted when required.
     *
     * @return boolean
     */
    public boolean isAlwaysQuote() {
        return alwaysQuote;
    }

    /**
     * set attribute eolDelimiter.
     * @param eolDelimiter to set the end of the line attribute
     */
    protected void setEolDelimiter(CsvDelimiters.EndOfLine eolDelimiter) {
        this.eolDelimiter = eolDelimiter;
        this.eolChar = eolDelimiter.getValue();
    }

    /**
     * return attribute eolDelimiter.
     * @return CsvDelimiters.EndOfLine
     */
    protected CsvDelimiters.EndOfLine getEolDelimiter() {
        return eolDelimiter;
    }
}
