package gov.va.cem.common.csvfile;


import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;

import java.util.ArrayList;

/**
 * CSVSimpleReader provides record set style interface for reading CSV files
 * with parameterized delimiters.
 *
 * This implementation supports embedded quotes and embedded line feeds (LF or CRLF)
 * within quoted fields.
 */
public class CSVSimpleReader extends CSVBase {

    /**
     * Buffer size for buffered reader.
     */
    protected static final int BLOCK_SIZE = 8 * 1024;

    private BufferedReader reader;

    private StringBuilder field = new StringBuilder(1024);
    private StringBuilder record = new StringBuilder(4 * 1024);

    private boolean eof;
    private boolean endOfRecord;
    private boolean hasRowData;
    private char curChar;

    private ArrayList<String> values = new ArrayList<String>(256);
    /*
     * Public interface
     */

    // Constructors - provided in both Reader and File interfaces

    /**
     * Defaults field delimiter and quote delimiter to , and " respectively.
     * @param reader source of data to be parsed
     * @throws IOException
     */
    public CSVSimpleReader(Reader reader) throws IOException {
        this(reader, CsvDelimiters.Field.Default, CsvDelimiters.Quote.Default);
    }

    /**
     * Defaults quote delimiter.
     * @param reader source of data to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @throws IOException
     */
    public CSVSimpleReader(Reader reader,
                           CsvDelimiters.Field fieldDelimiter) throws IOException {
        this(reader, fieldDelimiter, CsvDelimiters.Quote.Default);
    }

    /**
     * Creates object with specified Field and Quote delimiters.  If reader
     * is not a BufferedReader it is wrapped in a BufferedReader so that
     * mark is supported.
     * @param reader source of data to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @param quoteDelimiter enum representing char used to Quote field values
     * @throws IOException resulting for Reader
     */
    public CSVSimpleReader(Reader reader, CsvDelimiters.Field fieldDelimiter,
                           CsvDelimiters.Quote quoteDelimiter) throws IOException {
        super(fieldDelimiter, quoteDelimiter);
        if (reader instanceof BufferedReader)
            this.reader = (BufferedReader)reader;
        else
            this.reader = new BufferedReader(reader, BLOCK_SIZE);
    }

    /**
     * Creates object with default Field and Quote delimiters.
     * @param file source file to be parsed
     * @throws FileNotFoundException
     * @throws IOException
     */
    public CSVSimpleReader(File file) throws FileNotFoundException,
                                             IOException {
        this(file, CsvDelimiters.Field.Default, CsvDelimiters.Quote.Default);
    }

    /**
     * Creates object with specified Field delimiter.
     * @param file source file to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @throws FileNotFoundException
     * @throws IOException
     */
    public CSVSimpleReader(File file,
                           CsvDelimiters.Field fieldDelimiter) throws FileNotFoundException,
                                                                      IOException {
        this(file, fieldDelimiter, CsvDelimiters.Quote.Default);
    }

    /**
     * Creates object with specified Field and Quote delimiters.
     * @param file source file to be parsed
     * @param fieldDelimiter enum representing char separating field values
     * @param quoteDelimiter enum representing char used to Quote field values
     * @throws FileNotFoundException
     * @throws IOException
     */
    public CSVSimpleReader(File file, CsvDelimiters.Field fieldDelimiter,
                           CsvDelimiters.Quote quoteDelimiter) throws FileNotFoundException,
                                                                      IOException {
        super(fieldDelimiter, quoteDelimiter);
        
        String encoding = BOMDector.getEncoding(file);
        if ( encoding == null ) {  
            this.reader = new BufferedReader(new FileReader(file));
        } else {
            FileInputStream fis = new FileInputStream(file);
            this.reader = new BufferedReader(new InputStreamReader(fis,encoding));
            if ( encoding.equals(BOMDector.UTF8) ) {
                reader.mark(5);
                int bomChar = reader.read();
                if ( bomChar != BOMDector.bomChar ) {
                    reader.reset();
                }                
            }
        }
    }

    // Application Program Interface (API)

    /**
     * Parses record from data stream, and returns true if successful.
     * @return true if record found
     * @throws IOException error reading from Reader/File
     * @throws InvalidCSVFormatException failed to successfully parse record
     */
    public boolean next() throws IOException, InvalidCSVFormatException {
        return parseRecord();
    }

    /**
     * Returns number of fields parsed in last call to next().
     * <p>
     * If call to next return false or throws an exception, the object is left
     * in an invalid state.  Trying to call getFieldCount() with the object in
     * an invalid state will throw an InvalidOperationException.
     * </p>
     * @return number of fields in current record
     * @throws InvalidOperationException
     */
    public int getFieldCount() throws InvalidOperationException {
        if (hasRowData)
            return values.size();
        else
            throw new InvalidOperationException("no record data available");
    }

    /**
     * Returns value of specified field ( 0 >= index < getFieldCount() ).
     * <p>
     * If call to next return false or throws an exception, the object is left
     * in an invalid state.  Trying to call getValue() with the object in
     * an invalid state will throw an InvalidOperationException.
     * </p>
     * @param index field index starting at 0
     * @return String value of parsed field
     * @throws InvalidOperationException if no data available
     */
    public String getValue(int index) throws InvalidOperationException {
        if (hasRowData)
            return values.get(index);
        else
            throw new InvalidOperationException("no record data available");
    }

    /**
     * Returns String array of field values.
     * <p>
     * If call to next return false or throws an exception, the object is left
     * in an invalid state.  Trying to call getValues() with the object in
     * an invalid state will throw an InvalidOperationException.
     * </p>
     * @return String array of parsed field values
     * @throws InvalidOperationException
     */
    public String[] getValues() throws InvalidOperationException {
        if (hasRowData)
            return values.toArray(new String[0]);
        else
            throw new InvalidOperationException("no record data available");
    }

    /**
     * Return String representing unparsed data from stream.
     * <p>The record value may be slightly modified from the original source due
     * to the elimination of space characters around the field delimiter, but if
     * parsed a second time, will result in the same field results.
     * </p>
     * @return Unparsed record as read from Reader. Record starts with either
     *         first fldQuote or first non-space character and ends with the
     *         last fldQuote or non-space character in record. All spaces
     *         arround fldDelimitter will be removed. The record delimiter is
     *         not included.
     */
    protected String getRecord() {
        return record.toString();
    }

    /**
     *
     * @return indicates that data was parsed into record
     * @throws IOException
     * @throws InvalidCSVFormatException
     */
    private boolean parseRecord() throws IOException,
                                         InvalidCSVFormatException {
        // Initialize control values
        hasRowData = false;
        endOfRecord = false;

        record.setLength(0);

        values.clear();
        try {
            // find beginning of first field
            do {
                advanceToNonSpaceCharacter();
                if (!eof) {
                    if (curChar == quoteDelimiterChar) {
                        captureQuotedField();
                    } else {
                        captureToNextFldDelimiter();
                    }
                }
            } while (!endOfRecord && !eof);

        } catch (IOException e) {
            hasRowData = false;
            //throw e;
        } catch (InvalidCSVFormatException e) {
            hasRowData = false;
            //throw e;
        }

        return hasRowData;
    }

    /**
     * Captures data from inside a Quoted field
     */
    private void captureQuotedField() throws IOException,
                                             InvalidCSVFormatException {
        boolean endOfField = false;

        // Skip opening quote since it has already been processed
        advancePosWithoutCapture();

        /*
         * Parse out the field value.
         */
        do {
            captureToNextQuote();

            if (!eof) {
                // Check for two fldQuotes in a row
                if (peekNextChar() == quoteDelimiterChar) {
                    // We only keep one quote in field
                    field.append(quoteDelimiterChar);

                    // consume the escaping quote
                    advancePosWithoutCapture();

                    // consume the captured quote
                    advancePosWithoutCapture();
                } else {
                    // We've reached the end of field
                    endOfField = true;
                    addQuotedFieldToRecord();
                    saveCurrentField(false); // no trimming
                }
            } else {
                throw new InvalidCSVFormatException("Quoted field not closed");
            }

        } while (!endOfField);

        advanceToNonSpaceCharacter();
        checkForEndOfRecord();
    }

    /**
     *
     */
    private void addQuotedFieldToRecord() {
        StringBuilder sb = new StringBuilder(field);
        String quoteString = quoteDelimiter.getStringValue();

        // Escape any imbedded quoteChars
        int index = sb.lastIndexOf(quoteString);
        while ( index != -1 ) {
            sb.insert(index,quoteString);
            index = sb.lastIndexOf(quoteString,--index);
        }

        // surround with quoteCharacter
        sb.insert(0,quoteDelimiterChar);
        sb.append(quoteDelimiterChar);

        // And add to record
        if (record.length() > 0) {
            record.append(fieldDelimiterChar);
        }
        record.append(sb);
    }

    /**
     * Captures data from stream until fldDelimiter, \n, \r\n, or delQuote
     * is encountered. An InvalidCSVFormatException will be throw if delQuote
     * is found before either \n or fldDelimiter and if \r is encounter without
     * the next char being \n
     * @throws InvalidCSVFormatException
     */
    private void captureToNextFldDelimiter() throws InvalidCSVFormatException,
                                                    IOException {
        boolean endOfField = false;

        do {

            // fldDelimiter end field
            if (curChar == this.fieldDelimiterChar) {
                endOfField = true;

                // end of record is also end of field
            } else if (checkForEndOfRecord()) {
                endOfField = true;

                // We should not find a qoute until we find a field delimiter
            } else if (curChar == this.quoteDelimiterChar) {
                throw new InvalidCSVFormatException("Unexpected quote char found.");

                // Otherwise we need to cature the character into the field
            } else {
                field.append(curChar);
                advancePosWithoutCapture();
            }

        } while (!(eof || endOfField));

        addNonQuotedFldToRecord();
        saveCurrentField(true); /* needsTrimmed */

    }

    /**
     *
     */
    private void addNonQuotedFldToRecord() {
        if (record.length() > 0) {
            record.append(fieldDelimiterChar);
        }
        record.append(field.toString().trim());
    }

    /**
     * Read next character from reader, setting curChar to character read.
     * If end of file was returned, set eof to true.
     * @throws IOException
     */
    private void advancePosWithoutCapture() throws IOException {
        int ch = reader.read();
        curChar = (char)ch;
        if (ch == -1) {
            eof = true;
        }
    }

    /**
     * Read characters from file until first non-space character is found.
     * <p>
     * Space characters include both spaces and tabs unless the field delimiter
     * is a tab, in which case only spaces count.
     * </p>
     * @throws IOException
     */
    private void advanceToNonSpaceCharacter() throws IOException {
        char tabValue = CsvDelimiters.Field.Tab.getCharValue();
        if (fieldDelimiterChar == tabValue) {
            do {
                advancePosWithoutCapture();
            } while (!eof && curChar == ' ');

        } else {
            do {
                advancePosWithoutCapture();
            } while (!eof && (curChar == ' ' || curChar == tabValue));
        }
    }

    /**
     *
     * @throws IOException
     */
    private void captureToNextQuote() throws IOException {
        while (!eof && curChar != quoteDelimiterChar) {
            field.append(curChar);
            advancePosWithoutCapture();
        }
    }

    private boolean checkForEndOfRecord() throws IOException,
                                                 InvalidCSVFormatException {
        if (eof) {
            endOfRecord = true;
        }
        if (curChar == '\n') {
            endOfRecord = true;
        } else if (curChar == '\r') {
            if (peekNextChar() == '\n') {
                advancePosWithoutCapture();
                endOfRecord = true;
            } else {
                throw new InvalidCSVFormatException("CR found without LF");
            }
        }
        return endOfRecord;
    }

    /**
     *
     * @return
     * @throws IOException
     */
    private char peekNextChar() throws IOException {
        reader.mark(1);
        int retval = reader.read();
        reader.reset();
        return (char)retval;
    }

    /**
     * Takes contents from
     * @param needsTrimmed
     */
    private void saveCurrentField(boolean needsTrimmed) {

        hasRowData = true;

        if (needsTrimmed)
            values.add(field.toString().trim());
        else
            values.add(field.toString());

        field.setLength(0);
    }

    public void close() {
        try {
            reader.close();
        } catch (IOException e) {
            throw new RuntimeException("failed closing csv file", e);
        }
    }

}
