package com.agilex.vamf.missionhealth.datalayer;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.agilex.vamf.missionhealth.domain.*;
import com.agilex.vamf.missionhealth.domain.enumeration.StandingType;
import com.agilex.vamf.missionhealth.service.MissionHealthWeeklyStatus;
import com.agilex.vamf.missionhealth.service.PointsRule;
import com.agilex.vamf.missionhealth.service.ThresholdValues;

import com.agilex.vamf.missionhealth.service.WeeklyUserMetrics;
import org.bson.types.ObjectId;

import com.agilex.vamf.missionhealth.domain.enumeration.MissionStatus;
import com.agilex.vamf.utils.DateHelper;
import com.agilex.vamf.utils.NullChecker;
import com.agilex.vamf.utils.StringUtil;
import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;

public class MissionHealthTranslator {

    protected static final String MONGO_ID = "_id";

    /* MissionHealthDetail fields */
    private static final String MISSION_DETAIl_ID_KEY = "missionHealthDetailSID";
    private static final String MISSION_HEALTH_PROFILE = "missionHealthProfile";
    private static final String LOGIN_DATE = "loginDate";
    private static final String LAST_SESSION = "lastSessionStart";
    private static final String SMART_INFO_DATE="smartInfoDate";
    private static final String RANK = "rank";
    private static final String PENDING_RANK = "pendingRank";
    private static final String PREVIOUS_RANK = "previousRank";
    private static final String RANK_EFFECTIVE_DATE = "rankEffectiveDate";
    private static final String PROMOTED = "promoted";
    private static final String PROFILE_POINTS = "profilePoints";
    private static final String POINTS = "points";
    private static final String TROOPS = "troops";
    private static final String MISSIONS = "missions";
    private static final String CURRENT_MISSIONS = "currentMissions";
    private static final String TOTAL_MISSIONS = "totalMissions";
    private static final String TOTAL_COMPLETED_MISSIONS = "totalCompletedMissions";
    private static final String TOTAL_LOGINS_FOR_CYCLE = "totalLoginsForCycle";
    protected static final String TOTAL_LOGIN_DAYS_FOR_CYCLE = "totalLoginDaysForCycle";
    private static final String FAILED_MISSION_COUNTER="failedMissionCounter";
    protected static final String MISSION_READINESS_RPTS = "missionReadinessReports";
    private static final String LAST_MISSION_CREATE_DATE = "lastMissionCreateDate";
    private static final String LOST_TROOPS_SHOW_MESSAGE= "lostTroopsShowMessage";
    private static final String TOTAL_ATTEMPTED_MISSIONS= "totalAttemptedMissions";
    private static final String STANDING = "standing";
    private static final String ANNUAL_TOTAL_COMPLETED_MISSIONS="annualTotalCompletedMissions";
    private static final String ANNUAL_TOTAL_POINTS="annualTotalPoints";
    private static final String ANNUAL_TOTAL_TROOPS="annualTotalTroops";
    private static final String INACTIVE_CYCLES="inactiveCycles";
    private static final String DEVICE_ID="deviceId";
    
    /* MissionHealthProfile fields */
    private static final String PROFILE_ID_KEY = "missionHealthProfileSID";
	protected static final String PATIENT_ID = "patientId";
	private static final String NAME = "name";
	private static final String SERVICE = "service";
	private static final String GENDER = "gender";
	private static final String DATE_OF_BIRTH = "dateOfBirth";
	private static final String REMINDER = "reminder";
	private static final String EMAIL = "email";
    private static final String LAST_NAME = "lastName";
    private static final String SERVICE_LAST_RANKED = "serviceLastRankedAs";

    /* Mission fields */
    private static final String MISSION_ID_KEY = "missionId";
	private static final String TOTAL_POINTS = "totalPoints";
    private static final String POINTS_WON = "pointsWon";
    private static final String POINTS_FORFEITED = "pointsForfeited";
	private static final String THRESHOLD_PERCENTAGE = "thresholdPercentage";
    private static final String PERCENTAGE_COMPLETED = "percentageCompleted";
	private static final String STATUS = "status";
	private static final String HOME_SERVICE = "homeService";
	private static final String COMPETING_SERVICE = "competingService";
	private static final String START_DATE = "startDate";
    private static final String DESCRIPTION="description";
    private static final String SHOW_MESSAGE="showMessage";
    private static final String TROOPS_FORFEITED="troopsForfeited";

    /* MissionReadinessReport fields */
    private static final String MISSION_READINESS_RPT_HEALTH="health";
    private static final String MISSION_READINESS_RPT_PHYSICAL="physical";
    private static final String MISSION_READINESS_RPT_EMOTIONAL="emotional";
    private static final String MISSION_READINESS_RPT_INSDATE="insDate";

    /* PointsRule fields */
    private static final String MISSION_THRESHOLDS = "missionThresholds";
    private static final String POINT_THRESHOLDS = "pointThresholds";
    private static final String LOWER = "lower";
    private static final String UPPER = "upper";
    private static final String INTERVAL = "interval";
    private static final String MISSIONS_TO_ACHIEVE_LEVEL = "missionsToAchieveLevel";
    private static final String POINTS_FOR_NEW_TROOP = "pointsForNewTroop";
    private static final String TROOPS_ALLOCATED="troopsAllocated";

    /* MissionHealthRank fields */
    private static final String LEVEL = "level";
    private static final String PAY_GRADE = "payGrade";
    private static final String ABBR = "abbr";
    private static final String TITLE = "title";
    private static final String SHORT_TITLE = "shortTitle";

    /* MissionHealthStanding fields */
    private static final String VS_ALL = "vsAll";
    private static final String VS_SERVICE = "vsService";
    private static final String VS_LEVEL = "vsLevel";
    private static final String POINTS_LAST_4_MISSIONS = "pointsLast4Missions";
    private static final String LAST_4_MISSIONS = "last4Missions";
	
	public DBObject translateMissionHealthProfile(MissionHealthProfile missionHealthProfile, boolean hasRecentMissions) {
		DBObject dbMissionHealthProfile = new BasicDBObject();
		
		if (NullChecker.isNullish(missionHealthProfile.getId())) {
			dbMissionHealthProfile.put(PROFILE_ID_KEY, new ObjectId().toString());
		} else {
			dbMissionHealthProfile.put(PROFILE_ID_KEY, missionHealthProfile.getId());
		}
		dbMissionHealthProfile.put(PATIENT_ID, missionHealthProfile.getPatientId());
		dbMissionHealthProfile.put(NAME, missionHealthProfile.getName());
		dbMissionHealthProfile.put(SERVICE, missionHealthProfile.getService());
        if(null!= missionHealthProfile.getServiceLastRankedAs() && hasRecentMissions){
            dbMissionHealthProfile.put(SERVICE_LAST_RANKED, missionHealthProfile.getServiceLastRankedAs());
        }else{
            dbMissionHealthProfile.put(SERVICE_LAST_RANKED, missionHealthProfile.getService());
        }
        dbMissionHealthProfile.put(GENDER, missionHealthProfile.getGender());
		dbMissionHealthProfile.put(DATE_OF_BIRTH, DateHelper.formatDate(missionHealthProfile.getDateOfBirth()));
		dbMissionHealthProfile.put(REMINDER, missionHealthProfile.getReminder());
		dbMissionHealthProfile.put(EMAIL, missionHealthProfile.getEmail());
		dbMissionHealthProfile.put(LAST_NAME, missionHealthProfile.getLastName());
		return dbMissionHealthProfile;
	}

	public DBObject translateMissionHealthDetail(MissionHealthDetail missionHealthDetail) {
		DBObject dbMissionHealthDetail = new BasicDBObject();
		
		if (NullChecker.isNullish(missionHealthDetail.getId())) {
			dbMissionHealthDetail.put(MISSION_DETAIl_ID_KEY, new ObjectId().toString());
		} else {
			dbMissionHealthDetail.put(MISSION_DETAIl_ID_KEY, missionHealthDetail.getId());
		}
		dbMissionHealthDetail.put(PATIENT_ID, missionHealthDetail.getPatientId());
		dbMissionHealthDetail.put(RANK, translateMissionHealthRank(missionHealthDetail.getRank()));
        dbMissionHealthDetail.put(PENDING_RANK, translateMissionHealthRank(missionHealthDetail.getPendingRank()));
        dbMissionHealthDetail.put(PREVIOUS_RANK, translateMissionHealthRank(missionHealthDetail.getPreviousRank()));
        dbMissionHealthDetail.put(RANK_EFFECTIVE_DATE, DateHelper.formatDate(missionHealthDetail.getRankEffectiveDate()));
        dbMissionHealthDetail.put(PROMOTED, missionHealthDetail.isPromoted());
		dbMissionHealthDetail.put(POINTS, missionHealthDetail.getPoints());
		dbMissionHealthDetail.put(PROFILE_POINTS, missionHealthDetail.getMissionProfilePoints());
		dbMissionHealthDetail.put(TROOPS, missionHealthDetail.getTroops());
		dbMissionHealthDetail.put(FAILED_MISSION_COUNTER, missionHealthDetail.getFailedMissionCounter());
		dbMissionHealthDetail.put(LOGIN_DATE, DateHelper.formatDate(missionHealthDetail.getLoginDate()));
		dbMissionHealthDetail.put(SMART_INFO_DATE, DateHelper.formatDate(missionHealthDetail.getSmartInfoDate()));
        dbMissionHealthDetail.put(TOTAL_MISSIONS, missionHealthDetail.getTotalMissions());
        dbMissionHealthDetail.put(TOTAL_COMPLETED_MISSIONS, missionHealthDetail.getTotalCompletedMissions());
        dbMissionHealthDetail.put(ANNUAL_TOTAL_COMPLETED_MISSIONS, missionHealthDetail.getAnnualTotalCompletedMissions());
        dbMissionHealthDetail.put(ANNUAL_TOTAL_POINTS, missionHealthDetail.getAnnualTotalPoints());
        dbMissionHealthDetail.put(ANNUAL_TOTAL_TROOPS, missionHealthDetail.getAnnualTotalTroops());
        dbMissionHealthDetail.put(TOTAL_ATTEMPTED_MISSIONS, missionHealthDetail.getTotalAttemptedMissions());
        dbMissionHealthDetail.put(TOTAL_LOGIN_DAYS_FOR_CYCLE, missionHealthDetail.getTotalLoginDaysForCycle());
        dbMissionHealthDetail.put(TOTAL_LOGINS_FOR_CYCLE, missionHealthDetail.getTotalLoginsForCycle());
        dbMissionHealthDetail.put(LAST_SESSION,  DateHelper.formatDateTimeWithMilliseconds(missionHealthDetail.getLastSessionStart()));
        dbMissionHealthDetail.put(LAST_MISSION_CREATE_DATE, DateHelper.formatDateTimeInVistaFormat(missionHealthDetail.getLastMissionCreateDate()))         ;
		dbMissionHealthDetail.put(LOST_TROOPS_SHOW_MESSAGE, missionHealthDetail.isLostTroopsShowMessage());
        if(missionHealthDetail.getInactiveCycles() != null){
            dbMissionHealthDetail.put(INACTIVE_CYCLES, missionHealthDetail.getInactiveCycles());
        }
        if (missionHealthDetail.getMissionHealthProfile() != null) {
			DBObject dbMissionHealthProfile = translateMissionHealthProfile(missionHealthDetail.getMissionHealthProfile(), missionHealthDetail.hasRecentMissions());
			dbMissionHealthDetail.put(MISSION_HEALTH_PROFILE, dbMissionHealthProfile);
		}	
		if (missionHealthDetail.getMissions() != null) {
			BasicDBList dbList = new BasicDBList();
			if(!missionHealthDetail.getMissions().isEmpty()){
                Iterator<Mission> itr = missionHealthDetail.getMissions().iterator();
                while (itr.hasNext()) {
                    Mission mission = itr.next();
                    dbList.add(translateMission(mission));
                }
            }
			dbMissionHealthDetail.put(MISSIONS, dbList);
		}
	    if (missionHealthDetail.getMissionReadinessReports() != null && !missionHealthDetail.getMissionReadinessReports().isEmpty()) {
			dbMissionHealthDetail.put(MISSION_READINESS_RPTS, populateMissionReadinessReportsDbList(missionHealthDetail.getMissionReadinessReports()));
		}
        if (missionHealthDetail.getCurrentMissions() != null) {
            BasicDBList dbList = new BasicDBList();
            if(!missionHealthDetail.getCurrentMissions().isEmpty()){
                Iterator<Mission> itr = missionHealthDetail.getCurrentMissions().iterator();
                while (itr.hasNext()) {
                    Mission mission = itr.next();
                    dbList.add(translateMission(mission));
                }
            }
            dbMissionHealthDetail.put(CURRENT_MISSIONS, dbList);
        }
        if(null!=missionHealthDetail.getDeviceId()){
            dbMissionHealthDetail.put(DEVICE_ID, missionHealthDetail.getDeviceId());
        }
		return dbMissionHealthDetail;
	}

    protected BasicDBList populateMissionReadinessReportsDbList(List<MissionReadinessReport> missionReadinessReports){
        BasicDBList dbRptList = new BasicDBList();
        Iterator<MissionReadinessReport> rptItr = missionReadinessReports.iterator();
        while (rptItr.hasNext()) {
            MissionReadinessReport missionReadinessReport = rptItr.next();
            DBObject dbMissionReadinessReport = new BasicDBObject();
            dbMissionReadinessReport.put(MISSION_READINESS_RPT_HEALTH, missionReadinessReport.getHealth());
            dbMissionReadinessReport.put(MISSION_READINESS_RPT_PHYSICAL, missionReadinessReport.getPhysical());
            dbMissionReadinessReport.put(MISSION_READINESS_RPT_EMOTIONAL, missionReadinessReport.getEmotional());
            dbMissionReadinessReport.put(MISSION_READINESS_RPT_INSDATE, DateHelper.formatDate(missionReadinessReport.getInsDate()));
            dbRptList.add(dbMissionReadinessReport);
        }
        return dbRptList;
    }

    public DBObject translateMission(Mission mission){
        DBObject dbMission = new BasicDBObject();
        dbMission.put(MISSION_ID_KEY, mission.getId());
        dbMission.put(NAME, mission.getName());
        dbMission.put(DESCRIPTION, mission.getDescription());
        dbMission.put(STATUS, mission.getStatus().toString());
        dbMission.put(START_DATE, DateHelper.formatDateTimeInVistaFormat(mission.getStartDate()));
        dbMission.put(TOTAL_POINTS, mission.getTotalPoints());
        dbMission.put(POINTS_WON, mission.getPointsWon());
        dbMission.put(POINTS_FORFEITED, mission.getPointsForfeited());
        dbMission.put(THRESHOLD_PERCENTAGE,mission.getThresholdPercentage());
        dbMission.put(PERCENTAGE_COMPLETED,mission.getPercentageCompleted());
        dbMission.put(HOME_SERVICE, mission.getHomeService());
        dbMission.put(COMPETING_SERVICE, mission.getCompetingService());
        dbMission.put(SHOW_MESSAGE, mission.isShowMessage());
        dbMission.put(TROOPS_FORFEITED, mission.getTroopsForfeited());
        return dbMission;
    }
	
	public MissionHealthProfile translateToMissionHealthProfile(DBObject po) {
		MissionHealthProfile missionHealthProfile = new MissionHealthProfile();

        missionHealthProfile.setId(StringUtil.objectToString(po.get(PROFILE_ID_KEY)));
        missionHealthProfile.setPatientId(StringUtil.objectToString(po.get(PATIENT_ID)));
        missionHealthProfile.setName(StringUtil.objectToString(po.get(NAME)));
        missionHealthProfile.setService(StringUtil.objectToString(po.get(SERVICE)));
        if(null!=po.get(SERVICE_LAST_RANKED)){
            missionHealthProfile.setServiceLastRankedAs(StringUtil.objectToString(po.get(SERVICE_LAST_RANKED)));
        }else {
            missionHealthProfile.setServiceLastRankedAs(StringUtil.objectToString(po.get(SERVICE)));
        }
        missionHealthProfile.setGender(StringUtil.objectToString(po.get(GENDER)));
        missionHealthProfile.setDateOfBirth(DateHelper.parseDate(StringUtil.objectToString(po.get(DATE_OF_BIRTH))));
        missionHealthProfile.setReminder(StringUtil.objectToString(po.get(REMINDER)));
        missionHealthProfile.setEmail(StringUtil.objectToString(po.get(EMAIL)));
        missionHealthProfile.setLastName(StringUtil.objectToString(po.get(LAST_NAME)));
		
		return missionHealthProfile;
	}

	public MissionHealthDetail translateToMissionHealthDetail(DBObject po) {
		MissionHealthDetail missionHealthDetail = new MissionHealthDetail();
		
		missionHealthDetail.setId(StringUtil.objectToString(po.get(MISSION_DETAIl_ID_KEY)));
		missionHealthDetail.setPatientId(StringUtil.objectToString(po.get(PATIENT_ID)));
		if (po.get(POINTS) != null) {
			missionHealthDetail.setPoints(Integer.valueOf(StringUtil.objectToString(po.get(POINTS))));
		}
		if (po.get(PROFILE_POINTS) != null) {
			missionHealthDetail.setMissionProfilePoints(Integer.valueOf(StringUtil.objectToString(po.get(PROFILE_POINTS))));
		}
		missionHealthDetail.setSmartInfoDate(DateHelper.parseDate(StringUtil.objectToString(po.get(SMART_INFO_DATE))));
		missionHealthDetail.setRank(translateToMissionHealthRank((BasicDBObject)po.get(RANK)));
        missionHealthDetail.setPendingRank(translateToMissionHealthRank((BasicDBObject) po.get(PENDING_RANK)));
        missionHealthDetail.setPreviousRank(translateToMissionHealthRank((BasicDBObject) po.get(PREVIOUS_RANK)));
        missionHealthDetail.setRankEffectiveDate(DateHelper.parseDate(StringUtil.objectToString(po.get(RANK_EFFECTIVE_DATE))));
        missionHealthDetail.setPromoted(Boolean.parseBoolean(StringUtil.objectToString(po.get(PROMOTED))));
        missionHealthDetail.setTroops(Integer.valueOf(StringUtil.objectToString(po.get(TROOPS))));
        missionHealthDetail.setLostTroopsShowMessage(Boolean.parseBoolean((StringUtil.objectToString(po.get(LOST_TROOPS_SHOW_MESSAGE)))));
        missionHealthDetail.setFailedMissionCounter(Integer.valueOf(StringUtil.objectToString(po.get(FAILED_MISSION_COUNTER))));
		missionHealthDetail.setLoginDate(DateHelper.parseDate(StringUtil.objectToString(po.get(LOGIN_DATE))));
        missionHealthDetail.setTotalLoginDaysForCycle(Integer.valueOf(StringUtil.objectToString(po.get(TOTAL_LOGIN_DAYS_FOR_CYCLE))));
        missionHealthDetail.setTotalLoginsForCycle(Integer.valueOf(StringUtil.objectToString(po.get(TOTAL_LOGINS_FOR_CYCLE))));
        missionHealthDetail.setLastSessionStart(DateHelper.parseDateTimeWithMilliseconds(StringUtil.objectToString(po.get(LAST_SESSION))));
        missionHealthDetail.setLastMissionCreateDate(DateHelper.parseVistaDateTime(StringUtil.objectToString(po.get(LAST_MISSION_CREATE_DATE))));
        if(po.get(INACTIVE_CYCLES) != null){
            missionHealthDetail.setInactiveCycles(Integer.valueOf(StringUtil.objectToString(po.get(INACTIVE_CYCLES))));
        }
        if(po.get(TOTAL_MISSIONS) != null){
            missionHealthDetail.setTotalMissions(Integer.valueOf(StringUtil.objectToString(po.get(TOTAL_MISSIONS))));
        }
        if(po.get(TOTAL_COMPLETED_MISSIONS) != null){
            missionHealthDetail.setTotalCompletedMissions(Integer.valueOf(StringUtil.objectToString(po.get(TOTAL_COMPLETED_MISSIONS))));
        }
        if(po.get(TOTAL_ATTEMPTED_MISSIONS) != null){
            missionHealthDetail.setTotalAttemptedMissions(Integer.valueOf(StringUtil.objectToString(po.get(TOTAL_ATTEMPTED_MISSIONS))));
        }
        if(po.get(ANNUAL_TOTAL_COMPLETED_MISSIONS) != null){
            missionHealthDetail.setAnnualTotalCompletedMissions(Integer.valueOf(StringUtil.objectToString(po.get(ANNUAL_TOTAL_COMPLETED_MISSIONS))));
        }
        if(po.get(ANNUAL_TOTAL_POINTS) != null){
            missionHealthDetail.setAnnualTotalPoints(Integer.valueOf(StringUtil.objectToString(po.get(ANNUAL_TOTAL_POINTS))));
        }
        if(po.get(ANNUAL_TOTAL_TROOPS) != null){
            missionHealthDetail.setAnnualTotalTroops(Integer.valueOf(StringUtil.objectToString(po.get(ANNUAL_TOTAL_TROOPS))));
        }
		if (po.get(MISSION_HEALTH_PROFILE) != null) {
			BasicDBObject profilePo = (BasicDBObject) po.get(MISSION_HEALTH_PROFILE);
			missionHealthDetail.setMissionHealthProfile(translateToMissionHealthProfile(profilePo));
		}
		if (po.get(MISSIONS) != null) {
			List<Mission> missions = populateMissions((BasicDBList) po.get(MISSIONS));
			missionHealthDetail.setMissions(missions);
		}
		if (po.get(MISSION_READINESS_RPTS) != null) {
			List<MissionReadinessReport> missionReadinessReports = populateMissionReadinessReports((BasicDBList) po.get(MISSION_READINESS_RPTS));
			missionHealthDetail.setMissionReadinessReports(missionReadinessReports);
		}
        if (po.get(CURRENT_MISSIONS) != null) {
            List<Mission> missions = populateMissions((BasicDBList) po.get(CURRENT_MISSIONS));
            missionHealthDetail.setCurrentMissions(missions);
        }
        if (po.get(DEVICE_ID) != null) {
            missionHealthDetail.setDeviceId(StringUtil.objectToString(po.get(DEVICE_ID)));
        }
		return missionHealthDetail;
	}

	private List<Mission> populateMissions(BasicDBList missionList) {
		List<Mission> missions = null;

		if (missionList != null) {
			missions = new ArrayList<Mission>();

			Iterator<Object> dbMissions = missionList.iterator();
			while (dbMissions.hasNext()) {
			    BasicDBObject dbMission = (BasicDBObject) dbMissions.next();
			    missions.add(translateToMission(dbMission));
			}
		}
		return missions;
	}

    protected List<MissionReadinessReport> populateMissionReadinessReports(BasicDBList missionReadinessReportList) {
		List<MissionReadinessReport> missionReadinessReports = null;

		if (missionReadinessReportList != null) {
			missionReadinessReports = new ArrayList<MissionReadinessReport>();

			Iterator<Object> dbMissionReadinessReports = missionReadinessReportList.iterator();
			while (dbMissionReadinessReports.hasNext()) {
			    BasicDBObject dbMissionReadinessReport = (BasicDBObject) dbMissionReadinessReports.next();
			    missionReadinessReports.add(translateMissionReadinessReport(dbMissionReadinessReport));
			}
		}
		return missionReadinessReports;
	}
	
    private Mission translateToMission(DBObject po){
        Mission mission = new Mission();

        mission.setId(StringUtil.objectToString(po.get(MISSION_ID_KEY)));
        mission.setName(StringUtil.objectToString(po.get(NAME)));
        mission.setDescription(StringUtil.objectToString(po.get(DESCRIPTION)));
        mission.setStatus(MissionStatus.fromValue((StringUtil.objectToString(po.get(STATUS)))));
        mission.setStartDate(DateHelper.parseVistaDate(StringUtil.objectToString(po.get(START_DATE))));
        mission.setThresholdPercentage(StringUtil.objectToString(po.get(THRESHOLD_PERCENTAGE)));
        if(null!=po.get(PERCENTAGE_COMPLETED))
            mission.setPercentageCompleted(StringUtil.objectToString(po.get(PERCENTAGE_COMPLETED)));
        mission.setTotalPoints(Integer.parseInt(StringUtil.objectToString(po.get(TOTAL_POINTS))));
        if(null!=po.get(POINTS_WON))
            mission.setPointsWon(Integer.parseInt(StringUtil.objectToString(po.get(POINTS_WON))));
        if(null!=po.get(POINTS_FORFEITED))
            mission.setPointsForfeited(Integer.parseInt(StringUtil.objectToString(po.get(POINTS_FORFEITED))));
        mission.setHomeService(StringUtil.objectToString(po.get(HOME_SERVICE)));
        mission.setCompetingService(StringUtil.objectToString(po.get(COMPETING_SERVICE)));
        mission.setShowMessage(Boolean.valueOf(StringUtil.objectToString(po.get(SHOW_MESSAGE))));
        if(null!=po.get(TROOPS_FORFEITED))
            mission.setTroopsForfeited(Integer.parseInt(StringUtil.objectToString(po.get(TROOPS_FORFEITED))));
        return mission;
    }
    
    private MissionReadinessReport translateMissionReadinessReport(DBObject po){
    	MissionReadinessReport missionReadinessReport = new MissionReadinessReport();

    	missionReadinessReport.setHealth(Integer.parseInt(StringUtil.objectToString(po.get(MISSION_READINESS_RPT_HEALTH))));
    	missionReadinessReport.setPhysical(Integer.parseInt(StringUtil.objectToString(po.get(MISSION_READINESS_RPT_PHYSICAL))));
    	missionReadinessReport.setEmotional(Integer.parseInt(StringUtil.objectToString(po.get(MISSION_READINESS_RPT_EMOTIONAL))));
    	missionReadinessReport.setInsDate(DateHelper.parseDate(StringUtil.objectToString(po.get(MISSION_READINESS_RPT_INSDATE))));
        return missionReadinessReport;
    }

    public DBObject translatePointsRule(PointsRule pointsRule) {
        DBObject dbPointsRule = new BasicDBObject();

        dbPointsRule.put(MONGO_ID, pointsRule.getId());
        dbPointsRule.put(MISSION_THRESHOLDS, translateThresholdValues(pointsRule.getMissionThresholds()));
        dbPointsRule.put(POINT_THRESHOLDS, translateThresholdValues(pointsRule.getPointThresholds()));
        dbPointsRule.put(MISSIONS_TO_ACHIEVE_LEVEL, pointsRule.getMissionsToAchieveLevel());
        dbPointsRule.put(POINTS_FOR_NEW_TROOP, pointsRule.getPointsForNewTroop());
        dbPointsRule.put(TROOPS_ALLOCATED, pointsRule.getTroopsAllocated());
        return dbPointsRule;
    }

    public DBObject translateMissionHealthRank(MissionHealthRank missionHealthRank){
        if(null != missionHealthRank){
            DBObject dbMissionHealthRank = new BasicDBObject();
            dbMissionHealthRank.put(LEVEL, missionHealthRank.getLevel());
            if(missionHealthRank.getPayGrade() != null){
                dbMissionHealthRank.put(PAY_GRADE, missionHealthRank.getPayGrade());
            }
            if(missionHealthRank.getService() != null){
                dbMissionHealthRank.put(SERVICE, missionHealthRank.getService());
            }
            if(missionHealthRank.getAbbr() != null){
                dbMissionHealthRank.put(ABBR, missionHealthRank.getAbbr());
            }
            if(missionHealthRank.getTitle() != null){
                dbMissionHealthRank.put(TITLE, missionHealthRank.getTitle());
            }
            if(missionHealthRank.getShortTitle() != null){
                dbMissionHealthRank.put(SHORT_TITLE, missionHealthRank.getShortTitle());
            }
            return dbMissionHealthRank;
        }
        return null;
    }

    private DBObject translateThresholdValues(ThresholdValues thresholdValues){
        DBObject dbThresholdValues = new BasicDBObject();

        dbThresholdValues.put(LOWER, thresholdValues.getLower());
        dbThresholdValues.put(UPPER, thresholdValues.getUpper());
        dbThresholdValues.put(INTERVAL, thresholdValues.getInterval());
        return dbThresholdValues;
    }

    public PointsRule translateToPointsRule(DBObject po) {
        PointsRule pointsRule = new PointsRule();

        pointsRule.setId(Integer.valueOf(StringUtil.objectToString(po.get(MONGO_ID))));
        pointsRule.setMissionThresholds(translateToThresholdValues((DBObject) po.get(MISSION_THRESHOLDS)));
        pointsRule.setPointThresholds(translateToThresholdValues((DBObject) po.get(POINT_THRESHOLDS)));
        pointsRule.setMissionsToAchieveLevel(Integer.valueOf(StringUtil.objectToString(po.get(MISSIONS_TO_ACHIEVE_LEVEL))));
        pointsRule.setPointsForNewTroop(Integer.valueOf(StringUtil.objectToString(po.get(POINTS_FOR_NEW_TROOP))));
        pointsRule.setTroopsAllocated(Integer.valueOf(StringUtil.objectToString(po.get(TROOPS_ALLOCATED))));
        return pointsRule;
    }

    private ThresholdValues translateToThresholdValues(DBObject po){
        ThresholdValues thresholdValues = new ThresholdValues();

        thresholdValues.setLower(Integer.parseInt(StringUtil.objectToString(po.get(LOWER))));
        thresholdValues.setUpper(Integer.parseInt(StringUtil.objectToString(po.get(UPPER))));
        thresholdValues.setInterval(Integer.parseInt(StringUtil.objectToString(po.get(INTERVAL))));
        return thresholdValues;
    }

    public MissionHealthRank translateToMissionHealthRank(DBObject po){
        if(null != po) {
            MissionHealthRank missionHealthRank = new MissionHealthRank();
            missionHealthRank.setLevel(Integer.valueOf(StringUtil.objectToString(po.get(LEVEL))));
            if(po.get(PAY_GRADE) != null){
                missionHealthRank.setPayGrade(StringUtil.objectToString(po.get(PAY_GRADE)));
            }
            if(po.get(SERVICE) != null){
                missionHealthRank.setService(StringUtil.objectToString(po.get(SERVICE)));
            }
            if(po.get(ABBR) != null){
                missionHealthRank.setAbbr(StringUtil.objectToString(po.get(ABBR)));
            }
            if(po.get(TITLE) != null){
                missionHealthRank.setTitle(StringUtil.objectToString(po.get(TITLE)));
            }
            if(po.get(SHORT_TITLE) != null){
                missionHealthRank.setShortTitle(StringUtil.objectToString(po.get(SHORT_TITLE)));
            }else{
                if(po.get(TITLE) != null){
                    missionHealthRank.setShortTitle(StringUtil.objectToString(po.get(TITLE)));
                }
            }
            return missionHealthRank;
        }
        return null;
    }

    public MissionHealthStanding translateToMissionHealthStanding1(DBObject po, StandingType standingType){
        if(null!= po){
            MissionHealthStanding standing = new MissionHealthStanding();
            standing.setPatientId(StringUtil.objectToString(po.get(MONGO_ID)));
            standing.setName(StringUtil.objectToString(po.get(NAME)));
            standing.setPoints(Integer.valueOf(StringUtil.objectToString(po.get(POINTS))));
            standing.setLevel(Integer.valueOf(StringUtil.objectToString(po.get(LEVEL))));
            standing.setService(StringUtil.objectToString(po.get(SERVICE)));
            standing.setTroops(Integer.valueOf(StringUtil.objectToString(po.get(TROOPS))));
            standing.setType(standingType);
            if(standingType.equals(StandingType.LEVEL)){
                standing.setStanding(Integer.valueOf(StringUtil.objectToString(po.get(VS_LEVEL))));
            }else if(standingType.equals(StandingType.SERVICE)){
                standing.setStanding(Integer.valueOf(StringUtil.objectToString(po.get(VS_SERVICE))));
            }else{
                standing.setStanding(Integer.valueOf(StringUtil.objectToString(po.get(VS_ALL))));
            }
            return standing;
        }else{
            return null;
        }
    }

    public MissionHealthStanding translateToMissionHealthStanding(DBObject po, StandingType standingType){
        if(null!= po){
            MissionHealthStanding standing = new MissionHealthStanding();
            standing.setPatientId(StringUtil.objectToString(po.get(MONGO_ID)));
            standing.setName(StringUtil.objectToString(po.get(NAME)));
            standing.setPoints(Integer.valueOf(StringUtil.objectToString(po.get(POINTS))));
            standing.setLevel(Integer.valueOf(StringUtil.objectToString(po.get(LEVEL))));
            standing.setService(StringUtil.objectToString(po.get(SERVICE)));
            standing.setTroops(Integer.valueOf(StringUtil.objectToString(po.get(TROOPS))));
            standing.setType(standingType);
            if(standingType.equals(StandingType.LEVEL)){
                standing.setStanding(Integer.valueOf(StringUtil.objectToString(((DBObject) po.get(VS_LEVEL)).get("standing"))));
            }else if(standingType.equals(StandingType.SERVICE)){
                standing.setStanding(Integer.valueOf(StringUtil.objectToString(((DBObject) po.get(VS_SERVICE)).get("standing"))));
            }else{
                standing.setStanding(Integer.valueOf(StringUtil.objectToString(((DBObject)po.get(VS_ALL)).get("standing"))));
            }
            return standing;
        }else{
            return null;
        }
    }
}
