package com.agilex.vamf.missionhealth.datalayer;


import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import com.agilex.vamf.missionhealth.domain.*;
import com.agilex.vamf.missionhealth.domain.comparator.ServiceAgeComparator;
import com.agilex.vamf.missionhealth.domain.WeeklyServiceMetrics;

import org.junit.*;

import com.agilex.vamf.utils.DateHelper;
import com.agilex.vamf.missionhealth.domain.enumeration.MissionStatus;
import com.agilex.vamf.missionhealth.service.PointService;

public class MissionHealthDataServiceTest extends SpringBasedIntegrationRepo {
   	
	@Resource
	MissionHealthDataService missionHealthDataService;
	
	@Resource
	PointService pointService;

    private static final String missionhealthdb = "missionhealthdb";
    private static final String missionhealthdetails = "missionhealthdetails";

    private static final String Air_Force = "Air Force";
    private static final String Army = "Army";
    private static final String Coast_Guard = "Coast Guard";
    private static final String Marines = "Marines";
    private static final String Navy = "Navy";

    private PatientIdentifier lazyPatientId;
    private PatientIdentifier activePatientId;
    private PatientIdentifier otherPatientId;

    @Before
    public void clearPreviousTestData(){
        lazyPatientId = new PatientIdentifier("EDIPI", "lazyPatient");
        activePatientId = new PatientIdentifier("EDIPI", "activePatient");
        otherPatientId = new PatientIdentifier("EDIPI", "otherPatient");
    }

    @Test
    public void saveMissionHealthProfileTest(){
        PatientIdentifier saveProfileId = new PatientIdentifier("EDIPI", "saveProfileTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(saveProfileId));
        MissionHealthDetail missionHealthDetail = missionHealthDataService.getMissionHealthDetail(saveProfileId);

        saveMissionHealthDetail("saveProfileTestPatient", "saveProfileTestPatientName");

        missionHealthDetail = missionHealthDataService.getMissionHealthDetail(saveProfileId);
        MissionHealthProfile profile = missionHealthDetail.getMissionHealthProfile();
        profile.setPatientIdentifier(saveProfileId);
        profile.setName("saveProfileName");

        missionHealthDataService.saveMissionHealthProfile(profile);
        missionHealthDetail = missionHealthDataService.getMissionHealthDetail(saveProfileId);
        Assert.assertNotNull(missionHealthDetail.getMissionHealthProfile());
        Assert.assertEquals(missionHealthDetail.getMissionHealthProfile().getName(), profile.getName());
    }

    @Test
    public void updateMissionHealthProfileTest(){
        PatientIdentifier updateProfileId = new PatientIdentifier("EDIPI", "updateProfileTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(updateProfileId));
        MissionHealthDetail missionHealthDetail = missionHealthDataService.getMissionHealthDetail(updateProfileId);

        MissionHealthProfile profile = new MissionHealthProfile();
        profile.setPatientIdentifier(updateProfileId);
        profile.setName("updateProfileName");

        missionHealthDataService.saveMissionHealthProfile(profile);
        missionHealthDetail = missionHealthDataService.getMissionHealthDetail(updateProfileId);
        Assert.assertNotNull(missionHealthDetail.getMissionHealthProfile());

        profile.setService(Navy);
        profile.setReminder("None");
        profile.setEmail("email@address.com");

        missionHealthDataService.updateMissionHealthProfile(profile);
        missionHealthDetail = missionHealthDataService.getMissionHealthDetail(updateProfileId);
        Assert.assertNotNull(missionHealthDetail.getMissionHealthProfile());
        Assert.assertEquals(missionHealthDetail.getMissionHealthProfile().getName(), profile.getName());
        Assert.assertEquals(missionHealthDetail.getMissionHealthProfile().getService(), profile.getService());
        Assert.assertEquals(missionHealthDetail.getMissionHealthProfile().getReminder(), profile.getReminder());
        Assert.assertEquals(missionHealthDetail.getMissionHealthProfile().getEmail(), profile.getEmail());
    }

    @Test
    public void getMissionHealthDetailTest(){
        PatientIdentifier getDetailId = new PatientIdentifier("EDIPI", "getDetailTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(getDetailId));
        MissionHealthDetail missionHealthDetail = missionHealthDataService.getMissionHealthDetail(getDetailId, new Date());
        Assert.assertNotNull(missionHealthDetail);
        Assert.assertNotNull(missionHealthDetail.getId());
        Assert.assertNotNull(missionHealthDetail.getPatientId());
        Assert.assertNotNull(missionHealthDetail.getRank());
        Assert.assertNotNull(missionHealthDetail.getRankEffectiveDate());
        Assert.assertNotNull(missionHealthDetail.getMissionProfilePoints());
        Assert.assertEquals(new Integer(10), missionHealthDetail.getPoints());
        Assert.assertEquals(new Integer(0), missionHealthDetail.getTroops());
        Assert.assertEquals(new Integer(0), missionHealthDetail.getTotalMissions());
        Assert.assertEquals(new Integer(0), missionHealthDetail.getTotalCompletedMissions());
        Assert.assertEquals(new Integer(0), missionHealthDetail.getMissionProfilePoints());
        Assert.assertFalse(missionHealthDetail.isPromoted());
        Assert.assertNotNull(missionHealthDetail.getTotalLoginsForCycle());
        Assert.assertNotNull(missionHealthDetail.getTotalLoginDaysForCycle());
        Assert.assertNotNull(missionHealthDetail.getLastSessionStart());

        Integer logins = missionHealthDetail.getTotalLoginsForCycle();
        Date date = new Date();
        missionHealthDetail = missionHealthDataService.getMissionHealthDetail(getDetailId, date);
        Assert.assertTrue(missionHealthDetail.getTotalLoginsForCycle().equals(logins+1));
        Assert.assertEquals(date, missionHealthDetail.getLastSessionStart());
    }

    @Test
    public void updateMissionHealthDetailTest(){
        PatientIdentifier updateDetailId = new PatientIdentifier("EDIPI", "updateDetailTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(updateDetailId));
        MissionHealthDetail missionHealthDetail = missionHealthDataService.getMissionHealthDetail(updateDetailId);
        Assert.assertNotNull(missionHealthDetail);
        missionHealthDetail.setRank(new MissionHealthRank(1));
        Date date = DateHelper.beginOfDate(new Date());
        missionHealthDetail.setRankEffectiveDate(date);
        missionHealthDetail.setPromoted(true);
        missionHealthDetail.setTotalCompletedMissions(missionHealthDetail.getTotalCompletedMissions()+1);
        Integer currPoints = missionHealthDetail.getPoints();
        missionHealthDetail.setPoints(currPoints+20);
        missionHealthDataService.updateMissionHealthDetail(missionHealthDetail);

        missionHealthDetail = missionHealthDataService.getMissionHealthDetail(updateDetailId);
        Assert.assertEquals(missionHealthDetail.getId(), missionHealthDetail.getId());
        Assert.assertEquals(new MissionHealthRank(1), missionHealthDetail.getRank());
        Assert.assertEquals(date, missionHealthDetail.getRankEffectiveDate());
        Assert.assertTrue(missionHealthDetail.isPromoted());
        Assert.assertEquals(new Integer(1), missionHealthDetail.getTotalCompletedMissions());
        Assert.assertEquals(new Integer(currPoints+20), missionHealthDetail.getPoints());
    }

    @Test
    public void removeMissionHealthDetailTest(){
        PatientIdentifier removeDetailId = new PatientIdentifier("EDIPI", "removeDetailTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(removeDetailId));
        MissionHealthDetail missionHealthDetail = missionHealthDataService.getMissionHealthDetail(removeDetailId);
        Assert.assertNull(missionHealthDetail.getMissionHealthProfile());
    }
	
	 @Test
	 public void loggingPointsTest() {
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "loggingPointsTestPatient");
        MissionHealthDetail  mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		missionHealthDataService.removeMissionHealthDetail(mhd);
		
		saveMissionHealthDetail("loggingPointsTestPatient", "loggingPointsTestPatientName");
		
		mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		Integer prevPts = mhd.getPoints();
		Assert.assertEquals(new Integer(10), prevPts);
		Assert.assertEquals(new Integer(10), mhd.getAnnualTotalPoints());
		
		pointService.updatePointsForLogging(mhd);
		MissionHealthDetail updatedMhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        Integer updatedPts = updatedMhd.getPoints();
		Assert.assertEquals(prevPts, updatedPts);
		Assert.assertEquals(prevPts, updatedMhd.getAnnualTotalPoints());
		
		updatedMhd.setLoginDate(DateHelper.get30DaysFromNow());
		updatedMhd = missionHealthDataService.updateMissionHealthDetail(updatedMhd);
		Integer updatedPoints = updatedMhd.getPoints();
		Assert.assertEquals(new Integer(20), updatedPoints);
		mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		Assert.assertEquals(updatedPoints, mhd.getPoints());
		Assert.assertEquals(updatedPoints, updatedMhd.getAnnualTotalPoints());
	}
	
	@Test
	 public void updateSmartInfoPointsTest() {
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "updateSmartInfoPointsTestPatient");
		MissionHealthDetail mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		missionHealthDataService.removeMissionHealthDetail(mhd);
		mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        
		saveMissionHealthDetail("updateSmartInfoPointsTestPatient", "updateSmartInfoPointsTestPatientName");
		
		//Since new profile is created, smartInfoDate will be null and smartinfo points will be updated
		mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		Integer prevPts=mhd.getPoints();
		MissionHealthPartialDetail mhpd=missionHealthDataService.updateSmartInfoPoints(patientIdentifier);
		Assert.assertTrue(prevPts<mhpd.getPoints());
		mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		Assert.assertEquals(mhpd.getPoints(),mhd.getAnnualTotalPoints());
		
		//Try to update again for the same date and smartinfo points wont be updated
		prevPts=mhd.getPoints();
		mhpd=missionHealthDataService.updateSmartInfoPoints(patientIdentifier);
		Assert.assertEquals(prevPts, mhpd.getPoints());
		mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		Assert.assertEquals(mhpd.getPoints(),mhd.getAnnualTotalPoints());
		
		//Changing smartinfo date so that smart info points update can be tested
		mhd.setSmartInfoDate(DateHelper.get30DaysFromNow());
		mhd = missionHealthDataService.updateMissionHealthDetail(mhd);
		prevPts=mhd.getPoints();
		mhpd=missionHealthDataService.updateSmartInfoPoints(patientIdentifier);
		Assert.assertTrue(prevPts<mhpd.getPoints());
        mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        Assert.assertEquals(mhpd.getPoints(),mhd.getAnnualTotalPoints());
        missionHealthDataService.removeMissionHealthDetail(mhd);
	 }

	 private void saveMissionHealthDetail(String patientId, String name) {

			MissionHealthDetail missionHealthDetail = new MissionHealthDetail();
			missionHealthDetail.setPatientId(patientId);
			missionHealthDetail.setMissionHealthProfile(generateMissionHealthProfile(patientId, name));
			missionHealthDetail.setLoginDate(new Date());
			missionHealthDetail.setPoints(0);
			missionHealthDetail.setRank(new MissionHealthRank(1));
			missionHealthDetail.setTroops(2);

			missionHealthDataService.updateMissionHealthDetail(missionHealthDetail);
	 }
	    
	 private MissionHealthProfile generateMissionHealthProfile(String patientId, String name) {
	    	MissionHealthProfile missionHealthProfile = new MissionHealthProfile();
			missionHealthProfile.setPatientId(patientId);
			missionHealthProfile.setName(name);
			missionHealthProfile.setDateOfBirth(DateHelper.parseDate("08/08/1984"));
			missionHealthProfile.setGender("M");
			missionHealthProfile.setReminder("Weekly");
			missionHealthProfile.setEmail(name+"@agilexhealth.com");
			missionHealthProfile.setService("Navy");
			
			return missionHealthProfile;
	    }
	    
	@Test
	 public void updateEmailFacebookPointsTest() {
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "updateEmailFacebookPointsTestPatient");
		MissionHealthDetail  mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		missionHealthDataService.removeMissionHealthDetail(mhd);
	
		saveMissionHealthDetail("updateEmailFacebookPointsTestPatient", "updateEmailFacebookPointsTestPatientName");
		
		mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		int prevPts=mhd.getPoints()==null?0:mhd.getPoints();
		MissionHealthPartialDetail mhdSubset=missionHealthDataService.updateEmailFacebookPoints(patientIdentifier);
		Assert.assertEquals((int)mhdSubset.getPoints(),prevPts+10);
		mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
		Assert.assertEquals(mhdSubset.getPoints(),mhd.getAnnualTotalPoints());
        missionHealthDataService.removeMissionHealthDetail(mhd);
	}

    @Ignore
    @Test
    public void createNewMissionTest() {
        PatientIdentifier createMissionId = new PatientIdentifier("EDIPI", "createNewMissionTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(createMissionId));
        MissionHealthDetail mhd=missionHealthDataService.getMissionHealthDetail(createMissionId);

        saveMissionHealthDetail("createNewMissionTestPatient", "createNewMissionTestPatientName");

        mhd=missionHealthDataService.getMissionHealthDetail(createMissionId);
        Assert.assertEquals("Mission health detail created with rank=1", mhd.getRank(), new MissionHealthRank(1));

        Mission mission = new Mission();
        mission.setName("Build the mission-health app within this PSI");
        mission.setDescription("Build the mission-health app");
        mission.setCompetingService(Coast_Guard);
        mission.setStatus(MissionStatus.ACTIVE);

        mission = missionHealthDataService.createNewMission(createMissionId, mission);
        Assert.assertNotNull(mission);
        Assert.assertNotNull(mission.getTotalPoints());
        Assert.assertNotNull(mission.getId());
        Assert.assertNotNull(mission.getHomeService());
        Integer missionTotalPoints = mission.getTotalPoints();
        Assert.assertTrue("After creating a mission, the total points value is a multiple of 5 between 25 and 50",
                missionTotalPoints>=25 && missionTotalPoints<=50 && missionTotalPoints%5==0);
        Assert.assertNotNull(mission.getPointsRange());
        Assert.assertTrue(mission.getPointsRange().contains(Integer.valueOf(mission.getTotalPoints())));
    }

    @Ignore
    @Test
    public void updateCurrentMissionTest(){
        PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "updateCurrentMissionTestPatient");
        missionHealthDataService.removeMissionHealthDetail(missionHealthDataService.getMissionHealthDetail(patientIdentifier));
        MissionHealthDetail mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);

        saveMissionHealthDetail("updateCurrentMissionTestPatient", "updateCurrentMissionTestPatientName");
        mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        Mission mission = new Mission();
        mission.setName("Build the mission-health app within this PSI");
        mission.setDescription("Build the mission-health app");
        mission.setCompetingService(Coast_Guard);
        mission.setStatus(MissionStatus.ACTIVE);

        mission = missionHealthDataService.createNewMission(patientIdentifier, mission);
        mission.setPercentageCompleted("10");
        missionHealthDataService.updateMission(patientIdentifier, mission);

        mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        Assert.assertTrue("Percent completed updated successfully", "10".equals(mhd.getCurrentMissions().get(0).getPercentageCompleted()));

        Mission nonExistantMission = new Mission();
        mission.setId("FakeId");

        nonExistantMission = missionHealthDataService.updateMission(patientIdentifier, nonExistantMission);
        Assert.assertNull(nonExistantMission);
    }
    
    @Test
    public void insertMissionReadinessReportTest() {
        PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "insertMissionReadinessReportTestPatient");
        MissionHealthDetail  mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        missionHealthDataService.removeMissionHealthDetail(mhd);
        mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        
        saveMissionHealthDetail("insertMissionReadinessReportTestPatient", "insertMissionReadinessReportTestPatientName");
        
        mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        
        //Setting rank to 0 to test logic for rank 0 updates
        mhd.setRank(new MissionHealthRank(0));
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
        MissionReadinessReport mrr=new MissionReadinessReport();
        mrr.setEmotional(5);
        mrr.setHealth(10);
        mrr.setPhysical(15);
        mrr.setInsDate(DateHelper.getYesterday());
    	MissionHealthDetail partialmhd=missionHealthDataService.insertMissionReadinessReport(patientIdentifier,mrr);
    	Assert.assertNotNull(partialmhd);
    	Assert.assertEquals("Submitting mission readiness report at rank 0 should not update rank",new MissionHealthRank(0), partialmhd.getRank());
    	Assert.assertEquals(false, partialmhd.isPromoted());
    	
    	mhd.setRank(new MissionHealthRank(1));
    	mhd.setPreviousRank(new MissionHealthRank(0));
    	mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
    	mrr.setEmotional(21);
        mrr.setHealth(22);
        mrr.setPhysical(23);
        mrr.setInsDate(DateHelper.getToday());
        partialmhd=missionHealthDataService.insertMissionReadinessReport(patientIdentifier,mrr);
    	Assert.assertNull(partialmhd);
    	
    	mhd.setPreviousRank(new MissionHealthRank(1));
    	mhd.setRank(new MissionHealthRank(2));
    	mhd.setPendingRank(new MissionHealthRank(3));
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
    	mrr.setEmotional(20);
        mrr.setHealth(25);
        mrr.setPhysical(30);
        mrr.setInsDate(DateHelper.getToday());
        partialmhd=missionHealthDataService.insertMissionReadinessReport(patientIdentifier,mrr);
    	Assert.assertEquals("Previous Rank", new MissionHealthRank(2),partialmhd.getPreviousRank());
    	Assert.assertEquals("Current Rank", new MissionHealthRank(3),partialmhd.getRank());
    	Assert.assertNull("Pending Rank", partialmhd.getPendingRank());
    	
    	mhd.setPreviousRank(new MissionHealthRank(4));
    	mhd.setRank(new MissionHealthRank(5));
    	mhd.setPendingRank(new MissionHealthRank(6));
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
    	mrr.setEmotional(35);
        mrr.setHealth(40);
        mrr.setPhysical(45);
        mrr.setInsDate(DateHelper.getTomorrow());
        partialmhd=missionHealthDataService.insertMissionReadinessReport(patientIdentifier,mrr);
    	Assert.assertEquals("Previous Rank", new MissionHealthRank(5),partialmhd.getPreviousRank());
    	Assert.assertEquals("Current Rank", new MissionHealthRank(6),partialmhd.getRank());
    	Assert.assertNull("Pending Rank", partialmhd.getPendingRank());
    	
    	mhd.setPreviousRank(new MissionHealthRank(7));
    	mhd.setRank(new MissionHealthRank(8));
    	mhd.setPendingRank(new MissionHealthRank(9));
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
    	mrr.setEmotional(50);
        mrr.setHealth(55);
        mrr.setPhysical(65);
        mrr.setInsDate(DateHelper.get30DaysFromNow());
        partialmhd=missionHealthDataService.insertMissionReadinessReport(patientIdentifier,mrr);
    	Assert.assertEquals("Previous Rank", new MissionHealthRank(8),partialmhd.getPreviousRank());
    	Assert.assertEquals("Current Rank", new MissionHealthRank(9),partialmhd.getRank());
    	Assert.assertNull("Pending Rank", partialmhd.getPendingRank());
    	
    	mhd.setPreviousRank(new MissionHealthRank(10));
    	mhd.setRank(new MissionHealthRank(11));
    	mhd.setPendingRank(new MissionHealthRank(12));
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
    	mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
    	Integer prevPts=mhd.getPoints()+50;
    	mrr.setEmotional(70);
        mrr.setHealth(75);
        mrr.setPhysical(80);
        mrr.setInsDate(DateHelper.get120DaysFromNow());
    	partialmhd=missionHealthDataService.insertMissionReadinessReport(patientIdentifier,mrr);
    	Assert.assertNotNull(partialmhd);
    	Assert.assertEquals("Check 5th readiness report receives 50pts",prevPts, partialmhd.getPoints());
    	Assert.assertEquals(true, partialmhd.isPromoted());
    	Assert.assertEquals("Previous Rank", new MissionHealthRank(11),partialmhd.getPreviousRank());
    	Assert.assertEquals("Current Rank", new MissionHealthRank(12),partialmhd.getRank());
    	Assert.assertNull("Pending Rank",partialmhd.getPendingRank());
    	mhd=missionHealthDataService.getMissionHealthDetail(patientIdentifier);
    	Assert.assertEquals("Annual Points check",mhd.getPoints(),mhd.getAnnualTotalPoints());
    	Assert.assertNull(mhd.getPendingRank());
    }
    
    @Test
    public void savePromotedStatusTest() {
        PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "savePromotedStatusTestPatient");
        MissionHealthDetail  mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        missionHealthDataService.removeMissionHealthDetail(mhd);

        saveMissionHealthDetail("savePromotedStatusTestPatient","savePromotedStatusTestPatientName");
        mhd.setPromoted(true);
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
        boolean prevStatus=mhd.isPromoted();
        MissionHealthPartialDetail mhpd=missionHealthDataService.savePromotedStatus(patientIdentifier);
        Assert.assertNotNull("Check MissionHealthPartialDetail is not null",mhpd);
        Assert.assertFalse("Check promoted is updated correctly",mhpd.getPromoted());
        mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        boolean currentStatus=mhd.isPromoted();
        Assert.assertTrue("Check previous promoted flag from MissionHealthDetail",prevStatus);
        Assert.assertFalse("Check current promoted flag from MissionHealthDetail",currentStatus);
    }
    
    @Test
    public void saveLostTroopsShowMessage() {
        PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "saveLostTroopsTestPatient");
        MissionHealthDetail  mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        missionHealthDataService.removeMissionHealthDetail(mhd);

        saveMissionHealthDetail("saveLostTroopsTestPatient", "saveLostTroopsPatientName");
        mhd.setLostTroopsShowMessage(true);
        mhd=missionHealthDataService.updateMissionHealthDetail(mhd);
        boolean prevMsgStatus=mhd.isLostTroopsShowMessage();
        MissionHealthPartialDetail mhpd=missionHealthDataService.saveLostTroopsShowMessage(patientIdentifier);
        Assert.assertNotNull("Check MissionHealthPartialDetail is not null",mhpd);
        Assert.assertFalse("Check lostTroopsShowMessage is updated correctly",mhpd.getLostTroopsShowMessage());
        mhd = missionHealthDataService.getMissionHealthDetail(patientIdentifier);
        boolean currentMsgStatus=mhd.isLostTroopsShowMessage();
        Assert.assertTrue("Check previous LostTroopsShowMessgae flag from MissionHealthDetail",prevMsgStatus);
        Assert.assertFalse("Check current LostTroopsShowMessgae flag from MissionHealthDetail",currentMsgStatus);
     }
    
    @Test
    public void validateTieBreakerForAnnualServiceRollUp(){
    	WeeklyServiceMetrics wsm1= new WeeklyServiceMetrics();
    	wsm1.setService("Air Force");
    	wsm1.setTotalPoints(2);
    	wsm1.setTotalMissionsCompleted(2);
    	wsm1.setTotalTroops(2);
    	wsm1.setWinner(1);
    	WeeklyServiceMetrics wsm2= new WeeklyServiceMetrics();
    	wsm2.setService("Marines");
    	wsm2.setTotalPoints(2);
    	wsm2.setTotalMissionsCompleted(2);
    	wsm2.setTotalTroops(2);
    	wsm2.setWinner(2);
    	WeeklyServiceMetrics wsm3 = new WeeklyServiceMetrics();
    	wsm3.setService("Navy");
    	wsm3.setTotalPoints(2);
    	wsm3.setTotalMissionsCompleted(2);
    	wsm3.setTotalTroops(2);
    	wsm3.setWinner(3);
    	WeeklyServiceMetrics wsm4= new WeeklyServiceMetrics();
    	wsm4.setService("Coast Guard");
    	wsm4.setTotalPoints(2);
    	wsm4.setTotalMissionsCompleted(2);
    	wsm4.setTotalTroops(2);
    	wsm4.setWinner(4);
    	WeeklyServiceMetrics wsm5= new WeeklyServiceMetrics();
    	wsm5.setService("Army");
    	wsm5.setTotalPoints(2);
    	wsm5.setTotalMissionsCompleted(2);
    	wsm5.setTotalTroops(2);
    	wsm5.setWinner(5);
    	
		List<WeeklyServiceMetrics> wsmList=new ArrayList<WeeklyServiceMetrics>();
		wsmList.add(wsm1);
		wsmList.add(wsm2);
		wsmList.add(wsm3);
		wsmList.add(wsm4);
		wsmList.add(wsm5);
		
		List<WeeklyServiceMetrics> wsmListNew=missionHealthDataService.tieBreakerForAnnualServiceRollUp(wsmList);
		Collections.sort(wsmListNew,new ServiceAgeComparator());
		Assert.assertEquals("Check 1: Winner 1 1/5",wsmListNew.get(0).getWinner(),(Integer)1);
		Assert.assertEquals("Check 1: Winner Service 1 1/5",wsmListNew.get(0).getService(),"Army");
		Assert.assertEquals("Check 1: Winner 1 2/5",wsmListNew.get(1).getWinner(),(Integer)1);
		Assert.assertEquals("Check 1: Winner Service 1 2/5",wsmListNew.get(1).getService(),"Marines");
		Assert.assertEquals("Check 1: Winner 1 3/5",wsmListNew.get(2).getWinner(),(Integer)1);
		Assert.assertEquals("Check 1: Winner Service 1 3/5",wsmListNew.get(2).getService(),"Navy");
		Assert.assertEquals("Check 1: Winner 1 4/5",wsmListNew.get(3).getWinner(),(Integer)1);
		Assert.assertEquals("Check 1: Winner Service 1 4/5",wsmListNew.get(3).getService(),"Air Force");
		Assert.assertEquals("Check 1: Winner 1 5/5",wsmListNew.get(4).getWinner(),(Integer)1);
		Assert.assertEquals("Check 1: Winner Service 1 5/5",wsmListNew.get(4).getService(),"Coast Guard");
		
		wsm1.setTotalTroops(4);
		wsm1.setWinner(1);
		wsm2.setTotalTroops(4);
		wsm2.setWinner(2);
		wsm3.setTotalTroops(3);
		wsm3.setWinner(3);
		wsm4.setTotalTroops(2);
		wsm4.setWinner(4);
		wsm5.setTotalTroops(2);
		wsm5.setWinner(5);
		
		wsmList=new ArrayList<WeeklyServiceMetrics>();
		wsmList.add(wsm1);
		wsmList.add(wsm2);
		wsmList.add(wsm3);
		wsmList.add(wsm4);
		wsmList.add(wsm5);
		
		wsmListNew=missionHealthDataService.tieBreakerForAnnualServiceRollUp(wsmList);
		Collections.sort(wsmListNew,new ServiceAgeComparator());
		Assert.assertEquals("Check 2: Winner 1 1/5="+wsmList.get(0).getService(),(Integer)1,wsmList.get(0).getWinner());
		Assert.assertEquals("Check 2: Winner Service 1 1/5",wsmListNew.get(0).getService(),"Marines");
		Assert.assertEquals("Check 2: Winner 1 2/5="+wsmList.get(1).getService(),(Integer)1,wsmList.get(1).getWinner());
		Assert.assertEquals("Check 2: Winner Service 1 1/5",wsmListNew.get(1).getService(),"Air Force");
		Assert.assertEquals("Check 2: Winner 2 3/5="+wsmList.get(2).getService(),(Integer)3,wsmList.get(2).getWinner());
		Assert.assertEquals("Check 2: Winner Service 2 1/5",wsmListNew.get(2).getService(),"Navy");
		Assert.assertEquals("Check 2: Winner 3 4/5="+wsmList.get(3).getService(),(Integer)4,wsmList.get(3).getWinner());
		Assert.assertEquals("Check 2: Winner Service 3 1/5",wsmListNew.get(3).getService(),"Army");
		Assert.assertEquals("Check 2: Winner 3 5/5="+wsmList.get(4).getService(),(Integer)4,wsmList.get(4).getWinner());
		Assert.assertEquals("Check 2: Winner Service 3 1/5",wsmListNew.get(4).getService(),"Coast Guard");
		
    }
}
