package gov.va.med.mhv.sm.api.clinician.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.api.clinician.MessageAPIService;
import gov.va.med.mhv.sm.api.common.SessionAPIService;
import gov.va.med.mhv.sm.api.enumeration.ErrorEnum;
import gov.va.med.mhv.sm.api.transfer.AttachmentTO;
import gov.va.med.mhv.sm.api.transfer.AttachmentsTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianMessageTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianMessagesTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianStatusTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianStatusTypesTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianUserTO;
import gov.va.med.mhv.sm.api.transfer.CommentTO;
import gov.va.med.mhv.sm.api.transfer.CommentsTO;
import gov.va.med.mhv.sm.api.transfer.MessageCategoryTypesTO;
import gov.va.med.mhv.sm.api.transfer.MessageTO;
import gov.va.med.mhv.sm.api.transfer.Session;
import gov.va.med.mhv.sm.api.util.PropertiesHelper;
import gov.va.med.mhv.sm.api.util.SMApiUtility;
import gov.va.med.mhv.sm.enumeration.ClinicianStatusEnum;
import gov.va.med.mhv.sm.enumeration.MessageCategoryTypeEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.ReAssignMessageActionEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Annotation;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.DistributionGroup;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.MessageAttachment;
import gov.va.med.mhv.sm.model.NewMessage;
import gov.va.med.mhv.sm.model.ReAssignMessageHistory;
import gov.va.med.mhv.sm.model.SMClinicsTriageMap;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.Thread;
import gov.va.med.mhv.sm.model.TiuNotePreview;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.service.AuthenticationService;
import gov.va.med.mhv.sm.service.DistributionGroupService;
import gov.va.med.mhv.sm.service.MailboxService;
import gov.va.med.mhv.sm.service.MessageService;
import gov.va.med.mhv.sm.service.ReAssignMessageService;
import gov.va.med.mhv.sm.service.SendMessageService;
import gov.va.med.mhv.sm.service.ThreadService;
import gov.va.med.mhv.sm.service.TiuNoteService;
import gov.va.med.mhv.sm.service.TriageGroupService;
import gov.va.med.mhv.sm.service.UserManagementService;
import gov.va.med.mhv.sm.util.DateUtils;

import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.ws.rs.core.Response;

import org.apache.axis.utils.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.cxf.jaxrs.ext.MessageContext;
import org.apache.cxf.jaxrs.ext.multipart.Attachment;

public class MessageAPIServiceImpl implements MessageAPIService {
	private static final Log log = LogFactory.getLog(MessageAPIServiceImpl.class);

	private SessionAPIService sessionApiService;
	private MessageService messageService;
	private SendMessageService sendMessageService;
	private TriageGroupService triageGroupService;
	private MailboxService mailboxService;
	private ThreadService threadService;
	private UserManagementService userManagementService;
	private ReAssignMessageService reAssignMessageService;

	private DistributionGroupService distributionGroupService;
	private AuthenticationService authenticationService;
	private TiuNoteService tiuNoteService;

	private static Map<String,Boolean> supportedFileTypes = new HashMap<String, Boolean>();
	private static int FILE_SIZE_LIMIT_3MB = 3 * 1024 * 1024;
	private static List<MessageCategoryTypeEnum> messageCategoryTypeEnumList;
	private static List<ClinicianStatusEnum> clinicianStatusTypeEnumList;
	private static int EXPIRATION_PERIOD_IN_DAYS = -120;
	private static final String SM_WEB_PATIENT_REPLY_EXP_PARAM = "sm.web.patient.reply.expiration";
	private final static String DEFAULT_SM_CLINIC = "SECURE MESSAGING";
	private final static String DEFAULT_CPRS_NOTE_TITLE = "SECURE MESSAGING";

	@Resource
	MessageContext mc;

	static {
		if( PropertiesHelper.getProperties() != null && PropertiesHelper.getProperties().containsKey("sm.web.patient.fileupload.allowedTypes")) {
			String expStr = PropertiesHelper.getProperties().getProperty("sm.web.patient.fileupload.allowedTypes");
			String exts[] = expStr.split( ",\\s*" );
			for( String ext:exts ) {
				supportedFileTypes.put(ext.toUpperCase(), Boolean.TRUE);
			}
		}
		if( PropertiesHelper.getProperties() != null && PropertiesHelper.getProperties().containsKey(SM_WEB_PATIENT_REPLY_EXP_PARAM)) 
		{
			String expStr = PropertiesHelper.getProperties().getProperty(SM_WEB_PATIENT_REPLY_EXP_PARAM);
			try {
				int expInt = Integer.parseInt(expStr);
				if( expInt < 0 ) {
					EXPIRATION_PERIOD_IN_DAYS = expInt;
				}
			} catch (NumberFormatException e) {
				//do nothing
			}
		}

		messageCategoryTypeEnumList = MessageCategoryTypeEnum.toList();
		clinicianStatusTypeEnumList = ClinicianStatusEnum.toList();

	}

	private boolean isMessageExpired(Message currentMessage) {
		Date myDate = currentMessage.getCreatedDate();
		Calendar c = Calendar.getInstance();
		c.add(Calendar.DAY_OF_MONTH, EXPIRATION_PERIOD_IN_DAYS);
		Date expiration = c.getTime();

		if (expiration.compareTo(myDate) < 0) {
			//Not expired
			return false;
		}
		else {
			//Expired
			return true;
		}
	}

	/**
	 * Session Error Codes:  108, 109, 110, 111, 112, 115
	 * Error Codes:  904
	 */
	@Override
	public ClinicianMessageTO readMessage(Long messageId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		Long currentFolderId = null;
		Date reminderDate = null;

		ClinicianMessageTO messageTO = null;
		List<String> infoMsg = new ArrayList<String>();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		TriageGroup mailGroup = null;
		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					reminderDate = addr.getReminderDate();
					break;
				}
			}

			if( isRecipient ) {
				Addressee ownerAddrs = null;
				//For DRAFT/SENT do not modify
				for(Addressee addr:m.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						currentFolderId = addr.getFolderId();
						ownerAddrs = addr;
						break;
					}
				}
				if( currentFolderId == null ) {
					SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
				}			

				//Get Auto Assignment
				if(m.getAssignedTo() == null) {
					infoMsg = updateAssignmentStatus(m, c);
				}

				//Note: Message Service will return an error if the user is not the recipient of the message
				// TODO: How to deal with Surrogate....

				else if(!m.getAssignedTo().getId().equals(c.getId())) {
					Clinician mesgAssignedUser = userManagementService.fetchClinician(m.getAssignedTo().getId()).getPayload();
					infoMsg.add("Message is already assigned to another user: " +mesgAssignedUser.getFirstName() +" " + mesgAssignedUser.getLastName());
				}
				// If the Message is reassigned to another Triage Group/Staff, the following display message should appear when the Provider opens the message.
				ReAssignMessageHistory history = reAssignMessageService.getMsgHistoryByMsgAndAction(m.getId(), ReAssignMessageActionEnum.REASSIGN_ANOTHER_STAFF_WITH_IN_FACILITY).getPayload();
				if(history!=null){
					String reAssignedTG = history.getTriageGroupName().substring(history.getTriageGroupName().lastIndexOf("/")+1, history.getTriageGroupName().length());;
					infoMsg.add("This message has been reassigned from Triage Group: "+reAssignedTG);
				}
				ReAssignMessageHistory autoAssociatedHistory = reAssignMessageService.getAutoAssociatedTGHistoryByMessage(m.getId()).getPayload();
				if(autoAssociatedHistory!=null){
					String reAssignedTG = autoAssociatedHistory.getReAssignedTriageGroupName().substring(autoAssociatedHistory.getReAssignedTriageGroupName().lastIndexOf("/")+1, autoAssociatedHistory.getReAssignedTriageGroupName().length());;
					String displayMessage="NOTE: Patient "+m.getSenderName()+" is now associated with your Triage Group.  If you do not want to maintain the association with the patient, contact your SM Administrator to remove the association.";
					infoMsg.add(displayMessage);
				}

				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)c, false));
				// Due to unavailability of automatic assignment function for Escalated Folder and System Folders, System should capture the Message received history.
				ServiceResponse<Clinician> mailRes = mailboxService.getMailbox(c, false);
				Map<Long,Folder> folders = c.getMailbox().getFolders();

				Folder activeFolder = folders.get(currentFolderId);
				if(!activeFolder.isSystemFolder() || activeFolder.getName().equals(SystemFolderEnum.ESCALATED.getName()))
				{
					if(m.getAssignedTo()==null && m.getSenderType().equals(ParticipantTypeEnum.PATIENT)){
						if(!reAssignMessageService.findMsgReceivedHistoryByMessageId(m.getId())){

							try{
								reAssignMessageService.createMessageReceivedHistory(m.getId(),m.getRecipientId(),m.getSentDate(),c.getId());

								messageService.updateAddresseeReAssignedActive(m);
							}catch(Exception e2){
								System.out.println("Error while updateAddresseReassigned Status..."+e2);
							}
						}
					}
				}

				if(activeFolder.getName().equals(SystemFolderEnum.REMINDER.getName()) ){
					if(ownerAddrs.getReminderDate()!=null){
						infoMsg.add("The Reminder Date for this Message is "+DateUtils.formatDate(ownerAddrs.getReminderDate(),DateUtils.ENGLISH_DATE_FORMAT));
					}
				}
				//ONLY MARK MESSAGE AS READ IF THE FOLDER IS NOT SENT/DRAFT
				if( null == m.getReadReceipt() && !currentFolderId.equals(SystemFolderEnum.SENT.getId()) && !currentFolderId.equals(SystemFolderEnum.DRAFTS.getId()) ){
					//MARK MESSAGE AS READ
					ServiceResponse<Message> response4 = messageService.readMessage(m,c);
					SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response4);
					response4.getPayload();
					//UPDATE THE READ RECEIPT
					ServiceResponse<Message> response5 = messageService.updateReadReceipt(messageId);
					SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response5);
					response5.getPayload();
				}

				ServiceResponse<Message> response6 = messageService.fetchMessage(messageId);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response6);
				m = response6.getPayload();
				Thread messageThread = threadService.findThreadById(m.getThread().getId());

				ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(messageThread.getMailGroup().getId());
				mailGroup = trgRes.getPayload();

			}

			List<AttachmentTO> attList = new ArrayList<AttachmentTO>();
			if ( null != m.getAttachmentId() ){
				ServiceResponse<MessageAttachment> response2 = messageService.fetchAttachment(m.getAttachmentId());
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response2);
				MessageAttachment ma = response2.getPayload();
				if(ma!=null){
					attList.add(new AttachmentTO(ma.getId(),ma.getAttachmentName()));
				}
			}

			//IF DRAFT OR SENT FOLDER REVERSE THE SENDER/RECEIVER
			if( ! currentFolderId.equals(SystemFolderEnum.SENT.getId()) && !currentFolderId.equals(SystemFolderEnum.DRAFTS.getId()) ){
				messageTO = new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), m.getThread().getSubject(), 
						m.getBody(), m.isAttachment(), new AttachmentsTO(attList), m.getRecipientId(), m.getRecipientName(), 
						m.getSenderId(), m.getSenderName(), m.getSentDate(), m.getReadReceipt(), 
						m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null), infoMsg, 
						m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), reminderDate, mailGroup.getId());

			} else {
				//FLIP SENDER+RECEIVER
				messageTO = new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), m.getThread().getSubject(), 
						m.getBody(), m.isAttachment(), new AttachmentsTO(attList), m.getSenderId(), m.getSenderName(), 
						m.getRecipientId(), m.getRecipientName(), m.getSentDate(), m.getReadReceipt(), 
						m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null), infoMsg,
						m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), reminderDate, mailGroup.getId());
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return messageTO;
	}

	/**
	 * Session Error Codes: 108, 109, 110, 111
	 * Error Codes: 100, 904, 903, 902
	 */
	@Override
	public ClinicianMessageTO sendMessage(ClinicianMessageTO message) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		return sendMessage(message, c, null);
	}

	/**
	 * Session Error Codes: 108, 109, 110, 111
	 * Error Codes: 100, 113, 119, 904, 903, 902, 122, 123, 124
	 */
	@Override
	public ClinicianMessageTO sendMessageWithAttachment(ClinicianMessageTO message, final List<Attachment> rawAttachments) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		return sendMessage(message, c, rawAttachments);

	}

	private ClinicianMessageTO sendMessage(ClinicianMessageTO message, Clinician c, final List<Attachment> rawAttachments) {
		NewMessage nm = new NewMessage();
		List<AttachmentTO> list = new ArrayList<AttachmentTO>();
		List<Attachment> fileAttachments = new ArrayList<Attachment>();
		Attachment attachment = null;

		if( null != rawAttachments ) {
			for (Attachment oneAttachment : rawAttachments) {
				if( null == oneAttachment.getContentDisposition() ) {
					//Skip because it is not a file attachment type
					continue;
				}
				fileAttachments.add(oneAttachment);
			}
		}

		if(StringUtils.isEmpty(message.getBody())){
			SMApiUtility.throwException(ErrorEnum.BODY_CANT_BE_NULL_100);
		}

		if( message.getRecipientId() == null ) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_EMPTY_ERROR_137);
		}

		for( Attachment attach : fileAttachments ) {
			//Should only get one attachment other than "message".
			if( !attach.getContentDisposition().getParameter("name").equals("message") ) {
				attachment = attach;
				break;
			}
		}

		//At least one attachment to create an attachment for the message
		if( attachment != null ) {
			String mimeType = attachment.getContentType().toString();
			String fileName = attachment.getContentDisposition().getParameter("filename");
			if ( fileName == null || fileName.isEmpty() ) {
				SMApiUtility.throwException(ErrorEnum.FILENAME_IS_REQUIRED_123);
			}

			int mid = 0;
			String ext = null;

			if( fileName.contains(".") ) {
				mid = fileName.lastIndexOf(".")+1;
				ext = fileName.substring(mid,fileName.length()).toUpperCase();
			}

			if(null == ext || ext.isEmpty() || supportedFileTypes.containsKey(ext)) {
				SMApiUtility.throwException(ErrorEnum.UNSUPPORTED_ATTACHMENT_FILE_TYPE_122);
			}

			byte[] bytes = null;

			try {
				InputStream image = attachment.getDataHandler().getInputStream();
				bytes = SMApiUtility.extractByteArray(image);
				if( bytes.length > FILE_SIZE_LIMIT_3MB ) {
					SMApiUtility.throwException(ErrorEnum.MESSAGE_ATTACHMENT_EXCEEDS_SIZE_124);
				}
			
			} catch (IOException e) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_ATTACHMENT_ERROR_113);
			}

			ServiceResponse<MessageAttachment> response = messageService.saveAttachment(bytes, fileName, mimeType);
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
			MessageAttachment attachmentSaved = response.getPayload();

			nm.setAttachmentId(attachmentSaved.getId());
			nm.setAttachmentName(attachmentSaved.getAttachmentName());

			list.add(new AttachmentTO(attachmentSaved.getId(), attachmentSaved.getAttachmentName()));
		}

		nm.setBody(message.getBody());
		nm.setSubject(StringUtils.isEmpty(message.getSubject())?(message.getCategory().getName()+" Inquiry"):message.getSubject());
		nm.setMessageCategoryTypeId(message.getCategory().getId());
		nm.setFrom(c); 
		
		if(message.getRecipientType() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_SET_140);
		}
		if (message.getRecipientType().equals (ParticipantTypeEnum.PATIENT)) {
			User patToSend = userManagementService.findPatientById(message.getRecipientId());
			if(patToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(patToSend);
		}
		else if (message.getRecipientType().equals (ParticipantTypeEnum.CLINICIAN)) {
			ServiceResponse<Clinician> clinicToSendRes = userManagementService.fetchClinician(message.getRecipientId());
			User clinicToSend = clinicToSendRes.getPayload();
			if(clinicToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(clinicToSend);
		}
		else if (message.getRecipientType().equals (ParticipantTypeEnum.DISTRIBUTION_GROUP)) {
			DistributionGroup distGroup = distributionGroupService.findById(message.getRecipientId());
			if(distGroup == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(distGroup);
		}

		for(TriageGroup tg : c.getGroups()){
			if(tg.getId().equals(message.getSelectedTriageGroupId())){
				nm.setTriageGroup(tg);
				break;
			}
		}

		if(nm.getTriageGroup() ==null) {
			SMApiUtility.throwException(ErrorEnum.TRIAGE_TEAM_NOT_FOUND_119);
		}
		// Set the cc list
		if(message.getCcRecipientId()!=null) {
			User ccClinician = userManagementService.findById(message.getCcRecipientId());
			nm.setCc(ccClinician);
		}


		Message m = null;

		/* DETECT IF IT IS A DRAFT (PRESENCE OF A ID) AND SEND DRAFT */
		if( message.getId() != null && message.getId() != 0l) {

			ServiceResponse<Message> response = messageService.fetchMessage(message.getId());
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
			Message m1 = response.getPayload();

			if ( null != m1 ) {
				boolean isRecipient = false;
				for(Addressee addr:m1.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						isRecipient=true;
						break;
					}
				}

				if( isRecipient ) {
					Long currentFolderId = null;
					//Make sure it is a draft folder message
					for(Addressee addr:m1.getAddressees()) {
						if( addr.getOwner().getId().equals(c.getId()) ) {
							currentFolderId = addr.getFolderId();
							break;
						}
					}
					if( currentFolderId == null ) {
						SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
					}

					if( m1.getSentDate() != null || !currentFolderId.equals(SystemFolderEnum.DRAFTS.getId())) {
						SMApiUtility.throwException(ErrorEnum.MESSAGE_IS_NOT_A_DRAFT_SEND_128);
					}
					ServiceResponse<Message> response2 = sendMessageService.sendDraft(nm, m1.getId());
					SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
					m = response2.getPayload();
				}
			}
		} else {
			ServiceResponse<Message>response2 = sendMessageService.send(nm);
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
			m = response2.getPayload();
		}

		if( m == null ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}

		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();

		//Populate the MessageTO
		ClinicianMessageTO clinicianMessageTO = new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), !list.isEmpty(), new AttachmentsTO(list), 
				c.getId(), c.getName(), tg.getId(), tg.getName(), m.getSentDate(), m.getReadReceipt(), 
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.TRIAGE_GROUP:null), null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null),  m.isEscalated(),m.getStatus(), null, tg.getId());

		return clinicianMessageTO;
	}

	/**
	 * Session Error Codes: 108, 109, 110
	 * Error Codes: 112, 114, 900, 904
	 */
	@Override
	public Response deleteMessage(Long messageId) {
		ServiceResponse<Message> response2 = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response2);
		Message m = response2.getPayload();

		if (m.getSentDate() == null) {
			return deleteDraftMessage(messageId, SystemFolderEnum.DELETED.getId());
		}
		else 
			return moveMessage(messageId, SystemFolderEnum.DELETED.getId());
	}

	/**
	 * Session Error Codes: 102, 108, 109, 110, 111
	 * Error Codes: 112, 115, 114, 117, 118, 900, 904
	 */
	@Override
	public Response moveMessage(Long messageId, Long folderId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Clinician> response = mailboxService.getMailbox(c, false);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);

		Map<Long,Folder> map = c.getMailbox().getFolders();
		Folder moveFolder = map.get(folderId);
		if( null == moveFolder ) {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}
		
		if (moveFolder.getId().equals(SystemFolderEnum.ESCALATED.getId()) || moveFolder.getId().equals(SystemFolderEnum.DRAFTS.getId()) 
				||  moveFolder.getId().equals(SystemFolderEnum.SENT.getId()) || moveFolder.getId().equals(SystemFolderEnum.COMPLETED.getId()) 
				|| moveFolder.getId().equals(SystemFolderEnum.REMINDER.getId()) || moveFolder.getId().equals(SystemFolderEnum.REASSIGN.getId()) ) {
			SMApiUtility.throwException(ErrorEnum.NOT_ABLE_TO_MOVE_DRAFT_MESSAGES_118);
		}
			
		
		ServiceResponse<Message> response2 = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response2);
		Message m = response2.getPayload();

		boolean isRecipient = false;
		for(Addressee addr:m.getAddressees()) {
			if( addr.getOwner().getId().equals(c.getId()) ) {
				isRecipient=true;
				break;
			}
		}

		//CHECK IF THEY ARE THE OWNER
		if( isRecipient ) {
			Long messageFolderId = null;
			List<Addressee> list = m.getAddressees();
			for(Addressee addressee : list) {
				if( addressee.getOwner().getId().equals(c.getId()) ) {
					messageFolderId = addressee.getFolderId();
				}
			}

			if( null == messageFolderId ) {
				SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
			}

			ServiceResponse<Boolean> response3 = null;
			//Cannot move message from DRAFT folder and the Following Folder List
			if(messageFolderId.equals(SystemFolderEnum.DRAFTS.getId()) ){
				SMApiUtility.throwException(ErrorEnum.NOT_ABLE_TO_MOVE_DRAFT_MESSAGES_118);
			}
			else if ( messageFolderId.equals(SystemFolderEnum.COMPLETED.getId()) || messageFolderId.equals(SystemFolderEnum.ESCALATED.getId()) 
					|| messageFolderId.equals(SystemFolderEnum.REMINDER.getId()) || messageFolderId.equals(SystemFolderEnum.REASSIGN.getId()) 
					|| messageFolderId.equals(SystemFolderEnum.SENT.getId()) ) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_MOVE_ERROR_114);
			}
			else {
				response3 = messageService.moveMessage(m, c, moveFolder);
			}

			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response3);
			if( !response3.getPayload().booleanValue() ) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_MOVE_ERROR_114);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
		}

		return Response.ok("Success").build();
	}

	/**
	 * Session Error Codes: 108, 109, 110
	 * Error Codes: 904, 112, 115
	 */
	@Override
	public Response getMessageAttachment(Long messageId, Long attachmentId) {
		MessageAttachment ma = null;
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();

		boolean isRecipient = false;
		for(Addressee addr:m.getAddressees()) {
			if( addr.getOwner().getId().equals(c.getId()) ) {
				isRecipient=true;
				break;
			}
		}
		if( isRecipient ) {
			if( attachmentId.equals(m.getAttachmentId()) ) {
				ServiceResponse<MessageAttachment> response3 = messageService.fetchAttachment(m.getAttachmentId());
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response3);
				ma = response3.getPayload();
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
		}

		try {
			if( log.isInfoEnabled() ) {
			log.info("Message Attachment: {name=" + ma.getAttachmentName()+";size="+ ma.getAttachment().length() + ";type=" + ma.getMimeType()+"}");
			}
			return Response.ok().entity(ma.getAttachment()).header("Content-Type","application/octet-stream").header("Content-Length", ma.getAttachment().length()).header("Content-Disposition","attachment; filename="+ma.getAttachmentName()).build();
		}
		catch (SQLException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null; 
	

		//		if(ma.getSmFile().length == 0) {
		//			SMApiUtility.throwException(ErrorEnum.ATTACHMENT_IS_EMPTY_139);
		//		}
		
	}

	@Override
	public ClinicianMessageTO replyMessage(Long messageId, ClinicianMessageTO message) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		return replyMessage(messageId, message, c, null);
	}

	@Override
	public ClinicianMessageTO replyMessageWithAttachment(Long messageId, ClinicianMessageTO message, final List<Attachment> rawAttachments) {
		Session session = sessionApiService.checkSession(mc);
		Clinician pat = session.getClinician();

		return replyMessage(messageId, message, pat, rawAttachments);
	}

	private ClinicianMessageTO replyMessage(Long messageId, ClinicianMessageTO message, Clinician c, final List<Attachment> rawAttachments) {
		ServiceResponse<Message> rsp = messageService.fetchMessage(new Long(messageId));
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, rsp);

		Message currentMessage = rsp.getPayload();
		Long currentFolderId = null;
		//
		Thread messageThread = threadService.findThreadById(currentMessage.getThread().getId());

		if(StringUtils.isEmpty(message.getBody())){
			SMApiUtility.throwException(ErrorEnum.BODY_CANT_BE_NULL_100);
		}

		ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(messageThread.getMailGroup().getId());
		TriageGroup mailGroup = trgRes.getPayload();

		//To check whether mail group is active
		if(!mailGroup.isActive()){
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_REPLY_NOT_ASSOCIATED_TRIAGE_TEAM_129);
		}

		ServiceResponse<TriageGroup> triageGroup = triageGroupService.getRelationsForTriageGroup(mailGroup);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, triageGroup);

		for(Addressee addr:currentMessage.getAddressees()) {
			if( addr.getOwner().getId().equals(c.getId()) ) {
				currentFolderId = addr.getFolderId();
				break;
			}
		}
		if( currentFolderId == null ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}

		//Check that the message is not expired
		if(isMessageExpired(currentMessage)) {
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_REPLY_EXPIRED_MESSAGE_130);
		}

		//Check whether the recipient to the Original Message is a Distribution Group
		if (currentMessage.getRecipientType().equals(ParticipantTypeEnum.DISTRIBUTION_GROUP) ) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_ALLOWED_147);
		}
		
		NewMessage newMessage = new NewMessage();
		newMessage.setFrom(c);
		newMessage.setMessageCategoryTypeId(currentMessage.getThread().getMessageCategoryType().getId());
		newMessage.setSubject(currentMessage.getThread().getSubject());
		newMessage.setTriageGroup(currentMessage.getThread().getMailGroup());
		newMessage.setBody(message.getBody());
		if(currentMessage.getRecipientType() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_SET_140);
		}
		if (message.getRecipientType().equals (ParticipantTypeEnum.PATIENT)) {
			User patToSend = userManagementService.findPatientById(message.getRecipientId());
			if(patToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			newMessage.setTo(patToSend);
		}
		else if (message.getRecipientType().equals (ParticipantTypeEnum.CLINICIAN)) {
			ServiceResponse<Clinician> clinicToSendRes = userManagementService.fetchClinician(message.getRecipientId());
			User clinicToSend = clinicToSendRes.getPayload();
			if(clinicToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			newMessage.setTo(clinicToSend);
		}
		else if (message.getRecipientType().equals (ParticipantTypeEnum.DISTRIBUTION_GROUP)) {
			DistributionGroup distGroup = distributionGroupService.findById(message.getRecipientId());
			if(distGroup == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			newMessage.setTo(distGroup);
		}

		for(TriageGroup tg : c.getGroups()){
			if(tg.getId().equals(message.getSelectedTriageGroupId())){
				newMessage.setTriageGroup(tg);
				break;
			}
		}

		if(newMessage.getTriageGroup() ==null) {
			SMApiUtility.throwException(ErrorEnum.TRIAGE_TEAM_NOT_FOUND_119);
		}
		// Set the cc list
		if(message.getCcRecipientId()!=null) {
			User ccClinician = userManagementService.findById(message.getCcRecipientId());
			newMessage.setCc(ccClinician);
		}

		List<AttachmentTO> list = new ArrayList<AttachmentTO>();
		List<Attachment> fileAttachments = new ArrayList<Attachment>();

		if( null != rawAttachments ) {
			for (Attachment attachment : rawAttachments) {
				if( null == attachment.getContentDisposition() ) {
					//Skip because it is not a file attachment type
					continue;
				}
				fileAttachments.add(attachment);
			}
		}

		if( !fileAttachments.isEmpty() ) {
			for( Attachment a : fileAttachments ) {
				String mimeType = a.getContentType().toString();
				String fileName = a.getContentDisposition().getParameter("filename");
				if ( fileName == null || fileName.isEmpty() ) {
					SMApiUtility.throwException(ErrorEnum.FILENAME_IS_REQUIRED_123);
				}

				int mid = 0;
				String ext = null;

				if( fileName.contains(".") ) {
					mid = fileName.lastIndexOf(".")+1;
					ext = fileName.substring(mid,fileName.length()).toUpperCase();
				}

				if(null == ext || ext.isEmpty() || supportedFileTypes.containsKey(ext)) {
					SMApiUtility.throwException(ErrorEnum.UNSUPPORTED_ATTACHMENT_FILE_TYPE_122);
				}

				byte[] bytes = null;

				try {
					InputStream image = a.getDataHandler().getInputStream();
					bytes = SMApiUtility.extractByteArray(image);

					if( bytes.length > FILE_SIZE_LIMIT_3MB ) {
						SMApiUtility.throwException(ErrorEnum.MESSAGE_ATTACHMENT_EXCEEDS_SIZE_124);
					}
				} catch (IOException e) {
					SMApiUtility.throwException(ErrorEnum.MESSAGE_ATTACHMENT_ERROR_113);
				}

				ServiceResponse<MessageAttachment> response2 = messageService.saveAttachment(bytes, fileName, mimeType);

				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, triageGroup);

				MessageAttachment attachment = response2.getPayload();

				newMessage.setAttachmentId(attachment.getId());
				newMessage.setAttachmentName(attachment.getAttachmentName());

				list.add(new AttachmentTO(attachment.getId(), attachment.getAttachmentName()));
			}
		}

		ServiceResponse<Message> result = new ServiceResponse<Message>();
		if( currentFolderId.equals(SystemFolderEnum.SENT.getId()) ){
			result = sendMessageService.replyFromSentFolder(currentMessage, newMessage);
		} else if( message.getId() == null && currentFolderId.equals(SystemFolderEnum.DRAFTS.getId()) ){
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_REPLY_TO_A_DRAFT_MESSAGE_131);
		} 

		// Adding for Draft Crap...
		/* SEND DRAFT */
		if( message.getId() != null && message.getId() != 0l) {

			ServiceResponse<Message> response = messageService.fetchMessage(message.getId());
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
			Message m1 = response.getPayload();

			if ( null != m1 ) {
				//				if( m1.getSenderId().equals(c.getId()) || m1.getRecipientId().equals(c.getId()) ) {
				//Make sure it is a draft folder message
				for(Addressee addr:m1.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						currentFolderId = addr.getFolderId();
						break;
					}
				}
				if( currentFolderId == null ) {
					SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
				}

				if( m1.getSentDate() != null || !currentFolderId.equals(SystemFolderEnum.DRAFTS.getId())) {
					SMApiUtility.throwException(ErrorEnum.MESSAGE_IS_NOT_A_DRAFT_SEND_128);
				}
				ServiceResponse<Message> response2 = sendMessageService.sendReplyDraftForAPI(currentMessage, newMessage);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
				Message result2 = response2.getPayload();
				result.setPayload(result2);
				//				}
			}
		}
		// Ended Here

		else {
			result = sendMessageService.reply(currentMessage, newMessage);
		}

		SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, result);

		Message m = result.getPayload();

		ClinicianMessageTO messageTO = new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), !list.isEmpty(), new AttachmentsTO(), 
				c.getId(), c.getName(), mailGroup.getId(),mailGroup.getName(), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null), null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(),null, mailGroup.getId());

		return messageTO;
	}

	/**
	 * Session Error Codes: 108, 109, 110
	 * Error Codes: 904, 112, 115
	 */
	@Override
	public ClinicianMessagesTO getMessageHistory(Long messageId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		List<ClinicianMessageTO> messagesList = new ArrayList<ClinicianMessageTO>();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message message = response.getPayload();
		Date reminderDate = null;

		if ( null != message ) {
			boolean isRecipient = false;
			for(Addressee addr:message.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					reminderDate = addr.getReminderDate();
					break;
				}
			}
			if( isRecipient ) {
				Thread thread = message.getThread();
				if (thread != null) {
					List<Message> result = new ArrayList<Message>();
					for (Message msg : thread.getMessages()) {
						if (msg.getCreatedDate().before(message.getCreatedDate()) && msg.getSentDate() != null){
							result.add(msg);
						}
					}

					Collections.sort(result, new Comparator<Message>() {
						public int compare(Message msg1, Message msg2) {
							if (msg1 == null)
								return 1;
							else if (msg2 == null)
								return -1;
							else {
								Date date1 = msg1.getSentDate();
								Date date2 = msg2.getSentDate();
								if (date2.after(date1))
									return 1;
								else if (date2.before(date1))
									return -1;
								else
									return 0;
							}
						}
					});

					for( Message m : result ) {
						messagesList.add(new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(),
								m.getThread().getSubject(), m.getBody(), 
								m.isAttachment(), null, m.getSenderId(), 
								m.getSenderName(), m.getRecipientId(), 
								m.getRecipientName(), m.getSentDate(), m.getReadReceipt(),
								m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
								m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), reminderDate,null));
					}
				}
			}
			else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return new ClinicianMessagesTO(messagesList);
	}

	/**
	 * Session Error Codes: 108, 109, 110
	 * Error Codes: 117
	 */
	@Override
	public MessageCategoryTypesTO getMessageCategoryTypes() {
		Session session = sessionApiService.checkSession(mc);

		if( messageCategoryTypeEnumList.isEmpty() ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}
		return new MessageCategoryTypesTO(messageCategoryTypeEnumList);
	}

	@Override
	public ClinicianMessageTO saveDraftMessage(ClinicianMessageTO message) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		NewMessage nm = new NewMessage();
		Message m = null;
		Message m1 = null; 
		if(StringUtils.isEmpty(message.getBody())){
			SMApiUtility.throwException(ErrorEnum.BODY_CANT_BE_NULL_100);
		}

		//IF ID IS SET THEN TRY TO UPDATE THE CURRENT DRAFT INSTEAD OF CREATE
		if( message.getId() != null && message.getId() != 0l) {
			ServiceResponse<Message> response = messageService.fetchMessage(message.getId());
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
			m1 = response.getPayload();

			if ( null != m1 ) {
				boolean isRecipient = false;
				for(Addressee addr:m1.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						isRecipient=true;
						break;
					}
				}
				if( isRecipient ) {
					Long currentFolderId = null;
					//Make sure it is a draft folder message
					for(Addressee addr:m1.getAddressees()) {
						if( addr.getOwner().getId().equals(c.getId()) ) {
							currentFolderId = addr.getFolderId();
							break;
						}
					}
					if( currentFolderId == null ) {
						SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
					}

					if( m1.getSentDate() != null || !currentFolderId.equals(SystemFolderEnum.DRAFTS.getId())) {
						SMApiUtility.throwException(ErrorEnum.MESSAGE_IS_NOT_A_DRAFT_SAVE_134);
					}

					//CONTINUE THE SAVE WILL HAPPEN BASED ON THE PRESENCE OF AN ID OTHERWISE IT WILL BE NULL
				}
				User toUser = userManagementService.findById(m1.getRecipientId());
				nm.setTo(toUser);
			}
		} 

		nm.setBody(message.getBody());
		nm.setSubject(StringUtils.isEmpty(message.getSubject())?(message.getCategory().getName()+" Inquiry"):message.getSubject());
		nm.setMessageCategoryTypeId(message.getCategory().getId());
		nm.setFrom(c);

		if(message.getRecipientType() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_SET_140);
		}
		if (  (message.getId() == null || message.getId() == 0l) && message.getRecipientType().equals (ParticipantTypeEnum.PATIENT)) {
			User patToSend = userManagementService.findPatientById(message.getRecipientId());
			if(patToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(patToSend);
		}
		else if ((message.getId() == null || message.getId() == 0l) && message.getRecipientType().equals (ParticipantTypeEnum.CLINICIAN)) {
			ServiceResponse<Clinician> clinicToSendRes = userManagementService.fetchClinician(message.getRecipientId());
			User clinicToSend = clinicToSendRes.getPayload();
			if(clinicToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(clinicToSend);
		}
		else if ((message.getId() == null || message.getId() == 0l) &&  message.getRecipientType().equals (ParticipantTypeEnum.DISTRIBUTION_GROUP)) {
			DistributionGroup distGroup = distributionGroupService.findById(message.getRecipientId());
			if(distGroup == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(distGroup);
		}

		for(TriageGroup tg : c.getGroups()){
			if(tg.getId().equals(message.getSelectedTriageGroupId())){
				nm.setTriageGroup(tg);
				break;
			}
		}

		if( nm.getTo() == null ) {
			SMApiUtility.throwException(ErrorEnum.TRIAGE_TEAM_NOT_FOUND_119);
		}
		
		TriageGroup mailGroup = null;
		if (m1 != null ) {
			Thread messageThread = threadService.findThreadById(m1.getThread().getId());
			ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(messageThread.getMailGroup().getId());
			mailGroup = trgRes.getPayload();
		}
		if(nm.getTriageGroup() == null && mailGroup == null ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}	

		//If action is an update then message.getId will provide the indicator to update.
		ServiceResponse<Message> response2 = null;
		if (m1 == null)
			response2 = sendMessageService.saveDraft(nm, message.getId());
		else 
			response2 = sendMessageService.saveDraft(nm, m1.getId());
		
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
		m = response2.getPayload();

		Thread messageThread2 = threadService.findThreadById(m.getThread().getId());
		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(messageThread2.getMailGroup().getId());
				
		
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();

		//Populate the MessageTO
		return new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), false, new AttachmentsTO(), 
				c.getId(), c.getName(), tg.getId(), tg.getName(), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), null, tg.getId());

	}
	
	@Override
	public ClinicianMessageTO saveReplyDraft(Long currentMessageId, ClinicianMessageTO message) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		NewMessage nm = new NewMessage();
		Message m = null;

		if(StringUtils.isEmpty(message.getBody())){
			SMApiUtility.throwException(ErrorEnum.BODY_CANT_BE_NULL_100);
		}
		
		ServiceResponse<Message> rsp = messageService.fetchMessage(new Long(currentMessageId));
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, rsp);
		Message currentMessage = rsp.getPayload();
		Long currentFolderId = null;
		
		Thread messageThread = threadService.findThreadById(currentMessage.getThread().getId());
		ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(messageThread.getMailGroup().getId());
		TriageGroup mailGroup = trgRes.getPayload();

		//To check whether mail group is active
		if(!mailGroup.isActive()){
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_REPLY_NOT_ASSOCIATED_TRIAGE_TEAM_129);
		}

		ServiceResponse<TriageGroup> triageGroup = triageGroupService.getRelationsForTriageGroup(mailGroup);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, triageGroup);

		for(Addressee addr:currentMessage.getAddressees()) {
			if( addr.getOwner().getId().equals(c.getId()) ) {
				currentFolderId = addr.getFolderId();
				break;
			}
		}
		if( currentFolderId == null ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}

		//Check that the message is not expired
		if(isMessageExpired(currentMessage)) {
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_REPLY_EXPIRED_MESSAGE_130);
		}

		Message m1 = null;
		//IF ID IS SET THEN TRY TO UPDATE THE CURRENT DRAFT INSTEAD OF CREATE
		if( message.getId() != null && message.getId() != 0l) {
			ServiceResponse<Message> response = messageService.fetchMessage(message.getId());
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
			m1 = response.getPayload();

			if ( null != m1 ) {
				boolean isRecipient = false;
				for(Addressee addr:m1.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						isRecipient=true;
						break;
					}
				}
				if( isRecipient ) {
					Long draftMesgFolderId = null;
					//Make sure it is a draft folder message
					for(Addressee addr:m1.getAddressees()) {
						if( addr.getOwner().getId().equals(c.getId()) ) {
							draftMesgFolderId = addr.getFolderId();
							break;
						}
					}
					if( draftMesgFolderId == null ) {
						SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
					}

					if( m1.getSentDate() != null || !draftMesgFolderId.equals(SystemFolderEnum.DRAFTS.getId())) {
						SMApiUtility.throwException(ErrorEnum.MESSAGE_IS_NOT_A_DRAFT_SAVE_134);
					}

					//CONTINUE THE SAVE WILL HAPPEN BASED ON THE PRESENCE OF AN ID OTHERWISE IT WILL BE NULL
				}

			}
		} 
		User toUser = userManagementService.findById(currentMessage.getSenderId());
		nm.setTo(toUser);
		nm.setBody(message.getBody());
		nm.setSubject(messageThread.getSubject());
		nm.setMessageCategoryTypeId(messageThread.getMessageCategoryType().getId());
		nm.setFrom(c);

		if(message.getRecipientType() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_SET_140);
		}
		if (  (message.getId() == null || message.getId() == 0l) && message.getRecipientType().equals (ParticipantTypeEnum.PATIENT)) {
			User patToSend = userManagementService.findPatientById(message.getRecipientId());
			if(patToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(patToSend);
		}
		else if ((message.getId() == null || message.getId() == 0l) && message.getRecipientType().equals (ParticipantTypeEnum.CLINICIAN)) {
			ServiceResponse<Clinician> clinicToSendRes = userManagementService.fetchClinician(message.getRecipientId());
			User clinicToSend = clinicToSendRes.getPayload();
			if(clinicToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(clinicToSend);
		}
		else if ((message.getId() == null || message.getId() == 0l) &&  message.getRecipientType().equals (ParticipantTypeEnum.DISTRIBUTION_GROUP)) {
			DistributionGroup distGroup = distributionGroupService.findById(message.getRecipientId());
			if(distGroup == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(distGroup);
		}

		for(TriageGroup tg : c.getGroups()){
			if(tg.getId().equals(mailGroup.getId())){
				nm.setTriageGroup(tg);
				break;
			}
		}

		if( nm.getTo() == null ) {
			SMApiUtility.throwException(ErrorEnum.TRIAGE_TEAM_NOT_FOUND_119);
		}

		ServiceResponse<Message> response2 = null;
		//If action is an update then message.getId will provide the indicator to update.
		if( message.getId() != null && message.getId() != 0l) {
			response2 = sendMessageService.saveReplyDraft(m1, nm);
		}
		else {
			response2 = sendMessageService.saveReplyDraft(currentMessage, nm);
		}
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
		m = response2.getPayload();

		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();

		//Populate the MessageTO
		return new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), false, new AttachmentsTO(), 
				c.getId(), c.getName(), tg.getId(), tg.getName(), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), null, tg.getId());
		
	}
	
	public Response deleteDraftMessage(Long messageId, Long folderId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Clinician> response = mailboxService.getMailbox(c, false);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);

		Map<Long,Folder> map = c.getMailbox().getFolders();
		Folder moveFolder = map.get(folderId);
		if( null == moveFolder ) {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		ServiceResponse<Message> response2 = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response2);
		Message m = response2.getPayload();

		if( null != m ) {
			//CHECK IF THEY ARE THE OWNER
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}
			if( isRecipient ) {
				Long messageFolderId = null;
				List<Addressee> list = m.getAddressees();
				for(Addressee addressee : list) {
					if( addressee.getOwner().getId().equals(c.getId()) ) {
						messageFolderId = addressee.getFolderId();
					}
				}

				if( null == messageFolderId ) {
					SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
				}

				ServiceResponse<Boolean> response3 = null;

				if(messageFolderId.equals(SystemFolderEnum.DRAFTS.getId())){
					//SMApiUtility.throwException(ErrorEnum.NOT_ABLE_TO_MOVE_DRAFT_MESSAGES_118);

					response3 = messageService.deleteDraft(c, m.getId());	
				}

				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response3);
				if( !response3.getPayload().booleanValue() ) {
					SMApiUtility.throwException(ErrorEnum.MESSAGE_MOVE_ERROR_136);
				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return Response.ok("Success").build();
	}

	// START - PROVIDER SPECIFIC

	private String getSurrogateName(User user) {
		Surrogate surrogate = userManagementService.getCurrentSurrogateFor((Clinician)user);
		String surrogateName = null;
		if(surrogate!=null){
			if(surrogate.getSurrogateType()==ParticipantTypeEnum.CLINICIAN){
				User surrogateClinician = userManagementService.findById(surrogate.getSurrogateId());
				surrogateName = surrogateClinician.getName();
				surrogateClinician = null;
			}
			if(surrogate.getSurrogateType()==ParticipantTypeEnum.TRIAGE_GROUP){
				TriageGroup triageGroup = (TriageGroup)triageGroupService.findTriageGroupById(surrogate.getSurrogateId()).getPayload();
				surrogateName = triageGroup.getName();
				triageGroup=null;
			}
		}

		return surrogateName;			
	}

	private List<String> updateAssignmentStatus(Message message, Clinician clinician) {
		List<String> infoMessage = new ArrayList<String>();
		ServiceResponse<Message> response = messageService.updateAssignmentStatus( message.getId(), clinician);
		Message msg = response.getPayload();

		boolean isRecipient = false;
		for(Addressee addr:message.getAddressees()) {
			if( addr.getOwner().getId().equals(clinician.getId()) ) {
				isRecipient=true;
				break;
			}
		}

		if( isRecipient && msg.getSenderType().equals(ParticipantTypeEnum.PATIENT)){
			if(!reAssignMessageService.findMsgReceivedHistoryByMessageId(msg.getId())){
				reAssignMessageService.createMessageReceivedHistory(msg.getId(),msg.getRecipientId(),msg.getSentDate(),clinician.getId());
			}
			String userSurrogateName =" ";
			String currentSurrogate = getSurrogateName(clinician);
			reAssignMessageService.createMessageAutoAssignedHistory(msg.getId(),msg.getRecipientId(),clinician.getId(),clinician.getName()+" "+userSurrogateName);
			messageService.updateAddresseeReAssignedActive(message);
		}

		if(response.getPayload()!=null){
			infoMessage.add("This message has been automatically assigned to you");
			ReAssignMessageHistory history = reAssignMessageService.getMsgHistoryByMsgAndAction(msg.getId(), ReAssignMessageActionEnum.REASSIGN_ANOTHER_TG_WITH_IN_FACILITY).getPayload();
			if(history!=null){
				String reAssignedTG = history.getTriageGroupName().substring(history.getTriageGroupName().lastIndexOf("/")+1, history.getTriageGroupName().length());;
				infoMessage.add("This message has been reassigned from Triage Group: "+reAssignedTG);
			}
		}
		return infoMessage;
	}

	//	@Override
	//	public ClinicianStatusTO getClinicianStatus(Long messageId) {
	//		Session session = sessionApiService.checkSession(mc);
	//		Clinician c = session.getClinician();
	//		ClinicianStatusTO status = new ClinicianStatusTO();
	//		
	//		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
	//		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
	//		Message m = response.getPayload();
	//
	//		if ( null != m ) {
	//			boolean isRecipient = false;
	//			for(Addressee addr:m.getAddressees()) {
	//				if( addr.getOwner().getId().equals(c.getId()) ) {
	//					isRecipient=true;
	//					break;
	//				}
	//			}
	//			
	//			if( isRecipient ) {
	//				status.setClinicianStatusEnum(m.getStatus());
	//			} else {
	//				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
	//			}
	//		} else {
	//			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
	//		}
	//		
	//		return status;
	//	}

	@Override
	public CommentsTO getComments(Long messageId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		List<CommentTO> commentList = new ArrayList<CommentTO>();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();

		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			if( isRecipient ) {
				Thread mt = threadService.getCompleteThreadById(m.getThread().getId());
				for( Annotation a: mt.getAnnotations()) {
					commentList.add(new CommentTO(a.getId(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null),a.getAnnotation(),a.getCreatedDate()));
				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return new CommentsTO(commentList);
	}

	/**
	 * TODO:
	 * CHECK IF ONLY "COMPLETE" IS AN OPTION HERE?  CAN'T SET TO INPROCESS OR ASSIGNED etc.
	 * 
	 * DON'T POPULATE ALL DATA (NO REMINDER DATE, BODY, ETC)
	 */
	@Override
	public ClinicianMessageTO setStatus(Long messageId, ClinicianStatusTO clinicianStatus) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		Long currentFolderId = null;

		ClinicianMessageTO messageTO = new ClinicianMessageTO();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();

		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			if( isRecipient ) {
				// CAN'T setStatus IF THE MESSAGE IS COMPLETE
				for(Addressee addr:m.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						currentFolderId = addr.getFolderId();
						break;
					}
				}
				if( currentFolderId == null ) {
					SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
				}
				ServiceResponse<Clinician> mailResponse = mailboxService.getMailbox(c, false);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, mailResponse);
				Map<Long,Folder> folders = c.getMailbox().getFolders();
				Folder activeFolder = folders.get(currentFolderId);
				if(!activeFolder.getName().equals(SystemFolderEnum.COMPLETED.getName()) ) {
					messageService.setMessageStatus(m, clinicianStatus.getClinicianStatusEnum(), c, null);
				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();

		return new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(),null, false, new AttachmentsTO(), 
				c.getId(), c.getName(), (tg==null?null:tg.getId()), (tg==null?null:tg.getName()), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), null, tg.getId() );
	}

	@Override
	public ClinicianStatusTypesTO getClinicianStatusTypes() {
		Session session = sessionApiService.checkSession(mc);

		if( clinicianStatusTypeEnumList.isEmpty() ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}
		return new ClinicianStatusTypesTO(clinicianStatusTypeEnumList);
	}

	@Override
	public CommentsTO addComment(Long messageId, CommentTO comment) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		List<CommentTO> commentList = new ArrayList<CommentTO>();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();

		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			if( isRecipient ) {
				Annotation na = new Annotation();
				na.setActive(true);
				na.setAnnotation(comment.getText());
				na.setCreatedDate(new Date());
				na.setAuthor(c);
				na.setThread(m.getThread());
				ServiceResponse<Boolean> response1 = messageService.annotateThread(na);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response1);

				if( response1.getPayload().booleanValue() ) {
					Thread mt = threadService.getCompleteThreadById(m.getThread().getId());
					for( Annotation a: mt.getAnnotations()) {
						commentList.add(new CommentTO(a.getId(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null),a.getAnnotation(),a.getCreatedDate()));
					}
				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return new CommentsTO(commentList);
	}

	@Override
	public ClinicianMessageTO forwardMessage(Long messageId, ClinicianMessageTO message) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		return forwardMessage(messageId, message, c, null);
	}

	public ClinicianMessageTO forwardMessage(Long messageId, ClinicianMessageTO message, Clinician c, List<Attachment> rawAttachments) {
		NewMessage nm = new NewMessage();
		List<AttachmentTO> list = new ArrayList<AttachmentTO>();

		if( message.getRecipientId() == null ) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_EMPTY_ERROR_137);
		}

		ServiceResponse<Message> rsp = messageService.fetchMessage(new Long(messageId));
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, rsp);

		Message currentMessage = rsp.getPayload();
		Thread messageThread = threadService.findThreadById(currentMessage.getThread().getId());
		if(!StringUtils.isEmpty(message.getBody())) {
			nm.setBody(message.getBody() +"\n\n" + currentMessage.getBody());
		}
		else {
			nm.setBody( "\n\n" + currentMessage.getBody());
		}
		nm.setSubject("FW: " +(StringUtils.isEmpty(
				message.getSubject())?(messageThread.getSubject()):message.getSubject()));
		nm.setMessageCategoryTypeId(message.getCategory().getId());
		nm.setFrom(c); 
		if(message.getRecipientType() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_SET_140);
		}
		if (message.getRecipientType().equals (ParticipantTypeEnum.PATIENT)) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_FORWARD_NOT_ALLOWED_TO_PATIENT_141);
		}
		else if (message.getRecipientType().equals (ParticipantTypeEnum.CLINICIAN)) {
			ServiceResponse<Clinician> clinicToSendRes = userManagementService.fetchClinician(message.getRecipientId());
			User clinicToSend = clinicToSendRes.getPayload();
			if(clinicToSend == null) {
				SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
			}
			nm.setTo(clinicToSend);
		}
		else if (message.getRecipientType().equals (ParticipantTypeEnum.DISTRIBUTION_GROUP)) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_FORWARD_NOT_ALLOWED_TO_DIST_GRP_142);
		}

		for(TriageGroup tg : c.getGroups()){
			if(tg.getId().equals(message.getSelectedTriageGroupId())){
				nm.setTriageGroup(tg);
				break;
			}
		}

		if(nm.getTriageGroup() ==null) {
			SMApiUtility.throwException(ErrorEnum.TRIAGE_TEAM_NOT_FOUND_119);
		}
		// Set the cc list
		if(message.getCcRecipientId()!=null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_FORWARD_NOT_ALLOWED_TO_CC_143);
		}


		Message m = null;

		/* DETECT IF IT IS A DRAFT (PRESENCE OF A ID) AND SEND DRAFT */
		if( message.getId() != null && message.getId() != 0l) {

			ServiceResponse<Message> response = messageService.fetchMessage(message.getId());
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
			Message m1 = response.getPayload();

			if ( null != m1 ) {
				boolean isRecipient = false;
				for(Addressee addr:m1.getAddressees()) {
					if( addr.getOwner().getId().equals(c.getId()) ) {
						isRecipient=true;
						break;
					}
				}

				if( isRecipient ) {
					Long currentFolderId = null;
					//Make sure it is a draft folder message
					for(Addressee addr:m1.getAddressees()) {
						if( addr.getOwner().getId().equals(c.getId()) ) {
							currentFolderId = addr.getFolderId();
							break;
						}
					}
					if( currentFolderId == null ) {
						SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
					}

					if( m1.getSentDate() != null || !currentFolderId.equals(SystemFolderEnum.DRAFTS.getId())) {
						SMApiUtility.throwException(ErrorEnum.MESSAGE_IS_NOT_A_DRAFT_SEND_128);
					}
					ServiceResponse<Message> response2 = sendMessageService.sendDraft(nm, m1.getId());
					SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
					m = response2.getPayload();
				}
			}
		} else {
			ServiceResponse<Message>response2 = sendMessageService.send(nm);
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.SEND_MESSAGE_SERVICE_ERROR_903, response2);
			m = response2.getPayload();
		}

		if( m == null ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}

		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();

		//Populate the MessageTO
		ClinicianMessageTO clinicianMessageTO = new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), !list.isEmpty(), new AttachmentsTO(list), 
				c.getId(), c.getName(), tg.getId(), tg.getName(), m.getSentDate(), m.getReadReceipt(), 

				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.TRIAGE_GROUP:null), null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null),  m.isEscalated(),m.getStatus(), null, tg.getId());

		return clinicianMessageTO;
	}

	@Override
	public ClinicianMessageTO reassign(Long messageId, Long userId) {

		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		
		if(userId == null || userId.longValue() == 0 )
			SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_EMPTY_ERROR_137);
		
		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			User surrogate = null;
			CollectionServiceResponse<Surrogate> response2 = userManagementService.getSurrogatesFor(c);
			for( Surrogate s: response2.getCollection()) {
				surrogate = s.getSmsUser();
				break;
			}

			if( isRecipient ) {
				ServiceResponse<Clinician> response1 = authenticationService.fetchClinicianById(userId);
				Clinician assignedTo = response1.getPayload();
				if(assignedTo == null)
					SMApiUtility.throwException(ErrorEnum.MESSAGE_RECEIPIENT_NOT_FOUND_138);
				// Don't ASK why I did this...Lazy initialization Error
				Thread messageThread = threadService.findThreadById(m.getThread().getId());
				ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(messageThread.getMailGroup().getId());
				TriageGroup mailGroup = trgRes.getPayload();
				m.getThread().setMailGroup(mailGroup);
				
				ServiceResponse<Boolean> serviceResponse = getMessageService().assignMessageAndUpdateHistory(m, assignedTo, c, assignedTo.getName()+(surrogate==null?"":" "+surrogate.getFirstName()+" "+surrogate.getLastName()) );
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, serviceResponse);

			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();

		return new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), null, false, new AttachmentsTO(), 
				c.getId(), c.getName(), (tg==null?null:tg.getId()), (tg==null?null:tg.getName()), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), null, tg.getId());
	}

	public ClinicianMessageTO saveToCPRS(Long messageId, Long[] messages) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		if (messages.length <= 0) {
			SMApiUtility.throwException(ErrorEnum.NO_MESSAGE_FOR_TIU_SELECTED_145);	
		}

		ServiceResponse<Message> rsp = messageService.fetchMessage(new Long(messageId));
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, rsp);

		Message message = rsp.getPayload();
		Long currentFolderId = null;
		//
		Thread messageThread = threadService.findThreadById(message.getThread().getId());
		ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(messageThread.getMailGroup().getId());
		TriageGroup mailGroup = trgRes.getPayload();

		ServiceResponse<TiuNotePreview> response = getTiuNoteService().getNotePreview(messageThread, c);
		Long triageGroupId = message.getThread().getMailGroup().getId();
		SMClinicsTriageMap clinicsTriageMap = getTriageGroupService().getActiveSMClinicByTriageGroup(triageGroupId);

		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_TIU_ERROR_144, response);
		TiuNotePreview preview = response.getPayload();

		if(preview!=null){
			preview.setClinicName(DEFAULT_SM_CLINIC);
			preview.setCprsTitle(DEFAULT_CPRS_NOTE_TITLE);

			if(clinicsTriageMap!=null && clinicsTriageMap.getSmClinicName()!=null){
				preview.setClinicName(clinicsTriageMap.getSmClinicName());
				preview.setClinicIen(clinicsTriageMap.getSmClinicIen());
			}

			if(clinicsTriageMap!=null && clinicsTriageMap.getSmClinicNameCPRSTitle()!=null)
				preview.setCprsTitle(clinicsTriageMap.getSmClinicNameCPRSTitle());
			preview.setVistAPatch11Status(getTiuNoteService().getVistAPatch11Status());
			preview.setVistAPatch11Fields(getTiuNoteService().getVistAPatch11Fields());
		}		
		// Part 2
		ServiceResponse<Boolean> response2 = null;
		if(message.getAssignedTo()== null){
			message.setAssignedTo(c);
			response2 = getMessageService().assignMessage(message,c,c);
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.EROR_ASSIGNING_MESSAGE_INTIU_146, response2); 
		}


		String[] messages_list_str = new String[messages.length];

		for(int i = 0; i < messages.length; i++){
			messages_list_str[i] = String.valueOf(messages[i]);
		}

		response2 = getTiuNoteService().createProgressNote(messageThread,c, messages_list_str ,preview);

		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_TIU_ERROR_144, response2); 
		
		//Populate the MessageTO
		ClinicianMessageTO clinicianMessageTO = new ClinicianMessageTO(message.getId(), messageThread.getMessageCategoryType(), 
				message.getThread().getSubject(), message.getBody(), false, null, 
				c.getId(), c.getName(), mailGroup.getId(), mailGroup.getName(), message.getSentDate(), message.getReadReceipt(), 
				message.getRecipientType(), message.getCcRecipientName(), message.getCcRecipientId(), (message.getCcRecipientId()!=null?ParticipantTypeEnum.TRIAGE_GROUP:null), null,
				message.getEscalatedDate(), message.getCompletedDate(), (message.getAssignedTo()!=null? new ClinicianUserTO(message.getAssignedTo()):null),  message.isEscalated(),message.getStatus(), null, mailGroup.getId());

		return clinicianMessageTO;
	}
	
	@Override
	public ClinicianMessagesTO getMessageHistoryForCPRS(Long messageId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		List<ClinicianMessageTO> messagesList = new ArrayList<ClinicianMessageTO>();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message message = response.getPayload();
		Date reminderDate = null;

		if ( null != message ) {
			boolean isRecipient = false;
			for(Addressee addr:message.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					reminderDate = addr.getReminderDate();
					break;
				}
			}
			if( isRecipient ) {
				Thread thread = message.getThread();
				if (thread != null) {
					List<Message> result = new ArrayList<Message>();
					
					//CPRS Magic Starts Here
					ServiceResponse<TriageGroup> trgRes = triageGroupService.findTriageGroupById(thread.getMailGroup().getId());
					TriageGroup mailGroup = trgRes.getPayload();

					ServiceResponse<TiuNotePreview> tiuResponse = getTiuNoteService().getNotePreview(thread, c);
					Long triageGroupId = message.getThread().getMailGroup().getId();
					SMClinicsTriageMap clinicsTriageMap = getTriageGroupService().getActiveSMClinicByTriageGroup(triageGroupId);

					SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_TIU_ERROR_144, tiuResponse);
					TiuNotePreview preview = tiuResponse.getPayload();

					if(preview!=null){
						preview.setClinicName(DEFAULT_SM_CLINIC);
						preview.setCprsTitle(DEFAULT_CPRS_NOTE_TITLE);

						if(clinicsTriageMap!=null && clinicsTriageMap.getSmClinicName()!=null){
							preview.setClinicName(clinicsTriageMap.getSmClinicName());
							preview.setClinicIen(clinicsTriageMap.getSmClinicIen());
						}

						if(clinicsTriageMap!=null && clinicsTriageMap.getSmClinicNameCPRSTitle()!=null)
							preview.setCprsTitle(clinicsTriageMap.getSmClinicNameCPRSTitle());
						preview.setVistAPatch11Status(getTiuNoteService().getVistAPatch11Status());
						preview.setVistAPatch11Fields(getTiuNoteService().getVistAPatch11Fields());
					}		
					// Ends Here
					
					for (Message msg : preview.getProposedMessages()) {
						result.add(msg);
					}

					Collections.sort(result, new Comparator<Message>() {
						public int compare(Message msg1, Message msg2) {
							if (msg1 == null)
								return 1;
							else if (msg2 == null)
								return -1;
							else {
								Date date1 = msg1.getSentDate();
								Date date2 = msg2.getSentDate();
								if (date2.before(date1))
									return 1;
								else if (date2.after(date1))
									return -1;
								else
									return 0;
							}
						}
					});

					for( Message m : result ) {
						messagesList.add(new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(),
								m.getThread().getSubject(), m.getBody(), 
								m.isAttachment(), null, m.getSenderId(), 
								m.getSenderName(), m.getRecipientId(), 
								m.getRecipientName(), m.getSentDate(), m.getReadReceipt(),
								m.getRecipientType(), m.getCcRecipientName(), null, null,null,
								m.getEscalatedDate(), m.getCompletedDate(), null, m.isEscalated(), m.getStatus(), reminderDate,null));
					}
				}
			}
			else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return new ClinicianMessagesTO(messagesList);
	}

	@Override
	public ClinicianMessageTO setReminderForMessage(Long messageId, String reminderDate) {
		
		String RFC1123_PATTERN  =	"EEE, dd MMM yyyy HH:mm:ss z";
		SimpleDateFormat formatter = new SimpleDateFormat(RFC1123_PATTERN);
		Date date = null;
		try {
			date = formatter.parse(reminderDate);
		} catch (ParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
		}
		
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		TriageGroup tg = null;

		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			if( isRecipient ) {
				ServiceResponse<Boolean> response1 = messageService.setReminderDate(m, c, date);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response1);

				if( response1.getPayload().booleanValue() ) {
					ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
					SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
					tg = response3.getPayload();

				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), false, new AttachmentsTO(), 
				c.getId(), c.getName(), (tg==null?null:tg.getId()), (tg==null?null:tg.getName()), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), date, tg.getId());
	}

	@Override
	public ClinicianMessageTO removeReminderForMessage(Long messageId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();

		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		TriageGroup tg = null;

		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			if( isRecipient ) {
				ServiceResponse<Boolean> response1 = messageService.setReminderDate(m, c, null);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response1);

				if( response1.getPayload().booleanValue() ) {
					ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(m.getThread().getMailGroup().getId());
					SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
					tg = response3.getPayload();

				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

		return new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), 
				m.getThread().getSubject(), m.getBody(), false, new AttachmentsTO(), 
				c.getId(), c.getName(), (tg==null?null:tg.getId()), (tg==null?null:tg.getName()), m.getSentDate(), m.getReadReceipt(),
				m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), (m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null),null,
				m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), null, tg.getId());
	}
	// END - PROVIDER SPECIFIC

	public SessionAPIService getSessionApiService() {
		return sessionApiService;
	}
	public void setSessionApiService(SessionAPIService sessionApiService) {
		this.sessionApiService = sessionApiService;
	}
	public gov.va.med.mhv.sm.service.MessageService getMessageService() {
		return messageService;
	}
	public void setMessageService(MessageService messageService) {
		this.messageService = messageService;
	}
	public SendMessageService getSendMessageService() {
		return sendMessageService;
	}
	public void setSendMessageService(SendMessageService sendMessageService) {
		this.sendMessageService = sendMessageService;
	}
	public TriageGroupService getTriageGroupService() {
		return triageGroupService;
	}
	public void setTriageGroupService(TriageGroupService triageGroupService) {
		this.triageGroupService = triageGroupService;
	}
	public MailboxService getMailboxService() {
		return mailboxService;
	}
	public void setMailboxService(MailboxService mailboxService) {
		this.mailboxService = mailboxService;
	}
	public ThreadService getThreadService() {
		return threadService;
	}
	public void setThreadService(ThreadService threadService) {
		this.threadService = threadService;
	}
	public UserManagementService getUserManagementService() {
		return userManagementService;
	}
	public void setUserManagementService(UserManagementService userManagementService) {
		this.userManagementService = userManagementService;
	}
	public ReAssignMessageService getReAssignMessageService() {
		return reAssignMessageService;
	}
	public void setReAssignMessageService(ReAssignMessageService reAssignMessageService) {
		this.reAssignMessageService = reAssignMessageService;
	}
	public AuthenticationService getAuthenticationService() {
		return authenticationService;
	}
	public void setAuthenticationService(AuthenticationService authenticationService) {
		this.authenticationService = authenticationService;
	}

	public DistributionGroupService getDistributionGroupService() {
		return distributionGroupService;
	}

	public void setDistributionGroupService(
			DistributionGroupService distributionGroupService) {
		this.distributionGroupService = distributionGroupService;
	}

	public TiuNoteService getTiuNoteService() {
		return tiuNoteService;
	}

	public void setTiuNoteService(TiuNoteService tiuNoteService) {
		this.tiuNoteService = tiuNoteService;
	}


}
