/********************************************************************
 * Copyriight 2006 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.util;

import java.net.InetAddress;
import java.net.UnknownHostException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.util.StopWatch;

/**
 * Allows for configuring a separate logger for the StopWatch than for its
 * client/measured class.
 * 
 * <p>
 * Preferred usage is AOP approach via PerformanceMonitorInterceptor
 * </p>
 * 
 * Created Oct 13, 2006 12:11:16 PM
 * 
 * @author VHAISABOHMEG
 */
public class StopWatchLogger extends StopWatch {
	protected final Log logger = LogFactory.getLog(getClass());
	public static final String ENV_CONTEXT_SYS_PROPERTY = "server.Name";

	public static final int SLOW_THRESHOLD_MILLIS = 3000;
	public static final int FAIL_THRESHOLD_MILLIS = 10000;
	public int slowThresholdMillis = SLOW_THRESHOLD_MILLIS;
	public int failThresholdMillis = FAIL_THRESHOLD_MILLIS;

	public static final String SLOW_THRESHOLD_EXCEEDED_LOG_FLAG = "[SLOW] ";
	public static final String FAIL_THRESHOLD_EXCEEDED_LOG_FLAG = "[FAIL] ";
	public String slowThresholdExceededLogFlag = SLOW_THRESHOLD_EXCEEDED_LOG_FLAG;
	public String failThresholdExceededLogFlag = FAIL_THRESHOLD_EXCEEDED_LOG_FLAG;

	public boolean logBelowDurationThreshold = true;

	private String id;

	public StopWatchLogger() {
		this(StringUtils.EMPTY);
	}

	public StopWatchLogger(String id) {
		super(id);
		this.id = id;
		this.setKeepTaskList(false);
	}

	public void start() {
		start(this.id);
	}

	public void stopAndLog() {
		stop();
		log();
	}

	public void stopAndLog(String returnDataSummary) {
		stop();
		log(returnDataSummary);
	}

	public void stopAndLog(String returnDataSummary, Log targetLogger) {
		stop();
		log(returnDataSummary, targetLogger);
	}

	public void log() {
		log(null);
	}

	public void log(String returnDataSummary) {
		log(returnDataSummary, this.logger);
	}

	public void log(String returnDataSummary, Log targetLogger) {
		if (targetLogger.isTraceEnabled()) {
			String shortSummary = this.shortSummary();
			if (shortSummary == null)
				return;

			if (StringUtils.isNotBlank(returnDataSummary))
				targetLogger.trace(getEnvContext() + shortSummary + " - " + returnDataSummary);
			else
				targetLogger.trace(getEnvContext() + shortSummary + " -");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.springframework.util.StopWatch#shortSummary()
	 */
	@Override
	public String shortSummary() {
		long duration = getTotalTimeMillis();
		if (duration >= failThresholdMillis) {
			return failThresholdExceededLogFlag + super.shortSummary();
		} else if (duration >= slowThresholdMillis) { 
			return slowThresholdExceededLogFlag + super.shortSummary();
		} else {
			if (logBelowDurationThreshold) {
				return "[OK] " + super.shortSummary();
			}
		}

		return null;
	}

	protected String getEnvContext() {
		String envContext = System.getProperty(ENV_CONTEXT_SYS_PROPERTY);
		if (envContext == null) {
			try {
				envContext = InetAddress.getLocalHost().getHostName();
			} catch (UnknownHostException e) {
				// oh well, we tried
				envContext = "Unknown";
			}
		}
		return "[" + envContext + "] ";
	}

	public void reset() {
		if (this.isRunning())
			stop();
	}

	/**
	 * @return the logBelowDurationThreshold
	 */
	public boolean isLogBelowDurationThreshold() {
		return logBelowDurationThreshold;
	}

	/**
	 * @param logBelowDurationThreshold
	 *            the logBelowDurationThreshold to set
	 */
	public void setLogBelowDurationThreshold(boolean logBelowDurationThreshold) {
		this.logBelowDurationThreshold = logBelowDurationThreshold;
	}

	public int getSlowThresholdMillis() {
		return slowThresholdMillis;
	}

	public void setSlowThresholdMillis(int slowThresholdMillis) {
		this.slowThresholdMillis = slowThresholdMillis;
	}

	public int getFailThresholdMillis() {
		return failThresholdMillis;
	}

	public void setFailThresholdMillis(int failThresholdMillis) {
		this.failThresholdMillis = failThresholdMillis;
	}

	public String getSlowThresholdExceededLogFlag() {
		return slowThresholdExceededLogFlag;
	}

	public void setSlowThresholdExceededLogFlag(String slowThresholdExceededLogFlag) {
		this.slowThresholdExceededLogFlag = slowThresholdExceededLogFlag.trim() + " ";
	}

	public String getFailThresholdExceededLogFlag() {
		return failThresholdExceededLogFlag;
	}

	public void setFailThresholdExceededLogFlag(String failThresholdExceededLogFlag) {
		this.failThresholdExceededLogFlag = failThresholdExceededLogFlag.trim() + " ";
	}
}
