package gov.va.med.fw.service.jms;

import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.service.ServiceDescriptor;
import gov.va.med.fw.util.ExceptionHandler;
import gov.va.med.fw.util.ExceptionHandlerFactory;
import gov.va.med.fw.util.builder.Builder;
import gov.va.med.fw.util.builder.BuilderException;

import java.io.Serializable;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

public class MessageDrivenPOJO implements MessageListener, ApplicationContextAware, BeanFactoryAware {

	protected Log logger = LogFactory.getLog(getClass());
	private ApplicationContext applicationContext;
	private BeanFactory beanFactory;

	protected String getMessageTypeKey() {
		return ConfigurationConstants.DEFAULT_MESSAGE_TYPE;
	}

	protected String getMessageType(Message msg) throws InvalidMessageException {

		String type = null;
		try {
			type = msg.getStringProperty(getMessageTypeKey());
		} catch (JMSException e) {
			throw new InvalidMessageException(
					"Failed to get a message type from a message", e);
		}
		return type;
	}

	protected Object processBuilder(Builder builder, Message jmsMsg)
			throws IllegalArgumentException, BuilderException {

		if (builder == null || jmsMsg == null) {
			throw new IllegalArgumentException(
					"Missing required parameters (builder/jms message)");
		}
		return builder.build(new JMSMetaData(jmsMsg));
	}
	
	protected void processServiceInvocation(ServiceDescriptor desc, Message message,
			Object structuredMsg) throws Throwable {
		desc.invokeService(getBeanFactory(), (Serializable) structuredMsg, false);
	}
	
	protected void processException(Throwable e, Object input) {

		try {
			BeanFactory factory = this.getBeanFactory();
			ExceptionHandlerFactory exceptionFactory = (ExceptionHandlerFactory) factory
					.getBean(ExceptionHandlerFactory.class.getName());
			ExceptionHandler handler = exceptionFactory.getExceptionHandler(e.getClass().getName());
			handler.handleException(e, e.getMessage(), input);
		} catch (Throwable t) {
			if (logger.isDebugEnabled()) {
				logger.debug("Failed to get an exception handler to process exception ", t);
			}
		}
	}

	public void onMessage(Message message) {
		boolean isLoggedIn = false;
		try {
			String msgType = getMessageType(message);
			ServiceDescriptor desc = (ServiceDescriptor) getApplicationContext()
					.getBean(msgType, ServiceDescriptor.class);
			Object builtPayload = processBuilder(desc.getBuilder(), message);
			processServiceInvocation(desc, message, builtPayload);
		} catch (Throwable e) {
			processException(e, message);
			if (e instanceof RuntimeException)
				throw (RuntimeException) e;
			else
				throw new RuntimeException(e);
		}
	}

	public void setApplicationContext(ApplicationContext arg0)
			throws BeansException {
		this.applicationContext = arg0;
	}

	public ApplicationContext getApplicationContext() {
		return applicationContext;
	}

	public void setBeanFactory(BeanFactory arg0) throws BeansException {
		this.beanFactory = arg0;
	}
	
	public BeanFactory getBeanFactory() 
	{
		return beanFactory;
	}

}
