/********************************************************************
 * Copyriight 2007 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.service.jms;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.security.auth.login.LoginException;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.jms.listener.adapter.MessageListenerAdapter;

import gov.va.med.fw.security.LoginManager;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserCredentials;
import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.service.ServiceDescriptor;
import gov.va.med.fw.util.Reflector;
import gov.va.med.fw.util.builder.Builder;

/**
 * Default implementation of Spring JMS delegation to Message Driven Pojos
 * (MDP).
 * 
 * <P>
 * Can perform Acegi-based authentication if injected with a LoginManager.
 * 
 * Created Jun 26, 2007 2:46:19 PM
 * 
 * @author VHAISABOHMEG
 */
public class DefaultMessageListenerAdapter extends MessageListenerAdapter implements
		ApplicationContextAware {
	private ApplicationContext applicationContext;
	private LoginManager loginManager;

	@Override
	protected Object extractMessage(Message message) throws JMSException {
		String messageTargetServiceKey = message.getStringProperty(getMessageTargetServiceKey());
		if (message.getJMSRedelivered()) {
			logger.info("Dealing with a redelivered JMS Message with messageTargetServiceKey: "
					+ messageTargetServiceKey);
		}
		Object obj = super.extractMessage(message);
		Object convertedPayload = obj;

		UserCredentials initiater = null;
		SecureMessageWrapper secureWrapper = null;
		if (obj instanceof SecureMessageWrapper) {
			secureWrapper = (SecureMessageWrapper) obj;
			convertedPayload = secureWrapper.getPayload();
			initiater = secureWrapper.getUserCredentials();
		}

		if (initiater == null) {
			initiater = new UserCredentials();
			initiater.setAnonymous(true);
			initiater.setLogicalID(getMessageInitiater(message));
			if (secureWrapper != null)
				secureWrapper.setUserCredentials(initiater);
		}

		authenticate(initiater);

		ServiceArguments svc = new ServiceArguments();
		svc.setServiceId(messageTargetServiceKey);
		svc.setServiceArguments(convertedPayload);
		return svc;
	}

	/**
	 * Invoked as delegate method from Spring JMS.
	 * 
	 * @param svc
	 */
	public void executeService(ServiceArguments svc) throws Throwable {
		try {
			ServiceDescriptor config = (ServiceDescriptor) this.applicationContext.getBean(svc
					.getServiceId(), ServiceDescriptor.class);
			Object serviceArguments = svc.getServiceArguments();
			Builder customBuilder = config.getBuilder();
			if (customBuilder != null) {
				serviceArguments = customBuilder.build(serviceArguments);
			}
			// TOODO verify the following code
			Object service = this.applicationContext.getBean(config.getServiceName());
			Object[] args = null;
			if (serviceArguments instanceof Object[]) {
				args = (Object[]) serviceArguments;
			} else if (serviceArguments != null) {
				args = new Object[] { serviceArguments };
			}
			Reflector.invoke(service, config.getMethodName(), args);

		} catch (Throwable t) {
			// let's log the exception here rather than depend on Spring to log
			// it for us
			logger.error("Unable to execute service due to Throwable...", t);

			/*
			 * this is intentional...by rethrowing the Throwable we want the
			 * Spring framework to rollback the JTA transaction (if any). If
			 * there is a JTA transaction, it will be rolled back and the JMS
			 * message will be retried based on the configuration and
			 * implementation of the JMS Provider and the target Destination. If
			 * there is no JTA transaction, there is no retry potential (even
			 * though we bubble up this exception).
			 */
			throw t;
		} finally {
			logout();
		}
	}

	protected void logout() {
		if (loginManager == null)
			return;
		try {
			getLoginManager().logout();
		} catch (Throwable t) {
			/* nothing we can do here */
		}
	}

	protected void authenticate(UserCredentials cred) {
		if (loginManager == null)
			return;

		try {
			SecurityContextHelper.initSecurityContextOnThread(loginManager, cred);
		} catch (LoginException e) {
			throw new IllegalStateException("Unable to login with credentials: " + cred, e);
		}
	}

	protected String getMessageInitiater(Message msg) {
		String initiater = null;
		String key = getMessageInitiaterKey();
		try {
			initiater = msg.getStringProperty(key);
		} catch (JMSException e) {
			/* ok, to ignore if not there */
			if (logger.isDebugEnabled()) {
				logger.debug(key + " key is not transfered in a message");
			}
		}
		return initiater;
	}

	/**
	 * Returns a default message initiater key to look for its value in a JMS
	 * message
	 * 
	 * @return A message initiator
	 */
	protected String getMessageInitiaterKey() {
		return ConfigurationConstants.DEFAULT_MESSAGE_INITIATER;
	}

	/**
	 * Returns the target service key to drive the processing of the JMS message
	 * message
	 * 
	 * @return A message initiator
	 */
	protected String getMessageTargetServiceKey() {
		return ConfigurationConstants.DEFAULT_MESSAGE_TYPE;
	}

	public LoginManager getLoginManager() {
		return loginManager;
	}

	public void setLoginManager(LoginManager loginManager) {
		this.loginManager = loginManager;
	}

	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}
}

class ServiceArguments {
	private String serviceId;
	private Object serviceArguments;

	public Object getServiceArguments() {
		return serviceArguments;
	}

	public void setServiceArguments(Object serviceArguments) {
		this.serviceArguments = serviceArguments;
	}

	public String getServiceId() {
		return serviceId;
	}

	public void setServiceId(String serviceId) {
		this.serviceId = serviceId;
	}
}
