/*******************************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ******************************************************************************/
// Package
package gov.va.med.fw.service;

// Java classes
import java.io.Serializable;
import java.util.Map;

import org.springframework.beans.factory.BeanFactory;

import gov.va.med.fw.service.jms.MessagePayload;
import gov.va.med.fw.util.Reflector;
import gov.va.med.fw.util.builder.Builder;
import gov.va.med.fw.util.builder.BuilderException;

/**
 * Encapsulates information needed to invoke a service.
 * 
 * @author Vu Le
 * @version 1.0
 */
public class ServiceDescriptor implements Serializable {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -7855355465218555239L;

	/**
	 * A service name <code>serviceName</code>
	 */
	private String serviceName = null;

	/**
	 * A service's method name <code>methodName</code>
	 */
	private String methodName = null;

	/**
	 * An instance of dataMap
	 */
	private Map dataMap = null;

	/**
	 * An instance of builder
	 */
	private Builder builder = null;

	/**
	 * A default constructor
	 */
	public ServiceDescriptor() {
		super();
	}

	/**
	 * Returns a service's method name
	 * 
	 * @return Returns the methodName.
	 */
	public String getMethodName() {
		return methodName;
	}

	/**
	 * Sets a service's method name
	 * 
	 * @param methodName
	 *            The methodName to set.
	 */
	public void setMethodName(String methodName) {
		this.methodName = methodName;
	}

	/**
	 * Returns a service name to look up for a service in a container
	 * 
	 * @return Returns the serviceName.
	 */
	public String getServiceName() {
		return serviceName;
	}

	/**
	 * Sets a service name to look up for a service in a container
	 * 
	 * @param serviceName
	 *            The serviceName to set.
	 */
	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}

	/**
	 * @return Returns the dataMap.
	 */
	public Map getDataMap() {
		return dataMap;
	}

	/**
	 * @param dataMap
	 *            The dataMap to set.
	 */
	public void setDataMap(Map dataMap) {
		this.dataMap = dataMap;
	}

	/**
	 * @return Returns the builder.
	 */
	public Builder getBuilder() {
		return builder;
	}

	/**
	 * @param builder
	 *            The builder to set.
	 */
	public void setBuilder(Builder builder) {
		this.builder = builder;
	}

	public void invokeService(BeanFactory context, Serializable payload, boolean shouldBuildPayload)
			throws ServiceException {
		try {
			// build payload
			Object builtPayload = payload instanceof MessagePayload ? ((MessagePayload) payload)
					.getPayload() : payload;
			if (shouldBuildPayload)
				builtPayload = getBuilder().build(builtPayload);

			// invoice service
			Object[] params = null;
			if (builtPayload instanceof Object[]) {
				params = (Object[]) builtPayload;
			} else {
				params = new Object[] { builtPayload };
			}

			// Get a service and method name from a service descriptor
			String serviceName = getServiceName();
			String methodName = getMethodName();
			Object service = context.getBean(serviceName);

			Reflector.invoke(service, methodName, params);
		} catch (BuilderException e) {
			throw new ServiceException("Unable to build payload", e);
		} catch (Exception e) {
			throw new ServiceException("Unable to invoke service", e);
		}
	}

	/**
	 * Deallocate resources
	 * 
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		setServiceName(null);
		setMethodName(null);
		setBuilder(null);
		setDataMap(null);
	}
}
