package gov.va.med.fw.security;

import gov.va.med.fw.model.HashSummary;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.config.EnvironmentParamService;
import gov.va.med.fw.service.ldap.LdapPersonService;
import gov.va.med.fw.util.HashUtils;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.security.auth.login.AccountLockedException;
import javax.security.auth.login.AccountNotFoundException;
import javax.security.auth.login.CredentialException;
import javax.security.auth.login.LoginException;

import org.springframework.ldap.CommunicationException;
import org.springframework.security.Authentication;
import org.springframework.security.AuthenticationException;
import org.springframework.security.AuthenticationServiceException;
import org.springframework.security.GrantedAuthority;
import org.springframework.security.GrantedAuthorityImpl;
import org.springframework.security.providers.AuthenticationProvider;

/**
 * Central utility for authenticating against an LDAP implementation
 * 
 * Original Author - Madhu K. and VHAISWBOHMEG Creation Date - Oct 29, 2009
 * 3:09:41 PM
 */
public class LdapAuthenticationProviderImpl extends AbstractComponent implements
		AuthenticationProvider {

	private LdapPersonService ldapPersonService = null;
	private SecurityService securityService = null;
	private boolean cacheModeAlwaysSupported = false;
	private EnvironmentParamService environmentParamService;
	private String defaultRole = "Default_Role";
	private String subsystemPrefixToCheck = "Login System ";
	private String subsystemSuffixToAdd = "_USER";
	private boolean restrictInactiveUsers = false;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.springframework.security.providers.AuthenticationProvider#authenticate
	 * (org.springframework.security.Authentication)
	 */
	public Authentication authenticate(Authentication auth)
			throws AuthenticationException {
		// Process specific token only
		if (auth instanceof AuthenticationToken) {
			AuthenticationToken authToken = (AuthenticationToken) auth;

			// already logged in, return without doing anything.
			if (authToken.isLoggedIn()) {
				return auth;
			}

			try {

				// get the user from the security service (database cache)
				UserPrincipalImpl authUser = (UserPrincipalImpl) securityService
						.getUser(auth.getName());

				if (authUser == null) {
					throw new AccountNotFoundException("Account Not Found.");
				} else {
					if (authUser.isAccountLocked()) {
						throw new AccountLockedException("Account locked.");
					}

					if (restrictInactiveUsers && authUser.isInactive()) {
						throw new AccountInactiveException("Account Inactive.");
					}
				}

				// Found the user - now authenticate
				boolean ldapDown = false;
				try {
					if (!ldapPersonService.authenticate(authUser.getUserDn(),
							(String) auth.getCredentials())) {
						handleFailedLogin(auth.getName(), null);
					}
				} catch (CommunicationException e) {
					ldapDown = true;
					logger.error("** LDAP is down; reported exception: "
							+ e.getMessage());
					if (!cacheModeAlwaysSupported) {
						if (!environmentParamService.isLocalEnv())
							throw e;
					}

					if (authUser.getPassword() == null) {
						logger
								.warn("** LDAP is down, no cached password hash for user: "
										+ auth.getName());
						handleFailedLogin(auth.getName(),
								"Credential not found in Local LDAP Cache - LDAP is down.");
					}

					logger
							.warn("** LDAP is down, authenticating against cached password hash for user: "
									+ auth.getName());

					HashSummary cached = HashSummary.valueOf(authUser
							.getPassword());
					HashSummary incoming = HashUtils
							.hash(auth.getCredentials(), auth.getName()
									.toUpperCase(), cached.getAlgorithm());

					if (!incoming.getHash().equals(cached.getHash())) {
						handleFailedLogin(auth.getName(), null);
					}
				}

				authUser.setName(auth.getName().toUpperCase());
				authUser.setPassword((String) auth.getCredentials());

				GrantedAuthority[] roles = null;

				Set<String> masterRoles = null;
				if (!ldapDown) {
					masterRoles = authUser.getLdapRoles();
				} else {
					masterRoles = authUser.getCachedRoles();
				}

				if (masterRoles == null) {
					masterRoles = new HashSet<String>();
				}
				// Add default role for all valid users
				if (!masterRoles.contains(defaultRole)) {
					masterRoles.add(defaultRole);
				}

				Set<String> permissions = new HashSet<String>();
				// now loop through and get permissions for each role and add to
				// the authorities
				for (Iterator iter = masterRoles.iterator(); iter.hasNext();) {
					String roleName = (String) iter.next();
					Role role = securityService.getRole(roleName);
					if (role != null && role.getPermissions() != null) {
						for (Iterator i = role.getPermissions().iterator(); i
								.hasNext();) {
							Permission permission = (Permission) i.next();
							permissions.add(permission.getName());
						}
					}
				}
				permissions.addAll(masterRoles);

				Set<String> transientPermissions = new HashSet<String>();
				for (String permName : permissions) {
					if (permName.startsWith(subsystemPrefixToCheck)) {
						transientPermissions.add(permName.substring(
								subsystemPrefixToCheck.length(), permName
										.length())
								+ subsystemSuffixToAdd);
					}
				}
				permissions.addAll(transientPermissions);

				// add user name
				permissions.add(auth.getName().toUpperCase());

				roles = new GrantedAuthority[permissions.size()];
				int location = 0;

				// Add the roles from LDAP
				for (Iterator iter = permissions.iterator(); iter.hasNext();) {
					GrantedAuthority authority = new GrantedAuthorityImpl(
							(String) iter.next());
					roles[location++] = authority;
				}

				authUser.setAuthorities(roles);

				AuthenticationToken result = (AuthenticationToken) new AuthenticationToken(
						authToken.getPrincipal(), authToken.getCredentials(),
						(GrantedAuthority[]) roles);

				result.setUserPrincipal(authUser);
				result.setLoggedIn(true);

				// we're done, return the token.
				updateSuccessFulLogin(authUser, !ldapDown);
				return result;
			} catch (Exception e) {
				authToken.setLoggedIn(false);
				throw new AuthenticationServiceException(e.getMessage(), e);
			}
		}

		return null; // we are not processing anyother type of tokens
	}

	private void handleFailedLogin(String userName, String failureMessage)
			throws LoginException {
		try {
			if (securityService.updateFailedLogin(userName))
				throw new AccountLockedException("Account locked.");
		} catch (ServiceException se) {
			// log and ignore
		}
		if (failureMessage == null)
			failureMessage = "Invalid User Id or Password.";
		throw new CredentialException(failureMessage);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.springframework.security.providers.AuthenticationProvider#supports
	 * (java.lang.Class)
	 */
	public boolean supports(Class aClass) {
		return AuthenticationToken.class.isAssignableFrom(aClass);
	}

	/**
	 * @return the ldapPersonService
	 */
	public LdapPersonService getLdapPersonService() {
		return ldapPersonService;
	}

	/**
	 * @param ldapPersonService
	 *            the ldapPersonService to set
	 */
	public void setLdapPersonService(LdapPersonService ldapPersonService) {
		this.ldapPersonService = ldapPersonService;
	}

	/**
	 * @return the securityService
	 */
	public SecurityService getSecurityService() {
		return securityService;
	}

	/**
	 * @param securityService
	 *            the securityService to set
	 */
	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}

	public String getDefaultRole() {
		return defaultRole;
	}

	public void setDefaultRole(String defaultRole) {
		this.defaultRole = defaultRole;
	}

	private void updateSuccessFulLogin(UserPrincipal userPrincipal,
			boolean updateRoles) {
		try {
			securityService.updateSuccessFulLogin(userPrincipal, updateRoles);
		} catch (ServiceException se) {
			// log and ignore
		}
	}

	/**
	 * @return the subsystemPrefixToCheck
	 */
	public String getSubsystemPrefixToCheck() {
		return subsystemPrefixToCheck;
	}

	/**
	 * @param subsystemPrefixToCheck
	 *            the subsystemPrefixToCheck to set
	 */
	public void setSubsystemPrefixToCheck(String subsystemPrefixToCheck) {
		this.subsystemPrefixToCheck = subsystemPrefixToCheck;
	}

	/**
	 * @return the subsystemSuffixToAdd
	 */
	public String getSubsystemSuffixToAdd() {
		return subsystemSuffixToAdd;
	}

	/**
	 * @param subsystemSuffixToAdd
	 *            the subsystemSuffixToAdd to set
	 */
	public void setSubsystemSuffixToAdd(String subsystemSuffixToAdd) {
		this.subsystemSuffixToAdd = subsystemSuffixToAdd;
	}

	/**
	 * @return the restrictInactiveUsers
	 */
	public boolean isRestrictInactiveUsers() {
		return restrictInactiveUsers;
	}

	/**
	 * @param restrictInactiveUsers
	 *            the restrictInactiveUsers to set
	 */
	public void setRestrictInactiveUsers(boolean restrictInactiveUsers) {
		this.restrictInactiveUsers = restrictInactiveUsers;
	}

	/**
	 * @return the cacheModeAlwaysSupported
	 */
	public boolean isCacheModeAlwaysSupported() {
		return cacheModeAlwaysSupported;
	}

	/**
	 * @param cacheModeAlwaysSupported
	 *            the cacheModeAlwaysSupported to set
	 */
	public void setCacheModeAlwaysSupported(boolean cacheModeAlwaysSupported) {
		this.cacheModeAlwaysSupported = cacheModeAlwaysSupported;
	}

	/**
	 * @return the environmentParamService
	 */
	public EnvironmentParamService getEnvironmentParamService() {
		return environmentParamService;
	}

	/**
	 * @param environmentParamService
	 *            the environmentParamService to set
	 */
	public void setEnvironmentParamService(
			EnvironmentParamService environmentParamService) {
		this.environmentParamService = environmentParamService;
	}
}
