/*****************************************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ****************************************************************************************/
package gov.va.med.fw.rule;

// Java classes
import java.io.Serializable;
import java.math.BigDecimal;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.TimeZone;

// Library classes

// Framework classes

// EDB classes

/**
 * Project: Framework
 * 
 * @author VHAISALEV
 * @version 1.0
 */
public interface RuleMethod extends Serializable {

	/**
	 * Compares the 2 calendars for equality. This method allows a caller to
	 * pass in a significant calendar field to round off before a comparison of
	 * the 2 calendars take place.
	 * 
	 * @param arg1
	 *            A calendar to compare
	 * @param arg2
	 *            A calendar to compare
	 * @return True if 2 objects are equal. False otherwise
	 */
	public boolean isCalendarEqual(Calendar arg1, Calendar arg2, int field) throws RuleException;

	/**
	 * Compares two objects for equality, where either one or both objects may
	 * be null. For instance: ObjectUtils.equals(null, null) = true
	 * ObjectUtils.equals(null, "") = false ObjectUtils.equals("", null) = false
	 * ObjectUtils.equals("", "") = true ObjectUtils.equals(Boolean.TRUE, null)
	 * = false ObjectUtils.equals(Boolean.TRUE, "true") = false
	 * ObjectUtils.equals(Boolean.TRUE, Boolean.TRUE) = true
	 * ObjectUtils.equals(Boolean.TRUE, Boolean.FALSE) = false
	 * 
	 * @param arg1
	 *            An object to compare
	 * @param arg2
	 *            An object to compare
	 * @return True if 2 objects are equal. False otherwise
	 */
	public boolean isEqual(Object arg1, Object arg2);

	/**
	 * Compares a big decimal and a number
	 * 
	 * @param arg1
	 *            A big decima to compare
	 * @param arg2
	 *            A number to compare
	 * @return True if 2 objects are equal. False otherwise
	 */
	public boolean isEqual(BigDecimal arg1, int arg2);

	/**
	 * Checks if a value is within a range
	 * 
	 * @param lower
	 *            A lower bound
	 * @param upper
	 *            An upper bound
	 * @return True if a value is within a range. false otherwise
	 */
	public boolean isInRange(Integer value, int lower, int upper);

	/**
	 * Checks if a value is within a range
	 * 
	 * @param lower
	 *            A lower bound
	 * @param upper
	 *            An upper bound
	 * @return True if a value is within a range. false otherwise
	 */
	public boolean isInRange(BigDecimal value, double lower, double upper);

	/**
	 * Checks if a value is one of the items in a collection
	 * 
	 * @param value
	 *            An object to check
	 * @param pattern
	 *            A collection of values to compare
	 * @return True if an object is in a collection
	 */
	public boolean isPartOf(String value, String pattern);

	/**
	 * Checks if a collection is either empty or null
	 * 
	 * @param collection
	 *            A collection to check for empty and null
	 * @return True if a collection is empty or null
	 */
	public boolean isEmpty(Collection collection);

	/**
	 * Checks if a value is less than an upper bound
	 * 
	 * @param upper
	 *            An upper bound
	 * @return True if a value is less than an upper bound. false otherwise
	 */
	public boolean isLessThan(Integer value, int upper);

	/**
	 * Checks if a value is greater than a lower bound
	 * 
	 * @param lower
	 *            A lower bound
	 * @return True if a value is greater than a lower bound. false otherwise
	 */
	public boolean isGreaterThan(Integer value, int lower);

	/**
	 * Checks if a value is greater than a lower bound
	 * 
	 * @param lower
	 *            A lower bound
	 * @return True if a value is greater than a lower bound. false otherwise
	 */
	public boolean isGreaterThan(BigDecimal value, BigDecimal lower);

	/**
	 * Determines if a date to check is between two other dates (inclusive).
	 * 
	 * @param dateToCheck
	 *            The date to check
	 * @param lowerDate
	 *            The lower date to check against. If null is passed, no lower
	 *            bound will be used.
	 * @param upperDate
	 *            The upper date to check against. If null is passed, no upper
	 *            bound will be used.
	 * @return True if the date to check is between the other two dates or false
	 *         if not.
	 */
	public boolean isDateBetween(Date dateToCheck, Date lowerDate, Date upperDate);

	/**
	 * Checks if two dates are of same day
	 * 
	 * @return True if a dates are same day. false otherwise
	 */
	public boolean isSameDay(Date value1, Date value2);

	/**
	 * Checks if a date is before the specific date
	 * 
	 * @param upper
	 *            A date to compare
	 * @return True if a date is before a threshold. false otherwise
	 */
	public boolean isBefore(Date value, Date upper);

	/**
	 * Checks if a date is after the specific date
	 * 
	 * @param lower
	 *            A date to compare
	 * @return True if a date is after a threshold. false otherwise
	 */
	public boolean isAfter(Date value, Date lower);

	/**
	 * Converts a date to a calendar using the specific timezone.
	 * 
	 * @param date
	 *            A date to convert to calendar
	 * @return Calendar A calender for the specific date
	 * @throws IllegalArgumentException
	 *             Thrown if missing required date
	 */
	public Calendar toCalendar(Date date, TimeZone timezone);

	/**
	 * Converts a date to a calendar using the default system timezone.
	 * 
	 * @param date
	 *            A date to convert to calendar
	 * @return Calendar A calender for the specific date
	 * @throws IllegalArgumentException
	 *             Thrown if missing required date
	 */
	public Calendar toCalendar(Date date);

	/**
	 * Converts a Number to an Integer
	 * 
	 * @param value
	 *            A number of type Number class to be converted to Integer
	 * @return Integer A converted Integer value
	 */
	public Integer toInteger(Number value);

	/**
	 * Converts a primitive number to an Integer
	 * 
	 * @param value
	 *            A number to be converted to Integer
	 * @return Integer A converted Integer value
	 */
	public Integer toInteger(int value);

	/**
	 * @param value
	 * @return
	 * @throws ParseException
	 */
	public Date toDate(String value) throws ParseException;

	/**
	 * Returns a date formatted according to the specified pattern
	 * 
	 * @param value
	 *            a date in string format
	 * @param pattern
	 *            A pattern to format a string date
	 * @return A date formated according to a pattern
	 * @throws ParseException
	 */
	public Date toDate(String value, int pattern) throws ParseException;

	/**
	 * Returns the current system date
	 * 
	 * @return Date the current system date
	 */
	public Date getCurrentDate();

	/**
	 * Returns the current logged-in user
	 * 
	 * @return The authenticated user name
	 */
	public String getLoggedInUser();

	/**
	 * Returns the collection size. 0 if a collection is null.
	 * 
	 * @param collection
	 *            A collection to get a size
	 * @return 0 if a collection is null. Otherwise, a size() is called
	 */
	public int getSize(Collection collection);
}