/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.report.data.hibernate;

// Java classes
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Query;

import org.apache.commons.lang.Validate;
import org.hibernate.HibernateException;
import org.hibernate.ScrollMode;
import org.hibernate.ScrollableResults;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.jpa.JpaCallback;
import org.springframework.orm.jpa.JpaTemplate;

import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.report.ReportConfiguration;
import gov.va.med.fw.report.data.PostProcessedList;
import gov.va.med.fw.report.data.ReportDataDAO;
import gov.va.med.fw.report.data.ReportDataException;
import gov.va.med.fw.service.config.EnvironmentParamService;

/**
 * Encapsulates logics to execute a query to retrieve report data
 * 
 * Project: Framework</br> Created on: 4:32:29 PM </br>
 * 
 * @author VHAISALEV
 */
public abstract class HibernateReportDataDAO extends GenericDAOImpl implements ReportDataDAO {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 2573263207581367146L;

	/**
	 * A name of a named query in a hibernate mapping file
	 */
	private String queryName = null;

	/**
	 * An instance of fetchSize
	 */
	private int fetchSize = 100;

	/**
	 * An instance of cacheable
	 */
	private boolean cacheable = true;

	/**
	 * A flag to indicate whether to paginate
	 */
	private boolean paginated = true;

	/**
	 * An instance of postProcessRequired
	 */
	private boolean postProcessRequired = true;

	private EntityManagerFactory reportEntityManagerFactory;
	private EnvironmentParamService environmentParamService;

	public EntityManagerFactory getReportEntityManagerFactory() {
		return reportEntityManagerFactory;
	}

	public void setReportEntityManagerFactory(EntityManagerFactory reportEntityManagerFactory) {
		this.reportEntityManagerFactory = reportEntityManagerFactory;
	}

	public EntityManager getEntityManager() {
		return reportEntityManagerFactory.createEntityManager();
	}

	/**
	 * @return the environmentParamService
	 */
	public EnvironmentParamService getEnvironmentParamService() {
		return environmentParamService;
	}

	/**
	 * @param environmentParamService
	 *            the environmentParamService to set
	 */
	public void setEnvironmentParamService(EnvironmentParamService environmentParamService) {
		this.environmentParamService = environmentParamService;
	}

	/**
	 * A default constructor
	 */
	protected HibernateReportDataDAO() {
		super();
	}

	/**
	 * @see org.springframework.dao.support.DaoSupport#initDao()
	 */
	protected void initDao() throws Exception {
		super.initDao();
		Validate.notNull(this.queryName, "A hibernate quey name must be configured");
	}

	/**
	 * Returns a flag to determine if post processing is required
	 * 
	 * @return True if required to perform post processing. False otherwise
	 */
	public boolean isPostProcessRequired() {
		return postProcessRequired;
	}

	/**
	 * Sets a flag to indicate if post processing is required
	 * 
	 * @param postProcessRequired
	 */
	public void setPostProcessRequired(boolean postProcessRequired) {
		this.postProcessRequired = postProcessRequired;
	}

	/**
	 * The maximum number of records to be returned in a result list at one time
	 * 
	 * @param fetchSize
	 *            The fetchSize to set.
	 */
	public void setFetchSize(int fetchSize) {
		this.fetchSize = fetchSize;
	}

	/**
	 * @param queryName
	 *            The queryName to set.
	 */
	public void setQueryName(String queryName) {
		this.queryName = queryName;
	}

	/**
	 * @param cacheable
	 *            The cacheable to set.
	 */
	public void setCacheable(boolean cacheable) {
		this.cacheable = cacheable;
	}

	/**
	 * @param paginated
	 *            The paginated to set.
	 */
	public void setPaginated(boolean paginated) {
		this.paginated = paginated;
	}

	/**
	 * Returns an instance of cacheable
	 * 
	 * @return boolean cacheable.
	 */
	public boolean isCacheable() {
		return cacheable;
	}

	/**
	 * Returns an instance of fetchSize
	 * 
	 * @return int fetchSize.
	 */
	public int getFetchSize() {
		return fetchSize;
	}

	/**
	 * Returns an instance of paginated
	 * 
	 * @return boolean paginated.
	 */
	public boolean isPaginated() {
		return paginated;
	}

	public ScrollableResults getResults(final ReportConfiguration config)
			throws ReportDataException {
		// build the query and return the results
		ScrollableResults results = null;
		try {
			Query queryImpl = buildQuery(config, getEntityManager());
			org.hibernate.ejb.QueryImpl ejbQuery = (org.hibernate.ejb.QueryImpl) queryImpl;
			org.hibernate.Query hibernateQuery = ejbQuery.getHibernateQuery();
			hibernateQuery.setFetchSize(fetchSize);
			results = hibernateQuery.scroll(ScrollMode.FORWARD_ONLY);
		} catch (Throwable t) {
			logger.error("HibernateReportDataDAO:getResults failed " + t.getMessage(), t);
			throw new ReportDataException("HibernateReportDataDAO:getResults failed "
					+ t.getMessage(), t);
		}
		// Return the scrollable result set
		return results;
	}

	/**
	 * @see gov.va.med.fw.report.data.ReportDataDAO#getRecords(gov.va.med.fw.report.ReportCriteria,
	 *      int, int)
	 */
	public List getRecords(final ReportConfiguration config, final int firstRecord,
			final int maxRecord) throws ReportDataException {

		Validate.notNull(config, "A report configuration must not be NULL");
		Validate.notNull(config.getQueryCriteria(), "A report criteria must not be NULL");

		JpaCallback callback = new JpaCallback() {
			public Object doInJpa(EntityManager entityManager) {

				List results = null;
				// TODO Pagination query fetch size set to 100 records
				try {
					results = buildQuery(config, entityManager).setFirstResult(firstRecord)
							.setMaxResults(maxRecord).getResultList();
				} catch (ReportDataException e) {
					throw new HibernateException("Failed to retrieve a result set in a report dao",
							e);
				}
				return results;
			}
		};

		List results = null;

		results = buildQuery(config, getEntityManager()).setFirstResult(firstRecord).setMaxResults(
				maxRecord).getResultList();

		try {
			JpaTemplate tpl = this.getJpaTemplate();
			results = tpl.executeFind(callback);

			if (!results.isEmpty() && this.isPostProcessRequired()) {
				results = new PostProcessedList(this.postDataRetrieval(results, config), results);
			}
		} catch (DataAccessException e) {
			throw new ReportDataException("Failed to execute a query to retrieve report data", e);
		}

		return results;
	}

	/**
	 * Allows concrete implementation class to perform any pre process before
	 * data is retrieved
	 * 
	 * @param config
	 *            A report configuration
	 * @throws ReportDataException
	 *             In case of errors accessing a collection
	 */
	public void preDataRetrieval(final ReportConfiguration config) throws ReportDataException {
		// Empty implementation
	}

	/**
	 * Allows a derived class to perform any post process after data is
	 * retrieved
	 * 
	 * @param results
	 *            A collection of result items
	 * @param config
	 *            A report configuration
	 * @return A collection returned from a postDataRetrieval method is called
	 * @throws ReportDataException
	 *             In case of errors accessing a collection
	 */
	public List postDataRetrieval(final List results, final ReportConfiguration config)
			throws ReportDataException {
		return results;
	}

	/**
	 * Returns a named query using a query name from a setQueryName method
	 * 
	 * @return A hibernate query
	 */
	protected Query getNamedQuery() {
		return this.getNamedQuery(this.queryName);
	}

	/**
	 * Returns the specific named query defined in a mapping file
	 * 
	 * @param name
	 *            A name of a query string to look for in a mapping file
	 * @return A hibernate query
	 */
	protected Query getNamedQuery(String name) {
		return getJpaTemplate().getEntityManager().createNamedQuery(this.queryName);
	}

	/**
	 * Allows a derived class to build its specific report query
	 * 
	 * @param config
	 *            encapsulates parameter to build a report query
	 * @return A query to execute
	 */
	protected Query buildQuery(ReportConfiguration config, EntityManager entityManager)
			throws ReportDataException {
		return entityManager.createNamedQuery(this.queryName);
	}

	public Set<String> getNamedParameters(Query query) {
		org.hibernate.ejb.QueryImpl ejbQuery = (org.hibernate.ejb.QueryImpl) query;
		org.hibernate.Query hibernateQuery = ejbQuery.getHibernateQuery();
		String[] namedParameters = hibernateQuery.getNamedParameters();
		Set<String> names = new HashSet<String>();
		if (namedParameters != null && namedParameters.length > 0) {
			for (String parName : namedParameters) {
				names.add(parName);
			}
		}
		return names;
	}

	public TimeZone getReportingTimeZone() {
		return environmentParamService == null ? null : environmentParamService
				.getReportingTimeZone();
	}
}