/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.report.data;

// Java classes
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.hibernate.ScrollableResults;

import gov.va.med.fw.report.ReportConfiguration;

/**
 * Provides an interface to iterate through a report data's result set
 * 
 * Project: Framework</br> Created on: 5:26:54 PM </br>
 * 
 * @author VHAISALEV
 */
public class ReportDataIterator implements Serializable {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -4365696853678795611L;

	/**
	 * A default size of a result set. 100 records in a list
	 */
	private static final int PAGE_SIZE = 100;

	/**
	 * An instance of a report criteria encapsulating query parameters
	 */
	private ReportConfiguration reportConfig = null;

	/**
	 * An instance of reportDAO
	 */
	private ReportDataDAO reportDAO = null;

	private List resultPage = null;

	private int pageStart = Integer.MAX_VALUE;
	private int pageEnd = Integer.MIN_VALUE;
	private int pageSize = PAGE_SIZE;
	// private int resultPageDelta = 0;

	private ScrollableResults scrollableResults = null;

	/**
	 * A default constructor
	 * 
	 * @param provider
	 * @param config
	 */
	public ReportDataIterator(ReportDataDAO dao, ReportConfiguration config) {
		this(dao, config, PAGE_SIZE);
	}

	/**
	 * A default constructor
	 * 
	 * @param dao
	 * @param config
	 * @param pageSize
	 */
	public ReportDataIterator(ReportDataDAO dao, ReportConfiguration config, int pageSize) {
		super();
		this.reportDAO = dao;
		this.reportConfig = config;
		this.pageSize = pageSize;
	}

	/**
	 * @see gov.va.med.fw.report.data.DataIterator#getRecord(int)
	 */
	public final Object getRecord(int index) throws ReportDataException {

		if (this.resultPage == null || index < pageStart || index > pageEnd) {
			this.resultPage = getRecords(index, this.pageSize);
		}

		Object result = null;
		int pos = index - pageStart;
		if (this.resultPage != null && this.resultPage.size() > pos) {
			result = this.resultPage.get(pos);
		} else {
			// close the scrollable result set
			if (scrollableResults != null) {
				scrollableResults.close();
				scrollableResults = null;
			}
		}
		return result;
	}

	/**
	 * @see gov.va.med.fw.report.data.DataIterator#getRecords(int, int)
	 */
	public List getRecords(int firstRecord, int maxRecords) throws ReportDataException {

		if (scrollableResults == null) {
			scrollableResults = getReportDAO().getResults(this.getReportConfiguration());
		}
		if (resultPage == null) {
			resultPage = new ArrayList(maxRecords);
		}

		resultPage.clear();

		if (scrollableResults != null) {
			int total = 0;
			while (scrollableResults.next()) {
				total++;
				resultPage.add((scrollableResults.get()));
				if (total == maxRecords) {
					break;
				}
			}
		}

		// this.resultPageDelta = queryResults instanceof PostProcessedList ?
		// ((PostProcessedList)queryResults).getDeltaInSize() : 0;
		this.pageStart = firstRecord;
		this.pageEnd = firstRecord + resultPage.size() - 1;
		return resultPage;
	}

	/**
	 * @return Returns the reportDAO.
	 */
	public ReportDataDAO getReportDAO() {
		return reportDAO;
	}

	/**
	 * @return Returns the report configuration
	 */
	public ReportConfiguration getReportConfiguration() {
		return this.reportConfig;
	}

	/**
	 * @return Returns the pageSize.
	 */
	public int getPageSize() {
		return pageSize;
	}

	@Override
	protected void finalize() throws Throwable {
		// TODO Auto-generated method stub
		super.finalize();
		reportConfig = null;
		reportDAO = null;
		resultPage = null;
		if (scrollableResults != null) {
			scrollableResults.close();
			scrollableResults = null;
		}
	}
}