/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.report;

// Java classes

import org.apache.commons.lang.Validate;

import gov.va.med.fw.report.data.AbstractReportDataService;
import gov.va.med.fw.report.data.ReportData;
import gov.va.med.fw.report.data.ReportDataDAO;
import gov.va.med.fw.report.data.ReportDataService;
import gov.va.med.fw.service.AbstractComponent;

// ESR classes

/**
 * Allows concrete report builder implementations to be configured through
 * spring context file
 * 
 * Project: Framework</br> Created on: 4:59:46 PM </br>
 * 
 * @author VHAISALEV
 */
public class ConfigurableReportBuilder extends AbstractComponent implements ReportBuilder {

	public static final String DATA_SERVICE_TAG = ".dataService";
	public static final String REPORT_TEMPLATE_TAG = ".reportTemplate";

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 8201546777696454757L;
	private ReportDataService reportDataService = null;
	private ReportTemplate reportTemplate = null;
	private ReportNameResolver reportNameResolver = null;
	private ReportDataDAO reportDataDAO = null;
	private boolean asynchronous = false;

	/**
	 * A default constructor
	 */
	public ConfigurableReportBuilder() {
		super();
	}

	/**
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull(this.reportDataService, "A report data service map must be configured");
		Validate.notNull(this.reportTemplate, "A report template map must be configured");
		// set name resolver if it is set the report level
		if (reportNameResolver != null) {
			((AbstractReportDataService) this.reportDataService)
					.setNameResolver(reportNameResolver);
			((AbstractReportTemplate) this.reportTemplate).setNameResolver(reportNameResolver);
		}
	}

	/**
	 * @see gov.va.med.fw.report.ReportBuilder#buildReport(gov.va.med.fw.report.ReportConfiguration)
	 */
	public void buildReport(ReportConfiguration configuration) throws ReportException {
		this.buildReport(configuration, this.doRequestData(configuration));
	}

	/**
	 * @see gov.va.med.fw.report.ReportBuilder#buildReport(gov.va.med.fw.report.ReportConfiguration,
	 *      gov.va.med.fw.report.data.ReportData)
	 */
	public void buildReport(ReportConfiguration configuration, ReportData data)
			throws ReportException {
		this.doBuildReport(configuration, data);
	}

	/**
	 * @param configuration
	 * @return
	 * @throws ReportException
	 */
	protected ReportData doRequestData(ReportConfiguration configuration) throws ReportException {

		// If a service is configured to obtain a report's data, use it to
		// request for data
		ReportDataService service = this.getReportDataService(configuration);
		ReportData data = service != null ? service.requestReportData(configuration) : null;

		return data;
	}

	/**
	 * @param configuration
	 * @param data
	 * @return
	 * @throws ReportException
	 */
	protected void doBuildReport(ReportConfiguration configuration, ReportData data)
			throws ReportException {

		ReportExporter exporter = this.getReportExporter(configuration.getReportOutputType());
		ReportTemplate reportTemplate = this.getReportTemplate(configuration);
		if (asynchronous) {
			new Thread(
					new AsynchronousReportExporter(exporter, configuration, data, reportTemplate))
					.start();
		} else {
			exporter.exportReport(configuration, data, reportTemplate);
		}
	}

	/**
	 * @param format
	 * @return
	 * @throws ReportException
	 */
	protected ReportExporter getReportExporter(ReportExportedType.Code type) throws ReportException {

		Validate.notNull(type, "A report output type must not be NULL");

		ReportExporter exporter = null;
		try {
			// Get report exporter (pdf, excel..)
			exporter = (ReportExporter) this.getComponent(type.getAlias(), ReportExporter.class);
		} catch (Exception e) {
			throw new ReportException("Failed to obtain a report exporter ", e);
		}
		return exporter;
	}

	/**
	 * Returns a service that is used to retrieve report data
	 * 
	 * @param config
	 *            A report configuration
	 * @return ReportDataService a service for getting report data
	 */
	protected ReportDataService getReportDataService(ReportConfiguration config)
			throws ReportException {
		return reportDataService;
	}

	/**
	 * Returns a class that encapsulates an actual report template
	 * 
	 * @param config
	 *            A report configuration
	 * @return ReportTemplate a class that encapsulates an actual report
	 *         template
	 * @throws ReportException
	 *             In case of errors loading a template
	 */
	protected ReportTemplate getReportTemplate(ReportConfiguration config) throws ReportException {
		return reportTemplate;
	}

	public ReportNameResolver getReportNameResolver() {
		return reportNameResolver;
	}

	public void setReportNameResolver(ReportNameResolver reportNameResolver) {
		this.reportNameResolver = reportNameResolver;
	}

	public ReportDataDAO getReportDataDAO() {
		return reportDataDAO;
	}

	public void setReportDataDAO(ReportDataDAO reportDataDAO) {
		this.reportDataDAO = reportDataDAO;
	}

	public ReportDataService getReportDataService() {
		return reportDataService;
	}

	public void setReportDataService(ReportDataService reportDataService) {
		this.reportDataService = reportDataService;
	}

	public ReportTemplate getReportTemplate() {
		return reportTemplate;
	}

	public void setReportTemplate(ReportTemplate reportTemplate) {
		this.reportTemplate = reportTemplate;
	}

	public boolean isAsynchronous() {
		return asynchronous;
	}

	public void setAsynchronous(boolean asynchronous) {
		this.asynchronous = asynchronous;
	}

	/**
	 * Define AsynchronousReport exporter
	 * 
	 * @author vhaisakatikm
	 * 
	 */
	protected class AsynchronousReportExporter implements Runnable {
		private ReportExporter reportExporter;
		private ReportConfiguration configuration;
		private ReportData data;
		private ReportTemplate reportTemplate;

		public AsynchronousReportExporter(ReportExporter reportExporter,
				ReportConfiguration configuration, ReportData data, ReportTemplate reportTemplate) {
			this.reportExporter = reportExporter;
			this.configuration = configuration;
			this.data = data;
			this.reportTemplate = reportTemplate;
		}

		public void run() {
			try {
				reportExporter.exportReport(this.configuration, data, reportTemplate);
			} catch (Exception e) {
				logger.error("Report export failed ", e);
			} finally {
			}
		}
	}
}