/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.persistent.ldap;

import gov.va.med.fw.model.ldap.LdapConstants;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.util.StringUtils;

import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

import javax.naming.NamingException;
import javax.naming.directory.DirContext;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.ldap.CommunicationException;
import org.springframework.ldap.core.ContextMapper;
import org.springframework.ldap.core.DistinguishedName;
import org.springframework.ldap.core.LdapOperations;
import org.springframework.ldap.core.support.LdapContextSource;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.ldap.filter.LikeFilter;

/**
 * Data Access Object implementation for Ldap Group.
 * 
 * CISS ciss_framework Jun 23, 2008
 * 
 * @author VHAISAANDERC
 */
public class LdapPersonDaoImpl extends AbstractComponent implements LdapPersonDao, LdapConstants {

	private LdapOperations ldapReadOperations;
	private String domainContext = "DC=va,DC=gov";
	private String[] ldapServerUrls = null;
	private String groupOU;

	public void setGroupOU(String groupOU) {
		this.groupOU = groupOU;
	}

	public void setLdapReadOperations(LdapOperations ldapReadOperations) {
		this.ldapReadOperations = ldapReadOperations;
	}

	/**
	 * @return the domainContext
	 */
	public String getDomainContext() {
		return domainContext;
	}

	/**
	 * @param domainContext
	 *            the domainContext to set
	 */
	public void setDomainContext(String domainContext) {
		if (StringUtils.isNotEmpty(domainContext)) {
			this.domainContext = domainContext;
		}
	}

	/**
	 * @return the ldapReadServerUrl
	 */
	public String[] getLdapServerUrls() {
		return ldapServerUrls;
	}

	/**
	 * @param ldapReadServerUrl
	 *            the ldapReadServerUrl to set
	 */
	public void setLdapServerUrls(String[] ldapReadServerUrls) {
		this.ldapServerUrls = ldapReadServerUrls;
	}

	private ContextMapper getContextMapper() {
		PersonContextMapper personContextMapper = new PersonContextMapper();
		personContextMapper.setGroupOU(groupOU);
		return personContextMapper;
	}

	public List findAll() {
		EqualsFilter filter = new EqualsFilter(OBJECT_CLASS, USER);
		return ldapReadOperations.search(DistinguishedName.EMPTY_PATH, filter.encode(),
				getContextMapper());
	}

	public LdapPerson findByDistinguishedName(String dn) {
		return (LdapPerson) ldapReadOperations.lookup(dn, getContextMapper());
	}

	/**
	 * Searches ActiveDirectory by the given sAMAccountName
	 * 
	 * @param sAMAccountName
	 *            the sAmAccountaName to search by
	 * @return the found LdapPerson
	 */
	public LdapPerson findBySamAccountName(String sAMAccountName) {
		AndFilter andFilter = new AndFilter();
		andFilter.and(new EqualsFilter(OBJECT_CLASS, USER));
		andFilter.and(new EqualsFilter(SAM_ACCOUNT_NAME, sAMAccountName));
		List results = ldapReadOperations.search(DistinguishedName.EMPTY_PATH, andFilter.encode(),
				getContextMapper());
		if (results != null && results.size() > 0)
			return (LdapPerson) results.get(0);
		else
			return null;
	}

	public List find(SearchCriteria criteria) {
		AndFilter andFilter = new AndFilter();
		andFilter.and(new EqualsFilter(OBJECT_CLASS, "user"));

		// Check for minimum required search fields
		if (!isValid(criteria)) {
			return new ArrayList();
		}
		if (StringUtils.isNotEmpty(criteria.getSAMAccountName())) {
			andFilter.and(new EqualsFilter(SAM_ACCOUNT_NAME, criteria.getSAMAccountName()));
		}
		if (StringUtils.isNotEmpty(criteria.getFirstName())) {
			// andFilter.and(new EqualsFilter(GIVEN_NAME,
			// criteria.getFirstName()));
			andFilter.and(new LikeFilter(GIVEN_NAME, criteria.getFirstName()));
		}
		if (StringUtils.isNotEmpty(criteria.getLastName())) {
			andFilter.and(new EqualsFilter(SN, criteria.getLastName()));
		}
		if (StringUtils.isNotEmpty(criteria.getMiddleName())) {
			andFilter.and(new EqualsFilter(MIDDLE_NAME, criteria.getMiddleName()));
		}
		if (StringUtils.isNotEmpty(criteria.getGroupName())) {
			andFilter.and(new EqualsFilter(MEMBER_OF, criteria.getGroupName()));
		}

		// This is a convenience for testing and is not used by the application
		if (StringUtils.isNotEmpty(criteria.getName())) {
			andFilter.and(new LikeFilter(CN, criteria.getName()));
		}

		return ldapReadOperations.search(DistinguishedName.EMPTY_PATH, andFilter.encode(),
				getContextMapper());
	}

	/**
	 * Allow userid, lastname or group name search
	 * 
	 * @param criteria
	 * @return
	 */
	private boolean isValid(SearchCriteria criteria) {
		if (StringUtils.isNotEmpty(criteria.getSAMAccountName())
				|| StringUtils.isNotEmpty(criteria.getLastName())
				|| StringUtils.isNotEmpty(criteria.getName())
				|| StringUtils.isNotEmpty(criteria.getGroupName())) {
			return true;
		}
		return false;
	}

	/**
	 * Authenticate by logging into the ldap server
	 * 
	 * @param userDn
	 * @param password
	 * @return
	 */
	public boolean authenticate(String userDn, String password) throws CommunicationException {
		DirContext ctx = null;
		try {
			LdapContextSource context = new LdapContextSource();
			context.setUrls(getLdapServerUrls());
			context.setBase(getDomainContext());
			context.setUserDn(userDn);
			context.setPassword(password);
			context.setPooled(false);
			context.afterPropertiesSet();
			ctx = context.getReadOnlyContext();
			return true;
		} catch (Exception e) {
			logger.error("Unable to authenticate against LDAP", e);
			if(e instanceof CommunicationException && (ExceptionUtils.getRootCause(e) instanceof UnknownHostException || 
					ExceptionUtils.getRootCause(e) instanceof NoRouteToHostException ||
					ExceptionUtils.getRootCause(e) instanceof ConnectException)) {
				throw (CommunicationException) e; 
			}
			// TODO transalte the ldap messaged into user readable error
			// messages

			/*
			 * 525 - user not found 52e - invalid credentials 530 - not
			 * permitted to logon at this time 532 - password expired 533 -
			 * account disabled 701 - account expired 773 - user must reset
			 * password
			 */

			return false;
		} finally {
			close(ctx);
		}
	}

	/**
	 * Close the ldap directory context
	 * 
	 * @param ctx
	 */
	private void close(DirContext ctx) {
		if (ctx != null) {
			try {
				ctx.close();
			} catch (NamingException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}

}
