/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.persistent.ldap;

import java.util.Iterator;
import java.util.List;

import org.springframework.ldap.core.ContextMapper;
import org.springframework.ldap.core.DirContextAdapter;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.ldap.core.DistinguishedName;
import org.springframework.ldap.core.LdapOperations;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.ldap.filter.WhitespaceWildcardsFilter;

import gov.va.med.fw.model.ldap.LdapConstants;
import gov.va.med.fw.model.ldap.LdapGroup;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.security.Role;

/**
 * Data Access Object implementation for Ldap Group.
 * 
 * CISS ciss_framework Jun 23, 2008
 * 
 * @author VHAISAANDERC
 */
public class LdapGroupDaoImpl implements LdapGroupDao, LdapConstants {
	LdapOperations ldapReadOperations;
	LdapOperations ldapWriteOperations;
	LdapOperations ldapGroupEditOperations;
	private List<String> environmentOuList;

	private DistinguishedName buildDn(String cn) {
		DistinguishedName dn = new DistinguishedName();

		for (Iterator i = environmentOuList.iterator(); i.hasNext();) {
			String orgUnit = (String) i.next();

			/*
			 * When a brand new user is being added to ohrs groups they will
			 * also be added to the CISS User group so the ou for Roles needs to
			 * be changed to CISS Roles rather than OHRS Roles
			 */
			if (cn.equalsIgnoreCase(Role.DEFAULT_CISS_APP_ROLE)
					&& orgUnit.equalsIgnoreCase(OHRS_ROLES))
				dn.add("ou", CISS_ROLES);
			else
				dn.add("ou", orgUnit);
		}

		dn.add(CN, cn);
		return dn;
	}

	private DistinguishedName buildDnSetSamAccountName(LdapGroup group) {
		DistinguishedName dn = new DistinguishedName();

		for (Iterator i = environmentOuList.iterator(); i.hasNext();) {
			String ou = (String) i.next();
			dn.add("ou", ou);
			// append to the sAM account name if using the test or dev Ldap ou
			if (ou.startsWith(DEV_GROUPS) || ou.startsWith(TEST_GROUPS))
				group.setSamAccountName(buildEnvironmentSamAccountName(group.getSamAccountName(),
						ou));
		}

		dn.add(CN, group.getCnName());
		return dn;
	}

	private String buildEnvironmentSamAccountName(String sAM, String ou) {
		String envString = ou.replace(" ", "_");
		return (sAM + "_" + envString.substring(0, envString.indexOf(GROUPS)).toUpperCase());
	}

	private DirContextOperations setAttributes(DirContextOperations adapter, LdapGroup group) {
		adapter.setAttributeValues(OBJECT_CLASS, new String[] { TOP, GROUP });
		adapter.setAttributeValue(CN, group.getCnName());
		adapter.setAttributeValue("groupType", "-2147483640");
		adapter.setAttributeValue("sAMAccountName", group.getSamAccountName());
		if (group.getMembers() != null) {
			adapter.setAttributeValues(MEMBER, group.getMembers().toArray(new String[0]));
		}
		return adapter;
	}

	private ContextMapper getContextMapper() {
		return new GroupContextMapper();
	}

	public void setLdapReadOperations(LdapOperations ldapReadOperations) {
		this.ldapReadOperations = ldapReadOperations;
	}

	public void setLdapWriteOperations(LdapOperations ldapWriteOperations) {
		this.ldapWriteOperations = ldapWriteOperations;
	}

	public void setLdapGroupEditOperations(LdapOperations ldapGroupEditOperations) {
		this.ldapGroupEditOperations = ldapGroupEditOperations;
	}

	public void setEnvironmentOuList(List<String> environmentOuList) {
		this.environmentOuList = environmentOuList;
	}

	public void create(LdapGroup group) {
		ldapWriteOperations.bind(buildDnSetSamAccountName(group), setAttributes(
				new DirContextAdapter(), group), null);
	}

	public void delete(LdapGroup group) {
		ldapWriteOperations.unbind(buildDn(group.getCnName()));
	}

	/**
	 * Searches ActiveDirectory by the given sAMAccountName
	 * 
	 * @param sAMAccountName
	 *            the sAmAccountaName to search by
	 * @return the found LdapGroup
	 */
	public LdapGroup findBySamAccountName(String sAMAccountName) {
		AndFilter andFilter = new AndFilter();
		andFilter.and(new EqualsFilter(OBJECT_CLASS, GROUP));
		andFilter.and(new EqualsFilter(SAM_ACCOUNT_NAME, sAMAccountName));
		List results = ldapReadOperations.search(DistinguishedName.EMPTY_PATH, andFilter.encode(),
				getContextMapper());
		if (results != null && results.size() > 0)
			return (LdapGroup) results.get(0);
		else
			return null;
	}

	/**
	 * Searches ActiveDirectory by the distinguished name constructed from the
	 * given cn and the ou list for the environment.
	 * 
	 * @param cn
	 *            the cn to search by
	 * @return the found LdapGroup
	 */
	public LdapGroup findByDistinguishedName(String cn) {
		DistinguishedName dName = buildDn(cn);
		return (LdapGroup) ldapGroupEditOperations.lookup(dName, getContextMapper());
	}

	public List find(SearchCriteria criteria) {
		AndFilter filter = new AndFilter();
		filter.and(new EqualsFilter(OBJECT_CLASS, GROUP));
		filter.and(new WhitespaceWildcardsFilter(SAM_ACCOUNT_NAME, criteria.getSAMAccountName()));
		return ldapReadOperations.search(DistinguishedName.EMPTY_PATH, filter.encode(),
				getContextMapper());
	}

	public List findAll() {
		EqualsFilter filter = new EqualsFilter(OBJECT_CLASS, GROUP);
		return ldapReadOperations.search(DistinguishedName.EMPTY_PATH, filter.encode(),
				getContextMapper());
	}

	/*
	 * public LdapGroup findByDistinguishedName(String dn) { DistinguishedName
	 * dName = new DistinguishedName(dn); return (LdapGroup)
	 * ldapReadOperations.lookup(dName, getContextMapper()); }
	 */

	/**
	 * Saves changes made to the the Ldap Group
	 * 
	 * @param group
	 *            the Ldap Group to update
	 */
	public void update(LdapGroup group) {
		DistinguishedName dn = null;
		String groupDn = group.getDistinguishedName();
		if (groupDn != null && groupDn.length() > 0) {
			// Must remove the base portion of the dn
			String dnMinusBase = groupDn.substring(0, groupDn.indexOf(BASE_BEGIN) - 1);
			dn = new DistinguishedName(dnMinusBase);
		} else {
			dn = buildDn(group.getCnName());
		}

		DirContextOperations adapter = (DirContextOperations) ldapGroupEditOperations.lookup(dn);
		adapter = setAttributes(adapter, group);
		ldapWriteOperations.modifyAttributes(dn, adapter.getModificationItems());
	}

}
