/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.persistent.hibernate;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.fw.model.AbstractEntity;
import gov.va.med.fw.model.AbstractKeyedEntity;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.persistent.QueryIncrementTracker;
import gov.va.med.fw.persistent.QueryInfo;
import gov.va.med.fw.persistent.ScrollableCallback;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.PersistenceException;
import javax.persistence.Query;

import org.apache.commons.lang.Validate;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.hibernate.HibernateException;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.support.SQLExceptionTranslator;
import org.springframework.orm.jpa.JpaCallback;
import org.springframework.orm.jpa.JpaTemplate;

/**
 * Extends from an abstract dao class to provide additional convenient methods
 * to execute hql statements. An abtract dao class is aimed to serve as a base
 * class for specific DAO that has a one-to-one relationship to a persistent
 * entity. For instance, a person entity can be persisted, queried, or updated
 * through its PersonDAO class and named queries are automatically looked up in
 * the person's hibernate mapping file.</br></br> This generic DAO provides
 * generic usages for persisting any entity. Even though this implementation
 * uses Hibernate technology, specific Hibernate classes are encapsulated from a
 * caller to promote loose coupling. hence, to improve flexibility </b>
 * 
 * Project: Framework
 * 
 * @author VHAISALEV
 * @version 1.0
 */
public class GenericDAOImpl extends AbstractDAOImpl implements DAOOperations {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 2927972729631888104L;

	/**
	 * A default constructor
	 */
	public GenericDAOImpl() {
		super();
	}

	/**
	 * Executes a generic data base operation
	 * 
	 * @param action
	 *            An action to be executed in a session
	 * @return A return value from an action
	 * @throws DAOException
	 *             thrown if failed to execute an action
	 */
	public Object execute(AbstractDAOAction action) throws DAOException {
		try {
			return getJpaTemplate().execute(action);
		} catch (RuntimeException e) {
			throw new DAOException("Failed to execute an action", e);
		}
	}

	/**
	 * Check if an entity exist in the Hibernate session
	 * 
	 * @param entity
	 *            An entity to check for existence in a session
	 * @return true if an entity is in the session. false otherwise
	 */
	public boolean contains(Object entity) {
		return getJpaTemplate().contains(entity);
	}

	/**
	 * Deletes an entity from a database
	 * 
	 * @param entity
	 *            An entity to be deleted
	 * @throws DAOException
	 *             thrown if failed to delete from a database
	 */
	public void deleteEntity(Object obj) throws DAOException {
		try {
			Validate.notNull(obj);
			getJpaTemplate().remove(obj);
		} catch (RuntimeException e) {
			throw new DAOException("Failed to delete an entity", e);
		}
	}

	/**
	 * Deletes a collection of entities from a database
	 * 
	 * @param entities
	 *            A collection to be deleted
	 * @throws DAOException
	 *             thrown if failed to delete from a database
	 */
	public void deleteAll(Collection<?> entities) throws DAOException {
		try {
			Validate.notEmpty(entities);
			for (Object obj : entities)
				getJpaTemplate().remove(obj);
		} catch (RuntimeException e) {
			throw new DAOException("Failed to delete all entities", e);
		}
	}

	/**
	 * Execute the action specified by the given action object within a Session.
	 * 
	 * @param action
	 *            callback object that specifies the Hibernate action
	 * @param exposeNativeSession
	 *            whether to expose the native Hibernate Session to callback
	 *            code
	 * @return a result object returned by the action, or null
	 * @throws DAOException
	 *             in case of Hibernate errors
	 */
	public Object execute(AbstractDAOAction action, boolean exposeNativeSession)
			throws DAOException {
		try {
			Validate.notNull(action);
			return getJpaTemplate().execute(action, exposeNativeSession);
		} catch (RuntimeException e) {
			throw new DAOException("Failed to execute an action", e);
		}
	}

	/**
	 * Execute the action specified by the given action object within a Session.
	 * 
	 * @param action
	 *            callback object that specifies the Hibernate action
	 * @param exposeNativeSession
	 *            whether to expose the native Hibernate Session to callback
	 *            code
	 * @return a result object returned by the action, or null
	 * @throws DAOException
	 *             in case of Hibernate errors
	 */
	public List executeFind(AbstractDAOAction action) throws DAOException {
		return (List) execute(action, isExposeNativeSession());
	}

	/**
	 * Execute a query for persistent instances, binding one value to a "?"
	 * parameter in the query string
	 * 
	 * @param queryString
	 *            A query expression
	 * @param value
	 *            A value of a parameter
	 * @return A result entity
	 * @throws DAOException
	 *             in case of Hibernate errors
	 */
	public List find(String queryString, Object value) throws DAOException {
		try {
			Validate.notNull(queryString);
			return getJpaTemplate().find(queryString, value);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query: ").append(queryString).append(" value: ")
					.append(value);

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param query
	 * @param paramName
	 * @param value
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByNamedParam(String query, String paramName, Object value) throws DAOException {
		try {
			Validate.notNull(query);
			Validate.notNull(paramName);
			return getJpaTemplate().findByNamedParams(query,
					Collections.singletonMap(paramName, value));
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named query: ").append(query).append(
					" param Name: ").append(paramName).append(" value: ").append(
					ToStringBuilder.reflectionToString(value));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param queryString
	 * @param paramNames
	 * @param values
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByNamedParams(String queryString, String[] paramNames, Object[] values)
			throws DAOException {
		try {
			Validate.notNull(queryString, "A query string must not be null");
			Map<String, Object> mapParamValues = new HashMap<String, Object>();
			for (int i = 0; i < paramNames.length; i++) {
				mapParamValues.put(paramNames[i], values[i]);
			}
			return getJpaTemplate().findByNamedParams(queryString, mapParamValues);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named param: ").append(queryString).append(
					" param Names: ").append(ToStringBuilder.reflectionToString(paramNames))
					.append(" values: ").append(ToStringBuilder.reflectionToString(values));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param queryName
	 * @param value
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByNamedQuery(String queryName, Object value) throws DAOException {

		try {
			Validate.notNull(queryName, "A qyery string's name must not be null");
			return getJpaTemplate().findByNamedQuery(queryName, value);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named param: ").append(queryName).append(
					" values: ").append(ToStringBuilder.reflectionToString(value));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param queryName
	 * @param values
	 * @return
	 */
	public List findByNamedQuery(String queryName, Object[] values) {
		try {
			Validate.notNull(queryName, "A qyery string's name must not be null");
			return getJpaTemplate().findByNamedQuery(queryName, values);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named param: ").append(queryName).append(
					" valuess: ").append(ToStringBuilder.reflectionToString(values));

			// throw new DAOException( info.toString(), e );
		}
		return null;
	}

	/**
	 * @param queryName
	 * @param paramName
	 * @param value
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByNamedQueryAndNamedParam(String queryName, String paramName, Object value)
			throws DAOException {
		try {
			Validate.notNull(queryName, "A qyery string's name must not be null");
			Map<String, Object> map = new HashMap<String, Object>();
			map.put(paramName, value);
			return getJpaTemplate().findByNamedQueryAndNamedParams(queryName, map);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named query: ").append(queryName).append(
					" named param: ").append(paramName).append(" value: ").append(
					ToStringBuilder.reflectionToString(value));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * Execute a named query for persistent instances, binding a number of
	 * values to ":" named parameters in the query string. A named query is
	 * defined in a Hibernate mapping file
	 * 
	 * @param queryName
	 *            the name of a Hibernate query in a mapping file
	 * @param paramNames
	 *            the names of the parameters
	 * @param values
	 *            the values of the parameters
	 * @return a List containing 0 or more persistent instances
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByNamedQueryAndNamedParam(String queryName, String[] paramNames, Object[] values)
			throws DAOException {
		try {
			Validate.notNull(queryName, "A qyery string's name must not be null");
			Map<String, Object> mapParamValues = new HashMap<String, Object>();
			for (int i = 0; i < paramNames.length; i++) {
				mapParamValues.put(paramNames[i], values[i]);
			}
			return getJpaTemplate().findByNamedQueryAndNamedParams(queryName, mapParamValues);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named query: ").append(queryName).append(
					" named params: ").append(ToStringBuilder.reflectionToString(paramNames))
					.append(" value: ").append(ToStringBuilder.reflectionToString(values));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param queryName
	 * @param valueBean
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByNamedQueryAndValueBean(String queryName, Object valueBean)
			throws DAOException {
		try {
			Validate.notNull(queryName, "A query string's name must not be null");
			Validate.notNull(valueBean, "A value bean must not be null");

			return getJpaTemplate().findByNamedQuery(queryName, valueBean);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by named query: ").append(queryName).append(
					" value bean: ").append(ToStringBuilder.reflectionToString(valueBean));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param queryString
	 * @param valueBean
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List findByValueBean(String queryString, Object valueBean) throws DAOException {

		try {
			Validate.notNull(queryString, "A query string must not be null");
			Validate.notNull(valueBean, "A value bean must not be null");

			return getJpaTemplate().find(queryString, valueBean);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to execute a query by query string: ").append(queryString).append(
					" value: ").append(ToStringBuilder.reflectionToString(valueBean));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @see gov.va.med.fw.persistent.DAOOperations#getQueryString(java.lang.String)
	 */
	public String getQueryString(String queryName) throws DAOException {
		Validate.notNull(queryName);
		// FIX
		Map<String, Object> contextData = new HashMap<String, Object>();
		contextData.put("queryName", queryName);
		JpaCallback callback = new AbstractDAOAction(contextData) {
			public Object execute(EntityManager entityManager) {
				Query query = entityManager.createNamedQuery((String) getContextData().get(
						"queryName"));
				// FIX Not sure what the result is
				return query.toString();
			}

			public Object doInJpa(EntityManager arg0) throws PersistenceException {
				// TODO Auto-generated method stub
				return null;
			}
		};

		try {
			return (String) getJpaTemplate().execute(callback);
		} catch (Exception e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to get a query string by a given query name: ").append(queryName);
			throw new DAOException(info.toString(), e);
		}
	}

	public Integer scroll(ScrollableCallback callback, QueryInfo query,
			QueryIncrementTracker tracker) {
		Validate.notNull(callback);
		Validate.notNull(query);

		ScrollableDAOAction action = new ScrollableDAOAction(callback, query, tracker);
		return (Integer) getJpaTemplate().execute(action);
	}

	/**
    */
	public void flush() {
		getJpaTemplate().flush();
	}

	/**
	 * @return
	 */
	public FlushModeType getFlushMode() {
		return getJpaTemplate().getEntityManager().getFlushMode();
	}

	/**
	 * @return
	 */
	public SQLExceptionTranslator getJdbcExceptionTranslator() {
		return null;
	}

	/**
	 * @return
	 */
	public String getQueryCacheRegion() {
		return null;
	}

	/**
	 * @param proxy
	 * @throws org.springframework.dao.DAOException
	 */
	public void initialize(Object proxy) throws DAOException {
		try {
			// TOD getJpaTemplate().initialize( proxy );
		} catch (RuntimeException e) {
			throw new DAOException("Failed to initialize object", e);
		}
	}

	/**
	 * @return
	 */
	public boolean isAllowCreate() {
		return true; // TODO getJpaTemplate().isAllowCreate();
	}

	/**
	 * @return
	 */
	public boolean isAlwaysUseNewSession() {
		return true; // TODO getJpaTemplate().isAlwaysUseNewSession();
	}

	/**
	 * @return
	 */
	public boolean isCacheQueries() {
		return false; // TODO getJpaTemplate().isCacheQueries();
	}

	/**
	 * @return
	 */
	public boolean isCheckWriteOperations() {
		return true; // TODO getJpaTemplate().isCheckWriteOperations();
	}

	/**
	 * @return
	 */
	public boolean isExposeNativeSession() {
		return true; // TODO getJpaTemplate().isExposeNativeSession();
	}

	/**
	 * @param entityClass
	 * @param id
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public Object load(Class entityClass, Serializable id) throws DAOException {
		try {
			Validate.notNull(entityClass, "An entity class must not be null");
			Validate.notNull(id, "An entity ID must not be null");

			return getJpaTemplate().find(entityClass, id);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to load an entity from a session: ").append(entityClass).append(
					" with id: ").append(ToStringBuilder.reflectionToString(id));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param entity
	 * @param id
	 * @throws org.springframework.dao.DAOException
	 */
	public void load(Object entity, Serializable id) throws DAOException {
		try {
			Validate.notNull(entity, "An entity must not be null");
			Validate.notNull(id, "An entity ID must not be null");
			// TODO check
			getJpaTemplate().find(entity.getClass(), id);
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to load an entity from a session: ").append(entity).append(
					" with id: ").append(ToStringBuilder.reflectionToString(id));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param entityName
	 * @param id
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public Object load(String entityName, Serializable id) throws DAOException {
		try {
			Validate.notNull(entityName, "An entity's name must not be null");
			Validate.notNull(id, "An entity ID must not be null");

			return null;// TODO getJpaTemplate().load( entityName, id );
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to load from a session an entity named: ").append(entityName)
					.append(" with id: ").append(ToStringBuilder.reflectionToString(id));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param entityClass
	 * @return
	 * @throws org.springframework.dao.DAOException
	 */
	public List loadAll(Class entityClass) throws DAOException {
		try {
			Validate.notNull(entityClass, "An entity's class must not be null");

			return null; // TODO getJpaTemplate().loadAll( entityClass );
		} catch (RuntimeException e) {
			StringBuilder info = new StringBuilder();
			info.append("Failed to load all instances of this class: ").append(
					ToStringBuilder.reflectionToString(entityClass));

			throw new DAOException(info.toString(), e);
		}
	}

	/**
	 * @param entity
	 */
	public void refresh(Object entity) {
		if (getJpaTemplate().contains(entity)) {
			getJpaTemplate().refresh(entity);
		}
	}

	/**
	 * @param flushMode
	 */
	public void setFlushMode(FlushModeType flushModeType) {
		getJpaTemplate().getEntityManager().setFlushMode(flushModeType);
	}

	/**
	 * @param constantName
	 */
	public void setFlushModeName(String constantName) {
		FlushModeType.valueOf(constantName);
		getJpaTemplate().getEntityManager().setFlushMode(FlushModeType.valueOf(constantName));
	}

	public Object update(Object entity) throws DAOException {
		return saveObject(entity);
	}

	public Object update(AbstractEntity entity, AbstractKeyedEntity lock) throws DAOException {
		return saveObject(entity, lock);
	}

	@SuppressWarnings("unchecked")
	public List<? extends AbstractKeyedEntity> find(String query, int firstRecord, int maxRecord,
			int fetchSize) throws DAOException {
		Validate.notNull(query);

		Map<String, Object> contextData = new HashMap<String, Object>();
		contextData.put("query", query);
		contextData.put("firstRecord", new Integer(firstRecord));
		contextData.put("maxRecord", new Integer(maxRecord));
		contextData.put("fetchSize", new Integer(fetchSize));
		JpaCallback callback = new AbstractDAOAction(contextData) {
			public Object execute(EntityManager entityManager) {
				Query queryImpl = entityManager.createNamedQuery(
						(String) getContextData().get("query")).setFirstResult(
						((Integer) getContextData().get("firstRecord")).intValue()).setMaxResults(
						((Integer) getContextData().get("fetchSize")).intValue());
				return queryImpl.getResultList();
			}
		};

		return getJpaTemplate().executeFind(callback);
	}

	@SuppressWarnings("unchecked")
	public List<? extends AbstractKeyedEntity> find(String query, String[] paramNames,
			Object[] params, int firstRecord, int maxRecord, int fetchSize) throws DAOException {
		Map<String, Object> contextData = new HashMap<String, Object>();
		contextData.put("query", query);
		contextData.put("paramNames", paramNames);
		contextData.put("params", params);
		contextData.put("firstRecord", new Integer(firstRecord));
		contextData.put("maxRecord", new Integer(maxRecord));
		contextData.put("fetchSize", new Integer(fetchSize));
		JpaCallback callback = new AbstractDAOAction(contextData) {
			public Object execute(EntityManager entityManager) {
				Query queryImpl = entityManager.createNamedQuery(
						(String) getContextData().get("query")).setFirstResult(
						((Integer) getContextData().get("firstRecord")).intValue()).setMaxResults(
						((Integer) getContextData().get("fetchSize")).intValue());

				String[] targetParamNames = (String[]) getContextData().get("paramNames");
				Object[] targetParams = (Object[]) getContextData().get("params");
				if (targetParamNames != null) {
					for (int i = 0; i < targetParamNames.length; i++) {
						queryImpl.setParameter(targetParamNames[i], targetParams[i]);
					}
				}

				return queryImpl.getResultList();
			}
		};
		return getJpaTemplate().executeFind(callback);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.persistent.DAOOperations#bulkUpdate(java.lang.String)
	 */
	public int bulkUpdate(String query) throws DAOException {
		try {
			return 0; // TODO return getJpaTemplate().bulkUpdate(query);
		} catch (Exception e) {
			throw new DAOException("Unable to perform bulk update", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.persistent.DAOOperations#bulkUpdate(java.lang.String,
	 * java.lang.Object[])
	 */
	public int bulkUpdate(String query, Object[] parameters) throws DAOException {
		try {
			return 0; // TODO getJpaTemplate().bulkUpdate(query, parameters);
		} catch (Exception e) {
			throw new DAOException("Unable to perform bulk update", e);
		}
	}

	protected Throwable getRootExceptionOfType(Throwable e,
			Class<? extends Throwable> targetRootExceptionType) {
		if (e != null && targetRootExceptionType != null) {
			Throwable rootCause = ExceptionUtils.getRootCause(e);
			if (rootCause != null && targetRootExceptionType.isAssignableFrom(rootCause.getClass()))
				return rootCause;
		}
		return null;
	}

	public List<?> executeSQLQuery(final QueryAndParams queryAndParams) throws DAOException {
		List<?> results = null;
		try {
			JpaCallback callback = new AbstractDAOAction() {
				public Object execute(EntityManager entityManager) {
					String queryString = queryAndParams.getQuery();
					Query query = entityManager.createNativeQuery(queryString);
					setQueryParameters(query, queryAndParams.getParams());
					return query.getResultList();
				}
			};
			results = getJpaTemplate().executeFind(callback);
		} catch (DataAccessException e) {
			throw new DAOException("Failed to execute an sql query:" + queryAndParams.getQuery(), e);
		}

		return results;
	}

	@SuppressWarnings("unchecked")
	public <T> List<T> executeSQLQuery(final String sqlQuery, final Class<T> clazz,
			final Map<String, Object> params) throws DAOException {
		List<T> results = null;
		try {
			JpaCallback callback = new AbstractDAOAction() {
				public Object execute(EntityManager entityManager) {
					Query query = entityManager.createNativeQuery(sqlQuery, clazz);
					setQueryParameters(query, params);
					return query.getResultList();
				}
			};
			results = getJpaTemplate().executeFind(callback);
		} catch (DataAccessException e) {
			throw new DAOException("Failed to execute an sql query:" + sqlQuery, e);
		}

		return results;
	}

	public List executeNamedSQLQuery(final String queryName, final Map<String, Object> params)
			throws DAOException {
		List results = null;
		try {
			JpaTemplate tpl = getJpaTemplate();
			// results = tpl.executeFind(callback);
			results = tpl.executeFind(new JpaCallback() {
				public Object doInJpa(EntityManager entityManager) {
					List results;
					try {
						Query namedQuery = entityManager.createNamedQuery(queryName);
						AbstractDAOImpl.setQueryParameters(namedQuery, params);
						results = namedQuery.getResultList();
					} catch (Exception e) {
						throw new HibernateException(e.getMessage(), e);
					}
					return results;
				}
			});
		} catch (DataAccessException e) {
			throw new DAOException("Failed to load PAID Data", e);
		}
		return results;
	}

	public void persist(Object arg0) throws DataAccessException {
		getJpaTemplate().persist(arg0);
	}

	public void remove(Object arg0) throws DataAccessException {
		getJpaTemplate().remove(arg0);
	}

	public Object merge(Object entity) throws DataAccessException {
		return getJpaTemplate().merge(entity);
	}

	// unimplemented methods
	public void clear() throws DAOException {
	}

	public void closeIterator(Iterator it) throws DAOException {
		// TODO Auto-generated method stub
	}

	public void evict(Object entity) throws DAOException {
		// TODO Auto-generated method stub

	}

	public String[] getFilterNames() {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator iterate(String queryString) throws DAOException {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator iterate(String queryString, Object value) throws DAOException {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator iterate(String queryString, Object[] values) throws DAOException {
		// TODO Auto-generated method stub
		return null;
	}

	public Object merge(String entityName, Object entity) throws DAOException {
		// TODO Auto-generated method stub
		return null;
	}

	public void setAllowCreate(boolean allowCreate) {
		// TODO Auto-generated method stub

	}

	public void setAlwaysUseNewSession(boolean alwaysUseNewSession) {
		// TODO Auto-generated method stub

	}

	public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
		// TODO Auto-generated method stub

	}

	public void setCacheQueries(boolean cacheQueries) {
		// TODO Auto-generated method stub

	}

	public void setCheckWriteOperations(boolean checkWriteOperations) {
		// TODO Auto-generated method stub

	}

	public void setEntityInterceptorBeanName(String entityInterceptorBeanName) {
		// TODO Auto-generated method stub

	}

	public void setExposeNativeSession(boolean exposeNativeSession) {
		// TODO Auto-generated method stub

	}

	public void setFilterName(String filter) {
		// TODO Auto-generated method stub

	}

	public void setFilterNames(String[] filterNames) {
		// TODO Auto-generated method stub

	}

	public void setJdbcExceptionTranslator(SQLExceptionTranslator jdbcExceptionTranslator) {
		// TODO Auto-generated method stub
	}

	public void setQueryCacheRegion(String queryCacheRegion) {
		// TODO Auto-generated method stub
	}

}