/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.TimeZone;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.springframework.security.GrantedAuthority;
import org.springframework.security.GrantedAuthorityImpl;

import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.security.UserCredentials;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.util.StringUtils;

/**
 * UserPrincipal implementation.
 * 
 * @author Ghazenfer Mansoor
 * @version 1.0
 */
public class UserPrincipalImpl extends AbstractKeyedEntity implements UserPrincipal {

	private static final long serialVersionUID = -7044136313562745972L;

	// Authentication attributes
	private UserCredentials userCredentials = new UserCredentials();
	private GrantedAuthority[] authorities = null;
	private Set<String> ldapRoles = null;
	private String userDn = null;
	// User details from ldap
	private String givenName = null;
	private String middleName = null;
	private String familyName = null;
	private String title;
	private String department;
	private String description;
	private String office;
	private String telephoneNumber;
	private String email;
	private TimeZone currentTimeZone;

	/**
	 * Default Constructor.
	 */
	public UserPrincipalImpl() {
		super();
	}

	/**
	 * Construct using name
	 * 
	 * @param name
	 */
	public UserPrincipalImpl(String name) {
		super();
		userCredentials.setUserID(name);
	}

	/**
	 * Construct using name and password.
	 * 
	 * @param name
	 * @param password
	 */
	public UserPrincipalImpl(String name, String password) {
		this(name);
		userCredentials.setPassword(password);
	}

	public UserCredentials getUserCredentials() {
		return userCredentials;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.security.Principal#getName()
	 */
	public String getName() {
		return userCredentials.getUserID();
	}

	/**
	 * Set the Principal name
	 * 
	 * @param name
	 */
	public void setName(String name) {
		userCredentials.setUserID(name);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getPassword()
	 */
	public String getPassword() {
		return userCredentials.getPassword();
	}

	/**
	 * @param password
	 */
	public void setPassword(String password) {
		userCredentials.setPassword(password);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getGivenName()
	 */
	public String getGivenName() {
		return this.givenName;
	}

	/**
	 * @param givenName
	 */
	public void setGivenName(String givenName) {
		this.givenName = givenName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.vha.fw.security.UserPrincipal#getMiddleName()
	 */
	public String getMiddleName() {
		return this.middleName;
	}

	/**
	 * @param middleName
	 */
	public void setMiddleName(String middleName) {
		this.middleName = middleName;
	}

	/**
	 * 
	 * @see gov.vha.fw.security.UserPrincipal#getFamilyName()
	 */
	public String getFamilyName() {
		return this.familyName;
	}

	/**
	 * @param familyName
	 */
	public void setFamilyName(String familyName) {
		this.familyName = familyName;
	}

	/**
	 * @see gov.va.med.fw.security.UserPrincipal#isAnonymous()
	 */
	public boolean isAnonymous() {
		return userCredentials.isAnonymous();
	}

	/**
	 * Set the anonymous login property
	 * 
	 * @param anonymous
	 */
	public void setAnonymous(boolean anonymous) {
		userCredentials.setAnonymous(anonymous);
	}

	/**
	 * @return the ldapRoles
	 */
	public Set<String> getLdapRoles() {
		if (ldapRoles == null) {
			ldapRoles = new HashSet<String>();
		}
		return ldapRoles;
	}
	
	public Set<String> getCachedRoles() {
		return getLdapRoles();
	}

	/**
	 * @param ldapRoles
	 *            the ldapRoles to set
	 */
	public void setLdapRoles(Set<String> ldapRoles) {
		this.ldapRoles = ldapRoles;
	}

	/**
	 * default implementation
	 */
	public GrantedAuthority[] getAuthorities() {
		return authorities == null ? new GrantedAuthorityImpl[0] : authorities;
	}

	public void setAuthorities(GrantedAuthority[] authorities) {
		if (authorities == null) {
			this.authorities = new GrantedAuthorityImpl[0];
		} else {
			this.authorities = authorities;
		}
		Arrays.sort(this.authorities);
	}

	public boolean isPermissionGranted(String permission) {
		// new HashSet<GrantedAuthority>(getAuthorities());
		GrantedAuthority target = new GrantedAuthorityImpl(permission);
		GrantedAuthority[] authorities = getAuthorities();
		return (Arrays.binarySearch(authorities, target) > -1);
	}

	public boolean isRoleGranted(String permission) {
		Set permissions = new HashSet();
		permissions.add(permission);
		return isRoleGranted(permissions);
	}

	public boolean isRoleGranted(Collection permissions) {
		if (permissions != null) {
			Set roles = getLdapRoles();
			for (Iterator iter = permissions.iterator(); iter.hasNext();) {
				if (roles.contains((String) iter.next())) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * @see com.VHA.fw.model.AbstractEntity#buildToString()
	 */
	protected void buildToString(ToStringBuilder builder) {
		super.buildToString(builder);
		builder.append("name", getName());
		builder.append("anonymous", isAnonymous());
		builder.append("givenName", this.givenName);
		builder.append("middleName", this.middleName);
		builder.append("familyName", this.familyName);
		builder.append("title", this.title);
		builder.append("department", this.department);
		builder.append("description", this.description);
	}

	/**
	 * @return Returns the logicalName.
	 */
	public String getLogicalName() {
		return userCredentials.getLogicalID();
	}

	/**
	 * Logical name used for all internal anonymous logins
	 * 
	 * @param logicalName
	 *            The logicalName to set.
	 */
	public void setLogicalName(String logicalName) {
		userCredentials.setLogicalID(logicalName);
	}

	/**
	 * equals method
	 */
	public boolean equals(Object o) {
		return ((this == o) || (o instanceof UserPrincipal && this.equals((UserPrincipal) o)));
	}

	/**
	 * Identifies whether user name is same
	 * 
	 * @param userPrincipal
	 * @return
	 */
	protected boolean equals(UserPrincipal userPrincipal) {
		return (userPrincipal != null && this.getName() != null && userPrincipal.getName() != null && this
				.getName().equals(userPrincipal.getName()));
	}

	/**
	 * Returns user name
	 */
	public String getUsername() {
		return userCredentials.getUserID();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getFullName()
	 */
	public String getFullName() {

		StringBuilder name = new StringBuilder();
		if (familyName != null) {
			name.append(familyName);
			name.append(", ");
		}
		if (givenName != null) {
			name.append(givenName);
			name.append(" ");
		}

		if (middleName != null)
			name.append(middleName);
		String fullName = name.toString().trim();
		if (StringUtils.isEmpty(fullName)) {
			return getUsername();
		} else {
			return fullName;
		}
	}

	/**
	 * @return the userDn
	 */
	public String getUserDn() {
		return userDn;
	}

	/**
	 * @param userDn
	 *            the userDn to set
	 */
	public void setUserDn(String userDn) {
		this.userDn = userDn;
	}

	/**
	 * @param ldapPerson
	 *            the ldapPerson to set
	 */
	public void setLdapPerson(LdapPerson ldapPerson) {
		if (ldapPerson!=null)
		{	
			// set ldap roles, and user name details
			setUserDn(ldapPerson.getDn());
			setLdapRoles(ldapPerson.getCissGroups());
			setFamilyName(ldapPerson.getLastName());
			setGivenName(ldapPerson.getFirstName());
			setMiddleName(ldapPerson.getMiddleName());
			setTitle(ldapPerson.getTitle());
			setDepartment(ldapPerson.getDepartment());
			setDescription(ldapPerson.getDescription());
			setOffice(ldapPerson.getOffice());
			setEmail(ldapPerson.getEmail());
			setTelephoneNumber(ldapPerson.getTelephoneNumber());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getInactiveDate()
	 */
	public Date getInactiveDate() {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getInactiveMessage()
	 */
	public String getInactiveMessage() {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#isAccountLocked()
	 */
	public boolean isAccountLocked() {
		// TODO Auto-generated method stub
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#isInactive()
	 */
	public boolean isInactive() {
		// TODO Auto-generated method stub
		return false;
	}

	/**
	 * @return the currentTimeZone
	 */
	public TimeZone getCurrentTimeZone() {
		return currentTimeZone;
	}

	/**
	 * @param currentTimeZone
	 *            the currentTimeZone to set
	 */
	public void setCurrentTimeZone(TimeZone currentTimeZone) {
		this.currentTimeZone = currentTimeZone;
	}

	public String getDepartment() {
		return department;
	}

	public void setDepartment(String department) {
		this.department = department;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	public String getOffice() {
		return office;
	}

	public void setOffice(String office) {
		this.office = office;
	}

	public String getEmail() {
		return email;
	}

	public void setEmail(String email) {
		this.email = email;
	}

	public String getTelephoneNumber() {
		return telephoneNumber;
	}

	public void setTelephoneNumber(String telephoneNumber) {
		this.telephoneNumber = telephoneNumber;
	}
}