/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.hl7.builder;

// Java classes
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.jms.JMSException;

import org.apache.commons.lang.builder.ToStringBuilder;

import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.hl7.Message;
import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.service.jms.JMSMetaData;
import gov.va.med.fw.util.Reflector;
import gov.va.med.fw.util.builder.AbstractBuilder;
import gov.va.med.fw.util.builder.Builder;
import gov.va.med.fw.util.builder.BuilderException;

/**
 * @author Vu Le
 * @version 1.0
 */
public class MessageBuilder extends AbstractBuilder {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 2385097269686572241L;

	/**
	 * Comment for <code>MESSAGE_TYPE</code>
	 */
	public static final String MESSAGE_TYPE = ConfigurationConstants.DEFAULT_MESSAGE_TYPE;

	/**
	 * Comment for <code>messageTypes</code>
	 */
	private Map messages = null;

	private ArrayList builders = null;

	private Class messageClass = null;

	public MessageBuilder() {
		super();
	}

	public ArrayList getBuilders() {
		return builders;
	}

	public void setBuilders(ArrayList builders) {
		this.builders = builders;
	}

	public Class getMessageClass() {
		return messageClass;
	}

	public void setMessageClass(Class messageClass) {
		this.messageClass = messageClass;
	}

	/**
	 * @return Returns the messageTypes.
	 */
	public Map getMessages() {
		return messages;
	}

	/**
	 * @param messageTypes
	 *            The messageTypes to set.
	 */
	public void setMessages(Map messages) {
		this.messages = messages;
	}

	public Object build(JMSMetaData data) throws BuilderException {
		Object output = null;
		try {
			String text = data.getTextMessage();
			if (text != null) {
				String type = getMessageType(data.getJMSMessage());
				String msg = (String) getMessages().get(type);
				output = Reflector.instantiate(msg, new Object[] { text, type });
			}
		} catch (Exception e) {
			if (logger.isDebugEnabled()) {
				logger.debug("Input data");
				logger.debug(ToStringBuilder.reflectionToString(data));
			}
			throw new BuilderException("Failed to build an inbound message", e);
		}
		return output;
	}

	public Object build(HL7MetaData data) throws BuilderException {

		ArrayList builders = getBuilders();
		if (builders == null || builders.isEmpty()) {
			throw new BuilderException(
					"A list of segment builders must be configured prior to this call");
		}
		Class msgClass = getMessageClass();
		if (msgClass == null) {
			throw new BuilderException(
					"A message class [messageClass] must be configured prior to this call");
		}

		Message message = null;
		try {
			message = (Message) msgClass.newInstance();
			ArrayList segments = new ArrayList();
			for (int i = 0; i < builders.size(); i++) {
				Object item = builders.get(i);

				// A builder is configured with a list of concrete
				// segment builders which is also of type builder
				if (item instanceof Builder) {
					Builder builder = (Builder) item;
					Object output = builder.build(data);

					// A segment builder could return one or more segments
					if (output instanceof List) {
						segments.addAll((List) output);
					}
				}
			}
			// Completed building all segments so add all segments to a message
			message.setSegments(segments);
		} catch (Exception e) {
			if (logger.isDebugEnabled()) {
				logger.debug("Input data");
				logger.debug(ToStringBuilder.reflectionToString(data));
			}
			throw new BuilderException("Failed to build an outbound message", e);
		}
		return message;
	}

	/**
	 * Returns a concrete message type
	 * 
	 * @param msg
	 *            The message text.
	 * @return The message type.
	 */
	protected String getMessageType(javax.jms.Message msg) throws InvalidMessageException {
		String type = null;
		try {
			type = msg.getStringProperty(MESSAGE_TYPE);
		} catch (JMSException e) {
			throw new InvalidMessageException("Failed to get a message type from a message", e);
		}
		return type;
	}
}