/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.batchprocess;

import gov.va.med.fw.io.parser.WildcardFileFilter;
import gov.va.med.fw.model.batchprocess.JobConfig;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.Validate;
//import org.springframework.batch.core.BatchStatus;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.core.io.FileSystemResource;

/**
 * 
 * 
 * CISS ciss_framework Nov 19, 2008
 * 
 * @author vhaisakatikm
 */
public abstract class AbstractTasklet extends AbstractComponent implements StepExecutionListener, Tasklet {

	private ProcessStatisticsMailer processStatisticsMailer = null;
	private ProcessStatistics processStatistics = null;
	private String processName = "BatchProcess";
	private BatchProcessService batchProcessService;

	public ExitStatus afterStep(StepExecution stepExecution) {
		processStatistics.setProcessingEndDate(new Date());
		processStatistics.setExecutedOnServer(getServerName());

		ExitStatus exitStatus = stepExecution.getExitStatus();
		//BatchStatus status = stepExecution.getStatus();
		if (exitStatus != null) {
			if ("COMPLETED".equals(exitStatus.getExitCode())) {		
				// send email
				if (processStatisticsMailer != null) {
					List<String> addressList = getEmailDistributionList();
					if (addressList == null || addressList.size() == 0) {
						processStatisticsMailer.processStatistics(processStatistics);
					} else {
						processStatisticsMailer.processStatisticsWithDynamicRecipients(processStatistics,
								addressList);
					}
				}
				return createExistStatus(ExitStatus.COMPLETED, processStatistics.toFormattedString());
			}
			else if ("FAILED".equals(exitStatus.getExitCode())) {
				if (processStatisticsMailer != null) {
					List<String> addressList = getEmailDistributionList();
					if (addressList == null || addressList.size() == 0) {
						processStatisticsMailer.processFailure(processName, null,null);
					} else {
						processStatisticsMailer.processFailure(processName, null,addressList);
					}		
				}
				return createExistStatus(ExitStatus.FAILED, processStatistics.toFormattedString());
			}			
		}
		return stepExecution.getExitStatus();
	}

	public void beforeStep(StepExecution arg0) {
		processStatistics = new ProcessStatistics();
		processStatistics.setProcessName(getProcessName());
		processStatistics.setProcessingStartDate(new Date());
	}

	public ExitStatus onErrorInStep(StepExecution arg0, Throwable arg1) {
		if (processStatisticsMailer != null) {
			List<String> addressList = getEmailDistributionList();
			if (addressList == null || addressList.size() == 0) {
				processStatisticsMailer.processFailure(processName, arg1.getMessage(),null);
			} else {
				processStatisticsMailer.processFailure(processName, arg1.getMessage(),addressList);
			}		
		}
		return ExitStatus.FAILED;
	}

	protected List<String> getEmailDistributionList() {
		try {
			JobConfig jobConfig = batchProcessService.getJobConfig(processName, null);
			if (jobConfig != null && StringUtils.isNotEmpty(jobConfig.getEmailDistributionList())) {
				String email = jobConfig.getEmailDistributionList();
				String[] addresslist = email.split(",");
				// trim the spaces
				List<String> list = new ArrayList<String>();
				for (String address : addresslist) {
					list.add(address.trim());
				}
				return list;
			}
		} catch (Exception e) {
			// log and ignore
			logger.error("Unable to get Job Configuration for " + processName, e);
		}
		return null;
	}

	public String getProcessName() {
		return processName;
	}

	public void setProcessName(String processName) {
		this.processName = processName;
	}

	public ProcessStatistics getProcessStatistics() {
		if (processStatistics == null) {
			processStatistics = new ProcessStatistics();
		}
		return processStatistics;
	}

	public void setProcessStatistics(ProcessStatistics processStatistics) {
		this.processStatistics = processStatistics;
	}

	public ProcessStatisticsMailer getProcessStatisticsMailer() {
		return processStatisticsMailer;
	}

	public void setProcessStatisticsMailer(ProcessStatisticsMailer processStatisticsMailer) {
		this.processStatisticsMailer = processStatisticsMailer;
	}

	public BatchProcessService getBatchProcessService() {
		return batchProcessService;
	}

	public void setBatchProcessService(BatchProcessService batchProcessService) {
		this.batchProcessService = batchProcessService;
	}

	protected static boolean isValidFile(String path) {
		if (StringUtils.isNotEmpty(path)) {
			FileSystemResource fileSystemResource = new FileSystemResource(path);
			if (fileSystemResource.exists() && fileSystemResource.isReadable()) {
				return true;
			}
		}
		return false;
	}

	protected static boolean renameFile(String oldName, String newName) {
		if (StringUtils.isNotEmpty(oldName) && StringUtils.isNotEmpty(newName)) {
			File oldFile = new File(oldName);
			File newFile = new File(newName);
			return oldFile.renameTo(newFile);
		}
		return false;
	}

	protected static List<String> getFileNameList(String path) throws Exception {
		List<String> files = new ArrayList<String>();
		FilenameFilter inputFileFilter = null;
		String inputFileLocation = path;
		// determine the file location and the filter
		if (path.indexOf("*") != -1) {
			// Get the last index of file path separator '/' or '\'in the file
			// location
			int lastIndex = path.lastIndexOf("/");
			if (lastIndex == -1)
				lastIndex = path.lastIndexOf("\\");

			inputFileFilter = new WildcardFileFilter(path.substring(lastIndex + 1), true);
			inputFileLocation = path.substring(0, lastIndex);
		}

		FileSystemResource fileSystemResource = new FileSystemResource(inputFileLocation);
		if (fileSystemResource.getFile().isDirectory()) {
			// filter the list
			String[] fileNames = null;
			if (inputFileFilter != null) {
				fileNames = fileSystemResource.getFile().list(inputFileFilter);
			} else {
				fileNames = fileSystemResource.getFile().list();
			}
			if (fileNames != null && fileNames.length > 0) {
				for (int i = 0; i < fileNames.length; i++) {
					FileSystemResource fileResource = new FileSystemResource(fileSystemResource
							.getFile().getAbsolutePath()
							+ File.separator + fileNames[i]);
					File file = fileResource.getFile();
					if (!file.isDirectory() && file.exists()) {
						files.add(file.getAbsolutePath());
					}
				}
			}
		} else if (fileSystemResource.getFile().exists()) {
			files.add(fileSystemResource.getFile().getAbsolutePath());
		}

		return files;
	}

	protected ExitStatus createExistStatus(ExitStatus exitStatus, String msg) {
		return exitStatus.addExitDescription(msg);
	}

	/**
	 * Size limited to 1MB
	 * 
	 * @param fullPathFilename
	 * @param maxLength
	 * @return
	 * @throws Exception
	 */
	protected static String readTextFile(String fullPathFilename, long maxLength) throws Exception {
		File file = new File(fullPathFilename);
		if (file.exists() && file.canRead()) {
			if (maxLength <= 0)
				maxLength = 1024 * 1024;
			StringBuilder sb = new StringBuilder(1024);
			BufferedReader reader = new BufferedReader(new FileReader(fullPathFilename));
			char[] chars = new char[1024];
			int size = 0;
			int numRead = 0;
			while (size < maxLength && (numRead = reader.read(chars)) > -1) {
				sb.append(String.valueOf(chars));
				size += numRead;
			}
			reader.close();

			return sb.toString();
		} else {
			return null;
		}
	}
	
	private String getServerName() {
		try {
			return InetAddress.getLocalHost().getHostName();
		} catch (Exception e) {
			return "Unknown";
		}
	}
	
	@Override
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		// check for required properties
		Validate.notNull(batchProcessService, "Required BatchProcessService is not configured");
	}
}
