/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.ui.struts;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRAttachment;
import gov.va.med.ccht.model.qir.QIRForm;
import gov.va.med.ccht.model.qir.QIRRemarks;
import gov.va.med.ccht.model.qir.QIRSimpleAttachment;
import gov.va.med.ccht.model.qir.QIRType;
import gov.va.med.ccht.model.qir.QIRVendorAction;
import gov.va.med.ccht.model.report.CompletedReport;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.ccht.service.report.StandardReportService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.model.lookup.SeverityType;
import gov.va.med.fw.report.ReportExportedType;
import gov.va.med.fw.security.SecurityContext;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.TimeZoneUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.io.ClassPathResource;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.FontFactory;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Phrase;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.AcroFields;
import com.lowagie.text.pdf.ColumnText;
import com.lowagie.text.pdf.PdfCopy;
import com.lowagie.text.pdf.PdfFileSpecification;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfStamper;

public class ReportAction extends AbstractAction {

	private static final long serialVersionUID = -6987221568427816449L;

	private static final String REPORT_ID = "reportId";
	private static final String ATTACHMENT_ID = "attachmentId";
	private static final String QIR_ID = "qirId";
	private static final String CONTENT_TYPE_APPLICATION_PDF = "application/pdf";
	private static final String CONTENT_TYPE_APPLICATION_CSV = "application/csv";
	private static final String CONTENT_TYPE_APPLICATION_RTF = "application/rtf";
	private static final String CONTENT_TYPE_APPLICATION_XLS = "application/x-excel";
	private static final String QIR_FORM = "VA0729.pdf";
	private StandardReportService standardReportService;
	private QIRService qirService;
	private SecurityService securityService;

	protected transient Log logger = LogFactory.getLog(getClass());

	/**
	 * Display the selected report
	 * 
	 * @return
	 * @throws Exception
	 */
	public String viewReport() throws ServiceException,IOException {

		String reportId = getRequest().getParameter(REPORT_ID);
		CompletedReport compltReport = this.getCompletedReport(reportId);
		if (compltReport != null) {
			byte buffer[] = compltReport.getReportFileContent();
			String fileType = (compltReport.getFileType() != null) ? compltReport.getFileType()
					.getCode() : null;
			if (compltReport.getFileType() != null) {
				if (StringUtils.equals(fileType, ReportExportedType.PDF.getCode())) {
					getServletResponse().setHeader("Content-Disposition", "inline; filename=report.pdf");
					this.writeReportToReponse(buffer, CONTENT_TYPE_APPLICATION_PDF, getServletResponse());
				} else if (StringUtils.equals(fileType, ReportExportedType.CSV.getCode())) {
					getServletResponse().setHeader("Content-Disposition", "inline; filename=report.csv");
					this.writeReportToReponse(buffer, CONTENT_TYPE_APPLICATION_CSV, getServletResponse());
				} else if (StringUtils.equals(fileType, ReportExportedType.RTF.getCode())) {
					getServletResponse().setHeader("Content-Disposition", "inline; filename=report.rtf");
					this.writeReportToReponse(buffer, CONTENT_TYPE_APPLICATION_RTF, getServletResponse());
				} else if (StringUtils.equals(fileType, ReportExportedType.XLS.getCode())) {
					getServletResponse().setHeader("Content-Disposition", "inline; filename=report.xls");
					this.writeReportToReponse(buffer, CONTENT_TYPE_APPLICATION_XLS, getServletResponse());
				} else {
					this.writeMessageToReponse("Report/report unknown file type " + fileType,
							getServletResponse());
				}
			} else {
				this.writeMessageToReponse("Report/report unknown file type", getServletResponse());
			}
		} else {
			this.writeMessageToReponse("Report/report id is empty", getServletResponse());
		}
		return null;
	}

	public String viewAttachment() throws ServiceException, IOException {		
		String attachmentId = getRequest().getParameter(ATTACHMENT_ID);		
		if (StringUtils.isNotEmpty(attachmentId)){
			EntityKey<QIRAttachment> key = EntityKeyFactory.createEntityKey(new Long(attachmentId), QIRAttachment.class);
			QIRAttachment qirAttachment = getQirService().getQIRAttachment(key);
			if (qirAttachment != null) {
				String documentName = qirAttachment.getDocumentName();
				String fileType = documentName.substring(documentName.lastIndexOf(".")+1);
				getServletResponse().setHeader("Content-Disposition", "inline; filename=" + documentName);
				this.writeReportToReponse(qirAttachment.getData(), "application/"+fileType, getServletResponse());
			}
		}
		return null;
	}
	
	public String viewQIR()throws ServiceException, IOException, DocumentException {
		String qirId = getRequest().getParameter(QIR_ID);
		if( StringUtils.isNotEmpty(qirId) ) {
			EntityKey<QIR> key = EntityKeyFactory.createEntityKey(new Long(qirId), QIR.class);
			QIR qir = getQirService().getQIR(key);
			if( qir!= null ) {
				QIRType type = qir.getQirType();
				ClassPathResource resource = new ClassPathResource(QIR_FORM);
				PdfReader reader = new PdfReader(resource.getInputStream());
				ByteArrayOutputStream output = new ByteArrayOutputStream();
				PdfStamper stamper = new PdfStamper(reader, output);
				stamper.setFormFlattening(true);

				AcroFields form = stamper.getAcroFields();
				HashMap<String, String> fields = form.getFields();
				
				Set<String> keys = fields.keySet();
				
				StringBuffer actionText = new StringBuffer();
				StringBuffer complaintText = new StringBuffer();
				StringBuffer remarkText = new StringBuffer();
				
				for( String field : keys ) {
					String fvalue = form.getField(field);
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.ORIG_ROLE) ) {
						User user = getSecurityService().getUser(qir.getCreatedBy().getChangeUser());
						RegistrationReason reason = user != null ? user.getRegistrationReason() : null;
						form.setField(field, (reason != null ? reason.getName() : "None"));
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.SERIAL_NO) ) {
						form.setField(field, qir.getSerialNumber());
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.CONTRACT_NO) ) {
						form.setField(field, qir.getPurchaseOrderNumber());
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.ORIG_VISN) ) {
						form.setField(field, (qir.getVisn() != null ? qir.getVisn().getName() : ""));
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.ORIG_FACILITY) ) {
						form.setField(field, (qir.getFacility() != null ? qir.getFacility().getName() : ""));
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.ORIG_NAME) ) {
						form.setField(field, qir.getSubmittedByName());
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.MODEL_NO) ) {
						String model = qir.getDeviceType() != null ? qir.getDeviceType().getName() : "";
						form.setField(field, StringUtils.replace(model, "Version ", "V.") );
					}
					if( type != null ) {
						// Quality of Complaint
						if( StringUtils.equalsIgnoreCase(field, QIRForm.QUALITY_COMPLAINT) ) {
							if( StringUtils.equalsIgnoreCase(type.getCode(), QIRType.QUALITY_COMPLAINT) ) {
								form.setField(field, "1");
							}
						}
						// New Item
						if( StringUtils.equalsIgnoreCase(field, QIRForm.NEW_ITEM) ) {
							if( StringUtils.equalsIgnoreCase(type.getCode(), QIRType.NEW_ITEM) ) {
								form.setField(field, "1");
							}
						}
						// Similar Item
						if( StringUtils.equalsIgnoreCase(field, QIRForm.SIMILAR_ITEM) ) {
							if( StringUtils.equalsIgnoreCase(type.getCode(), QIRType.SIMILAR_ITEM) ) {
								form.setField(field, "1");
							}
						}
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.VENDOR) ) {
						form.setField(field, (qir.getVendor() != null ? qir.getVendor().getName() : ""));
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.MM_DD_YYYY) ) {
						DateFormat formatter = new SimpleDateFormat("MM/dd/yyyy");
						form.setField(field, formatter.format(qir.getCreatedOn()));
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.TN) ) {
						form.setField(field, String.valueOf(qir.getId()));
					}
					
					// Handle fields that could exceed the field limit
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.VENDOR_ACTION) ) {
						//Vendor Actions
						actionText = new StringBuffer();
						List<QIRVendorAction> actions = new ArrayList<QIRVendorAction>(qir.getVendorActions());
						Collections.sort(actions, Collections.reverseOrder());
						for (QIRVendorAction action:actions) {
							actionText.append( TimeZoneUtils.convertDateToTimeZone(action.getDate(), TimeZone.getDefault()))
  									  .append( ", " )
									  .append( action.getSubmittedByName() )
									  .append( ": " )
									  .append( action.getAction() )
									  .append( "\n" );
						}
						if( actions.isEmpty() ) {
							actionText.append("None");
						}
						// A limit of characters in a vendor action field is QIRForm.TEXT_OVERFLOW_LIMIT
						if( actionText.toString().length() <= QIRForm.TEXT_OVERFLOW_LIMIT ) {
							form.setField(field, actionText.toString());
						}
						else {
							form.setField(field, getText("field.data.overflown"));
						}
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.CAUSE_OF_COMPLAINT) ) {
						complaintText.append((qir.getComplaint() != null ? qir.getComplaint() : ""));
						if( complaintText.length() <= QIRForm.TEXT_OVERFLOW_LIMIT ) {
							form.setField(field, complaintText.toString());
						}
						else {
							form.setField(field, getText("field.data.overflown"));
						}
					}
					if( StringUtils.equalsIgnoreCase(fvalue, QIRForm.REC_REMARKS) ) {
						
						//Remarks
						remarkText = new StringBuffer();
						List<QIRRemarks> remarks = new ArrayList<QIRRemarks>(qir.getRemarks());
						Collections.sort(remarks, Collections.reverseOrder());
						for (QIRRemarks remark : remarks) {
							remarkText.append( TimeZoneUtils.convertDateToTimeZone(remark.getDate(),TimeZone.getDefault()) )
								      .append( ", " )
								      .append( remark.getSubmittedByName() )
								      .append( ": " )
								      .append( remark.getRemarks() )
								      .append( "\n" );
						}
						if( remarks.isEmpty() ) {
							remarkText.append("None");
						}
						if( remarkText.length() <= QIRForm.TEXT_OVERFLOW_LIMIT ) {
							form.setField(field, remarkText.toString());
						}
						else {
							form.setField(field, getText("field.data.overflown"));
						}
					}
				}
				
				// Add attachment appendix
				Set<QIRSimpleAttachment> attachments = qir.getAttachments();
				StringBuffer info = new StringBuffer();
				for( QIRSimpleAttachment attachment : attachments ) {
					info.append(TimeZoneUtils.convertDateToTimeZone(attachment.getCreatedOnWithTimeZone().getDate(),TimeZone.getDefault()))
						.append(", ")
						.append(attachment.getDocumentName())
						.append("\n");
				}
				if( attachments.isEmpty() ) {
					info.append("None");
				}
				addAppendix("Attachment(s)",info.toString(),stamper,reader);	

				// Add cause of complaint appendix
				if( complaintText.length() > QIRForm.TEXT_OVERFLOW_LIMIT ) {
					addAppendix("Cause of Complaint(s)",complaintText.toString(),stamper,reader);	
				}
								
				// Add Vendor Action appendix
				if( actionText.length() > QIRForm.TEXT_OVERFLOW_LIMIT ) {
					addAppendix("Vendor Action(s)",actionText.toString(),stamper,reader);	
				}
				// Add remarks appendix
				if( remarkText.length() > QIRForm.TEXT_OVERFLOW_LIMIT ) {
					addAppendix("Remark(s)",remarkText.toString(),stamper,reader);	
				}
				
				stamper.close();
				reader.close();
				output.flush();
				
		        final String formName = qir.getId() + ".pdf";
				getServletResponse().setHeader("Content-Disposition", "inline; filename="+formName );
				writeReportToReponse(addPageNumber(output,qir).toByteArray(), CONTENT_TYPE_APPLICATION_PDF, getServletResponse());
			}
		}
		return null;
	}
	private void addAppendix( String title, String text, PdfStamper stamper, PdfReader reader ) throws DocumentException {
		
		ColumnText ct = new ColumnText(null);
		ct.addElement( new Paragraph(24, text, FontFactory.getFont(FontFactory.HELVETICA,10)) );

		int i = reader.getNumberOfPages();
        while(true) {
            // Add a new page
            stamper.insertPage(++i, reader.getPageSize(1));

            // Insert a header
            getHeaderTable(title).writeSelectedRows(0, -1, 34, 780, stamper.getOverContent(i));
            
            // Add as much content of the column as possible
            ct.setCanvas(stamper.getOverContent(i));
            ct.setSimpleColumn(36, 36, 559, 760);
            if (!ColumnText.hasMoreText(ct.go()))
                break;
        }
	}
	
	private ByteArrayOutputStream addPageNumber(ByteArrayOutputStream output, QIR qir) throws IOException,DocumentException,ServiceException {
		
		// Insert page number
		PdfReader reader = new PdfReader(output.toByteArray());
		int pages = reader.getNumberOfPages();
		
		Document document = new Document();
		ByteArrayOutputStream result = new ByteArrayOutputStream();
		PdfCopy copy = new PdfCopy(document,result);
		document.open();
		
		PdfImportedPage page;
        PdfCopy.PageStamp stamp;
        for (int i = 0; i < pages; i++) {
            page = copy.getImportedPage(reader, (i+1));
            stamp = copy.createPageStamp(page);
            
            // add page numbers
            ColumnText.showTextAligned(
                    stamp.getUnderContent(), Element.ALIGN_RIGHT,
                    new Phrase(String.format("Page %d of %d", (i+1), pages),FontFactory.getFont(FontFactory.HELVETICA,8)),
                      580.5f, 28, 0);
            stamp.alterContents();
            copy.addPage(page);
        }
        
        // Insert attachments into the document
		Set<QIRSimpleAttachment> attachments = qir.getAttachments();
		for( QIRSimpleAttachment attachment : attachments ) {
			EntityKey<QIRAttachment> id = EntityKeyFactory.createEntityKey(new Long(attachment.getId()), QIRAttachment.class);
			QIRAttachment qirAttachment = getQirService().getQIRAttachment(id);
			
	        PdfFileSpecification fs
	          = PdfFileSpecification.fileEmbedded(copy, null, qirAttachment.getDocumentName(), qirAttachment.getData());
	        copy.addFileAttachment(fs);
		}
        copy.close();
        document.close();
        result.close();
        return result;
	}
	
	private PdfPTable getHeaderTable( String title ) {
        PdfPTable table = new PdfPTable(1);
        table.setTotalWidth(527);
        table.setLockedWidth(true);
        table.getDefaultCell().setFixedHeight(20);
        table.getDefaultCell().setBorder(Rectangle.BOTTOM);
        table.addCell( new Phrase("Appendix - " + title, FontFactory.getFont(FontFactory.HELVETICA,10)) );
        return table;
    }

	public Log getLogger() {
		return logger;
	}

	public void setLogger(Log logger) {
		this.logger = logger;
	}

	protected UserPrincipal getLoggedInUser() {
		SecurityContext securityContext = SecurityContextHelper.getSecurityContext();
		return (securityContext != null) ? securityContext.getUserPrincipal() : null;
	}

	protected CompletedReport getCompletedReport(String reportId) throws ServiceException {
		if (StringUtils.isNotEmpty(reportId)){
			EntityKey<CompletedReport> key = EntityKeyFactory.createEntityKey(new Long(reportId), CompletedReport.class);
			return getStandardReportService().getCompletedReport(key);
		}
		return null;
	}
	public HttpSession getHttpSession() {
		return this.getRequest().getSession();
	}

	protected void handleException(Exception exception, Object producer) {
		handleException(null, exception, producer);
	}

	protected void handleException(String message, Exception exception, Object producer) {
		handleException(message, exception, producer, SeverityType.ERROR);
	}

	protected void handleException(String message, Exception exception, Object producer,
			SeverityType severityType) {
	}
	protected void writeReportToReponse(byte buffer[], String contentType,
			HttpServletResponse response) throws IOException {
		response.setContentType(contentType);
		OutputStream outStream = response.getOutputStream();
		outStream.write(buffer);
		outStream.close();
	}

	private void writeMessageToReponse(String msg, HttpServletResponse response)
			throws IOException {
		PrintWriter writer = response.getWriter();
		writer.write(msg);
		writer.close();
	}
	public void afterPropertiesSet() throws Exception {
		// Validate.notNull(exceptionService,
		// "exceptionService must not be null");
	}

	public StandardReportService getStandardReportService() {
		return standardReportService;
	}

	public void setStandardReportService(StandardReportService standardReportService) {
		this.standardReportService = standardReportService;
	}

	public QIRService getQirService() {
		return qirService;
	}

	public void setQirService(QIRService qirService) {
		this.qirService = qirService;
	}

	/**
	 * @param securityService the securityService to set
	 */
	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}

	/**
	 * @return the securityService
	 */
	public SecurityService getSecurityService() {
		return securityService;
	}
}