/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.ui.struts;

import java.util.Properties;
import java.util.TimeZone;

import javax.security.auth.login.AccountNotFoundException;
import javax.security.auth.login.CredentialException;
import javax.security.auth.login.LoginException;

import org.apache.commons.lang.Validate;

import gov.va.med.fw.security.AccountInactiveException;
import gov.va.med.fw.security.AccountLockedException;
import gov.va.med.fw.security.InsufficientPrivilegesException;
import gov.va.med.fw.security.LoginManager;
import gov.va.med.fw.security.RegistrationDeniedException;
import gov.va.med.fw.security.RegistrationPendingException;
import gov.va.med.fw.security.SecurityContext;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserCredentials;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.TimeZoneUsageMode;
import gov.va.med.fw.util.date.TimeZoneUtils;

/**
 * Login Action logs out the user if already logged in and relogins the user
 * 
 * @author vhaisakatikm
 * 
 */
public class LoginAction extends AbstractAction {

	private static final long serialVersionUID = 5949113385413332479L;

	// Services
	private LoginManager loginManager = null;

	// Attributes
	private String username;
	private String password;
	private int autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving;
	private Boolean autoDetectedUserTimeZoneObservesDaylightSaving;
	private TimeZoneUsageMode explicitTimeZoneUsageMode;
	

	private Boolean loginSubmit;

	public String login() {

		//Validate and process login only if the user clicks login
		if(loginSubmit != null && loginSubmit){
			// Validate username and password
			if (StringUtils.isEmpty(getUsername())
					|| StringUtils.isEmpty(getPassword())) {
	
				if (StringUtils.isEmpty(getUsername())) {
					addActionError(getText(MESSAGE_USER_NAME_REQUIRED));
				}
				if (StringUtils.isEmpty(getPassword())) {
					addActionError(getText(MESSAGE_PASSWORD_REQUIRED));
				}
				return ERROR;
			}
	
			try {
				// Authenticate the user and get the user details
				processSignout(getUsername().toUpperCase());
	
				// store the user in the security context (session)
				UserCredentials userCredentials = new UserCredentials();
				userCredentials.setUserID(getUsername().toUpperCase());
				userCredentials.setPassword(getPassword());
	
				TimeZone currentTimeZone = initUserTimeZone();
				TimeZoneUtils.setThreadCurrentTimezone(currentTimeZone);
				loginManager.login(getRequest(), userCredentials, currentTimeZone);
				/*getIHTASession().setUserContext(
						SecurityContextHelper.getSecurityContext()
								.getUserPrincipal());*/
	
				return SUCCESS;
			} catch (LoginException le) {
				handleException(le, this);
				addActionError(translateException(le));
			} catch (Exception ae) {
				handleException(ae, this);
				addActionError(getText(MESSAGE_SYSTEM_ERROR));
			} finally {
				TimeZoneUtils.removeThreadCurrentTimezone();
			}
			
		}
		return this.hasErrors() ? ERROR : INPUT;
	}
	
	/**
	 * Retrieve UserProfile to compare incoming vs. onfile....if not match,
	 * prompt user
	 * 
	 * @throws LoginException
	 */
	protected TimeZone initUserTimeZone() throws LoginException {
		Integer sessionTimeZone = null;

		sessionTimeZone = autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving;

		TimeZone userTimeZone = TimeZoneUtils.getTimeZone(sessionTimeZone,
				autoDetectedUserTimeZoneObservesDaylightSaving);

		return userTimeZone;
	}

	private void clear() {
		setUsername(null);
		setPassword(null);
	}

	/**
	 * logout the current user
	 * 
	 * @return
	 * @throws Exception
	 */
	public String logout(){
		// get the currently loggedin user name and logout
		UserPrincipal user = getCurrentUser();
		if (user != null) {
			processSignout(user.getName().toUpperCase());
		}
		return SUCCESS;
	}

	public String display(){
		return SUCCESS;
	}
	
	/**
	 * @return the loginManager
	 */
	public LoginManager getLoginManager() {
		return loginManager;
	}

	/**
	 * @param loginManager
	 *            the loginManager to set
	 */
	public void setLoginManager(LoginManager loginManager) {
		this.loginManager = loginManager;
	}

	/**
	 * @return the password
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * @param password
	 *            the password to set
	 */
	public void setPassword(String password) {
		this.password = password;
	}

	/**
	 * @return the username
	 */
	public String getUsername() {
		return username;
	}

	/**
	 * @param username
	 *            the username to set
	 */
	public void setUsername(String username) {
		this.username = username;
	}

	/**
	 * Return currently loggedin user
	 * 
	 * @return
	 * @throws Exception
	 */
	public UserPrincipal getCurrentUser(){
		UserPrincipal user = null;
		SecurityContext securityContext = SecurityContextHelper
				.getSecurityContext();
		user = (securityContext != null) ? securityContext.getUserPrincipal()
				: null;
		return user;
	}

	/**
	 * processSingout for a user
	 * 
	 * @param logicalId
	 * @throws Exception
	 */
	private void processSignout(String logicalId) {
		// Clean application data in a session and invalidate it
		setApplicationData(null);

		// Logout using login manager.
		if (logicalId != null)
			getLoginManager().loginAnonymous(logicalId);
		else
			getLoginManager().loginAnonymous();

		// Invalidate the session
		//super.getIHTASession().invalidate();
	}

	private String translateException(LoginException loginException) {
		Properties props = new Properties();
		props.setProperty(AccountNotFoundException.class.getName(),
				getText(MESSAGE_ACCOUNT_NOTFOUND));
		props.setProperty(AccountLockedException.class.getName(),
				getText(MESSAGE_LOGON_ACCOUNT_LOCKED));
		props.setProperty(AccountInactiveException.class.getName(),
				getText(MESSAGE_LOGON_ACCOUNT_INACTIVE));
		props.setProperty(CredentialException.class.getName(),
				getText(MESSAGE_LOGON_BAD_CREDENDIALS));
		props.setProperty(InsufficientPrivilegesException.class.getName(),
				getText(MESSAGE_INSUFFICIENT_PRIVILEGES));
		props.setProperty(javax.security.auth.login.AccountLockedException.class.getName(), 
				getText(MESSAGE_LOGON_ACCOUNT_LOCKED));
		props.setProperty(RegistrationDeniedException.class.getName(), 
				getText(MESSAGE_REGISTRATION_DENIED));
		props.setProperty(RegistrationPendingException.class.getName(), 
				getText(MESSAGE_REGISTRATION_PENDING));		
		
		String message = (String) props.getProperty(loginException.getClass()
				.getName());
		// Return the error message by default
		return message == null ? loginException.getMessage() : message;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.IHTA.ui.struts.AbstractAction#afterPropertiesSet()
	 */
	@Override
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();

		Validate.notNull(loginManager, "loginManager can not be null");
	}

	// test code
	public String test(){
		return SUCCESS;
	}

	public static final String MESSAGE_USER_NAME_REQUIRED = "username.required";
	public static final String MESSAGE_PASSWORD_REQUIRED = "password.required";
	public static final String MESSAGE_ACCOUNT_NOTFOUND = "account.notfound";
	public static final String MESSAGE_LOGON_ACCOUNT_LOCKED = "account.locked";
	public static final String MESSAGE_LOGON_ACCOUNT_INACTIVE = "account.inactive";
	public static final String MESSAGE_LOGON_BAD_CREDENDIALS = "bad.credentials";
	public static final String MESSAGE_INSUFFICIENT_PRIVILEGES = "insufficient.privileges";
	public static final String MESSAGE_REGISTRATION_DENIED = "registration.denied";
	public static final String MESSAGE_REGISTRATION_PENDING = "registration.pending";

	/**
	 * @return the autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving
	 */
	public int getAutoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving() {
		return autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving;
	}

	/**
	 * @param autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving
	 *            the autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving to
	 *            set
	 */
	public void setAutoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving(
			int autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving) {
		this.autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving = autoDetectedUserTimeZoneRawOffsetWithoutDaylightSaving;
	}

	/**
	 * @return the explicitTimeZoneUsageMode
	 */
	public TimeZoneUsageMode getExplicitTimeZoneUsageMode() {
		return explicitTimeZoneUsageMode;
	}

	/**
	 * @param explicitTimeZoneUsageMode
	 *            the explicitTimeZoneUsageMode to set
	 */
	public void setExplicitTimeZoneUsageMode(
			TimeZoneUsageMode explicitTimeZoneUsageMode) {
		this.explicitTimeZoneUsageMode = explicitTimeZoneUsageMode;
	}

	/**
	 * @return the autoDetectedUserTimeZoneObservesDaylightSaving
	 */
	public Boolean getAutoDetectedUserTimeZoneObservesDaylightSaving() {
		return autoDetectedUserTimeZoneObservesDaylightSaving;
	}

	/**
	 * @param autoDetectedUserTimeZoneObservesDaylightSaving
	 *            the autoDetectedUserTimeZoneObservesDaylightSaving to set
	 */
	public void setAutoDetectedUserTimeZoneObservesDaylightSaving(
			Boolean autoDetectedUserTimeZoneObservesDaylightSaving) {
		this.autoDetectedUserTimeZoneObservesDaylightSaving = autoDetectedUserTimeZoneObservesDaylightSaving;
	}
	/**
	 * @return loginSubmit.
	 */
	public Boolean getLoginSubmit() {
		return loginSubmit;
	}

	/**
	 * @param loginSubmit
	 */
	public void setLoginSubmit(Boolean loginSubmit) {
		this.loginSubmit = loginSubmit;
	}
}
